# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['statemachine', 'statemachine.contrib']

package_data = \
{'': ['*'],
 'statemachine': ['locale/en/LC_MESSAGES/*', 'locale/pt_BR/LC_MESSAGES/*']}

setup_kwargs = {
    'name': 'python-statemachine',
    'version': '2.1.0',
    'description': 'Python Finite State Machines made easy.',
    'long_description': '# Python StateMachine\n\n[![pypi](https://img.shields.io/pypi/v/python-statemachine.svg)](https://pypi.python.org/pypi/python-statemachine)\n[![downloads](https://img.shields.io/pypi/dm/python-statemachine.svg)](https://pypi.python.org/pypi/python-statemachine)\n[![build status](https://github.com/fgmacedo/python-statemachine/actions/workflows/python-package.yml/badge.svg?branch=develop)](https://github.com/fgmacedo/python-statemachine/actions/workflows/python-package.yml?query=branch%3Adevelop)\n[![Coverage report](https://codecov.io/gh/fgmacedo/python-statemachine/branch/develop/graph/badge.svg)](https://codecov.io/gh/fgmacedo/python-statemachine)\n[![Documentation Status](https://readthedocs.org/projects/python-statemachine/badge/?version=latest)](https://python-statemachine.readthedocs.io/en/latest/?badge=latest)\n[![GitHub commits since last release (main)](https://img.shields.io/github/commits-since/fgmacedo/python-statemachine/main/develop)](https://github.com/fgmacedo/python-statemachine/compare/main...develop)\n\n\nPython [finite-state machines](https://en.wikipedia.org/wiki/Finite-state_machine) made easy.\n\n\n* Free software: MIT license\n* Documentation: https://python-statemachine.readthedocs.io.\n\n\nWelcome to python-statemachine, an intuitive and powerful state machine framework designed for a\ngreat developer experience.\n\n🚀 With StateMachine, you can easily create complex, dynamic systems with clean, readable code.\n\n💡 Our framework makes it easy to understand and reason about the different states, events and\ntransitions in your system, so you can focus on building great products.\n\n🔒 python-statemachine also provides robust error handling and ensures that your system stays\nin a valid state at all times.\n\n\nA few reasons why you may consider using it:\n\n* 📈 python-statemachine is designed to help you build scalable,\n  maintainable systems that can handle any complexity.\n* 💪 You can easily create and manage multiple state machines within a single application.\n* 🚫 Prevents common mistakes and ensures that your system stays in a valid state at all times.\n\n\n## Getting started\n\n\nTo install Python State Machine, run this command in your terminal:\n\n    pip install python-statemachine\n\nTo generate diagrams from your machines, you\'ll also need `pydot` and `Graphviz`. You can\ninstall this library already with `pydot` dependency using the `extras` install option. See\nour docs for more details.\n\n    pip install python-statemachine[diagrams]\n\nDefine your state machine:\n\n```py\n>>> from statemachine import StateMachine, State\n\n>>> class TrafficLightMachine(StateMachine):\n...     "A traffic light machine"\n...     green = State(initial=True)\n...     yellow = State()\n...     red = State()\n...\n...     cycle = (\n...         green.to(yellow)\n...         | yellow.to(red)\n...         | red.to(green)\n...     )\n...\n...     def before_cycle(self, event: str, source: State, target: State, message: str = ""):\n...         message = ". " + message if message else ""\n...         return f"Running {event} from {source.id} to {target.id}{message}"\n...\n...     def on_enter_red(self):\n...         print("Don\'t move.")\n...\n...     def on_exit_red(self):\n...         print("Go ahead!")\n\n```\n\nYou can now create an instance:\n\n```py\n>>> sm = TrafficLightMachine()\n\n```\n\nThis state machine can be represented graphically as follows:\n\n```py\n>>> img_path = "docs/images/readme_trafficlightmachine.png"\n>>> sm._graph().write_png(img_path)\n\n```\n\n![](https://raw.githubusercontent.com/fgmacedo/python-statemachine/develop/docs/images/readme_trafficlightmachine.png)\n\n\nWhere on the `TrafficLightMachine`, we\'ve defined `green`, `yellow`, and `red` as states, and\none event called `cycle`, which is bound to the transitions from `green` to `yellow`, `yellow` to `red`,\nand `red` to `green`. We also have defined three callbacks by name convention, `before_cycle`, `on_enter_red`, and `on_exit_red`.\n\n\nThen start sending events to your new state machine:\n\n```py\n>>> sm.send("cycle")\n\'Running cycle from green to yellow\'\n\n```\n\nThat\'s it. This is all an external object needs to know about your state machine: How to send events.\nIdeally, all states, transitions, and actions should be kept internally and not checked externally to avoid unnecessary coupling.\n\nBut if your use case needs, you can inspect state machine properties, like the current state:\n\n```py\n>>> sm.current_state.id\n\'yellow\'\n\n```\n\nOr get a complete state representation for debugging purposes:\n\n```py\n>>> sm.current_state\nState(\'Yellow\', id=\'yellow\', value=\'yellow\', initial=False, final=False)\n\n```\n\nThe `State` instance can also be checked by equality:\n\n```py\n>>> sm.current_state == TrafficLightMachine.yellow\nTrue\n\n>>> sm.current_state == sm.yellow\nTrue\n\n```\n\nOr you can check if a state is active at any time:\n\n```py\n>>> sm.green.is_active\nFalse\n\n>>> sm.yellow.is_active\nTrue\n\n>>> sm.red.is_active\nFalse\n\n```\n\nEasily iterate over all states:\n\n```py\n>>> [s.id for s in sm.states]\n[\'green\', \'red\', \'yellow\']\n\n```\n\nOr over events:\n\n```py\n>>> [t.name for t in sm.events]\n[\'cycle\']\n\n```\n\nCall an event by its name:\n\n```py\n>>> sm.cycle()\nDon\'t move.\n\'Running cycle from yellow to red\'\n\n```\nOr send an event with the event name:\n\n```py\n>>> sm.send(\'cycle\')\nGo ahead!\n\'Running cycle from red to green\'\n\n>>> sm.green.is_active\nTrue\n\n```\n\nYou can pass arbitrary positional or keyword arguments to the event, and\nthey will be propagated to all actions and callbacks using something similar to dependency injection. In other words, the library will only inject the parameters declared on the\ncallback method.\n\nNote how `before_cycle` was declared:\n\n```py\ndef before_cycle(self, event: str, source: State, target: State, message: str = ""):\n    message = ". " + message if message else ""\n    return f"Running {event} from {source.id} to {target.id}{message}"\n```\n\nThe params `event`, `source`, `target` (and others) are available built-in to be used on any action.\nThe param `message` is user-defined, in our example we made it default empty so we can call `cycle` with\nor without a `message` parameter.\n\nIf we pass a `message` parameter, it will be used on the `before_cycle` action:\n\n```py\n>>> sm.send("cycle", message="Please, now slowdown.")\n\'Running cycle from green to yellow. Please, now slowdown.\'\n\n```\n\n\nBy default, events with transitions that cannot run from the current state or unknown events\nraise a `TransitionNotAllowed` exception:\n\n```py\n>>> sm.send("go")\nTraceback (most recent call last):\nstatemachine.exceptions.TransitionNotAllowed: Can\'t go when in Yellow.\n\n```\n\nKeeping the same state as expected:\n\n```py\n>>> sm.yellow.is_active\nTrue\n\n```\n\nA human-readable name is automatically derived from the `State.id`, which is used on the messages\nand in diagrams:\n\n```py\n>>> sm.current_state.name\n\'Yellow\'\n\n```\n\n## A more useful example\n\nA simple didactic state machine for controlling an `Order`:\n\n```py\n>>> class OrderControl(StateMachine):\n...     waiting_for_payment = State(initial=True)\n...     processing = State()\n...     shipping = State()\n...     completed = State(final=True)\n...\n...     add_to_order = waiting_for_payment.to(waiting_for_payment)\n...     receive_payment = (\n...         waiting_for_payment.to(processing, cond="payments_enough")\n...         | waiting_for_payment.to(waiting_for_payment, unless="payments_enough")\n...     )\n...     process_order = processing.to(shipping, cond="payment_received")\n...     ship_order = shipping.to(completed)\n...\n...     def __init__(self):\n...         self.order_total = 0\n...         self.payments = []\n...         self.payment_received = False\n...         super(OrderControl, self).__init__()\n...\n...     def payments_enough(self, amount):\n...         return sum(self.payments) + amount >= self.order_total\n...\n...     def before_add_to_order(self, amount):\n...         self.order_total += amount\n...         return self.order_total\n...\n...     def before_receive_payment(self, amount):\n...         self.payments.append(amount)\n...         return self.payments\n...\n...     def after_receive_payment(self):\n...         self.payment_received = True\n...\n...     def on_enter_waiting_for_payment(self):\n...         self.payment_received = False\n\n```\n\nYou can use this machine as follows.\n\n```py\n>>> control = OrderControl()\n\n>>> control.add_to_order(3)\n3\n\n>>> control.add_to_order(7)\n10\n\n>>> control.receive_payment(4)\n[4]\n\n>>> control.current_state.id\n\'waiting_for_payment\'\n\n>>> control.current_state.name\n\'Waiting for payment\'\n\n>>> control.process_order()\nTraceback (most recent call last):\n...\nstatemachine.exceptions.TransitionNotAllowed: Can\'t process_order when in Waiting for payment.\n\n>>> control.receive_payment(6)\n[4, 6]\n\n>>> control.current_state.id\n\'processing\'\n\n>>> control.process_order()\n\n>>> control.ship_order()\n\n>>> control.payment_received\nTrue\n\n>>> control.order_total\n10\n\n>>> control.payments\n[4, 6]\n\n>>> control.completed.is_active\nTrue\n\n```\n\nThere\'s a lot more to cover, please take a look at our docs:\nhttps://python-statemachine.readthedocs.io.\n\n\n## Contributing to the project\n\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine" data-icon="octicon-star" aria-label="Star fgmacedo/python-statemachine on GitHub">Star this project</a>\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine/issues" data-icon="octicon-issue-opened" aria-label="Issue fgmacedo/python-statemachine on GitHub">Open an Issue</a>\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine/fork" data-icon="octicon-repo-forked" aria-label="Fork fgmacedo/python-statemachine on GitHub">Fork</a>\n\n- If you found this project helpful, please consider giving it a star on GitHub.\n\n- **Contribute code**: If you would like to contribute code to this project, please submit a pull\nrequest. For more information on how to contribute, please see our [contributing.md]contributing.md) file.\n\n- **Report bugs**: If you find any bugs in this project, please report them by opening an issue\n  on our GitHub issue tracker.\n\n- **Suggest features**: If you have a great idea for a new feature, please let us know by opening\n  an issue on our GitHub issue tracker.\n\n- **Documentation**: Help improve this project\'s documentation by submitting pull requests.\n\n- **Promote the project**: Help spread the word about this project by sharing it on social media,\n  writing a blog post, or giving a talk about it. Tag me on Twitter\n  [@fgmacedo](https://twitter.com/fgmacedo) so I can share it too!\n',
    'author': 'Fernando Macedo',
    'author_email': 'fgmacedo@gmail.com',
    'maintainer': 'Fernando Macedo',
    'maintainer_email': 'fgmacedo@gmail.com',
    'url': 'https://github.com/fgmacedo/python-statemachine',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<3.12',
}


setup(**setup_kwargs)
