# StateMachine 2.4.0

*November 5, 2024*

## What's new in 2.4.0

This release introduces powerful new features for the `StateMachine` library: {ref}`Condition expressions` and explicit definition of {ref}`Events`. These updates make it easier to define complex transition conditions and enhance performance, especially in workflows with nested or recursive event structures.

### Python compatibility in 2.4.0

StateMachine 2.4.0 supports Python 3.7, 3.8, 3.9, 3.10, 3.11, 3.12, and 3.13.

### Conditions expressions in 2.4.0

This release introduces support for conditionals with Boolean algebra. You can now use expressions like `or`, `and`, and `not` directly within transition conditions, simplifying the definition of complex state transitions. This allows for more flexible and readable condition setups in your state machine configurations.

Example (with a spoiler of the next highlight):

```py
>>> from statemachine import StateMachine, State, Event

>>> class AnyConditionSM(StateMachine):
...     start = State(initial=True)
...     end = State(final=True)
...
...     submit = Event(
...         start.to(end, cond="used_money or used_credit"),
...         name="finish order",
...     )
...
...     used_money: bool = False
...     used_credit: bool = False

>>> sm = AnyConditionSM()
>>> sm.submit()
Traceback (most recent call last):
TransitionNotAllowed: Can't finish order when in Start.

>>> sm.used_credit = True
>>> sm.submit()
>>> sm.current_state.id
'end'

```

```{seealso}
See {ref}`Condition expressions` for more details or take a look at the {ref}`sphx_glr_auto_examples_lor_machine.py` example.
```

### Explicit event creation in 2.4.0

Now you can explicit declare {ref}`Events` using the {ref}`event` class. This allows custom naming, translations, and also helps your IDE to know that events are callable.

```py
>>> from statemachine import StateMachine, State, Event

>>> class StartMachine(StateMachine):
...     created = State(initial=True)
...     started = State(final=True)
...
...     start = Event(created.to(started), name="Launch the machine")
...
>>> [e.id for e in StartMachine.events]
['start']
>>> [e.name for e in StartMachine.events]
['Launch the machine']
>>> StartMachine.start.name
'Launch the machine'

```

```{seealso}
See {ref}`Events` for more details.
```

### Recursive state machines (infinite loop)

We removed a note from the docs saying to avoid recursion loops. Since the {ref}`StateMachine 2.0.0` release we've turned the RTC model enabled by default, allowing nested events to occour as all events are put on an internal queue before being executed.

```{seealso}
See {ref}`sphx_glr_auto_examples_recursive_event_machine.py` for an example of an infinite loop state machine declaration using `after` action callback to call the same event over and over again.

```


## Bugfixes in 2.4.0

- Fixes [#484](https://github.com/fgmacedo/python-statemachine/issues/484) issue where nested events inside loops could leak memory by incorrectly
  referencing previous `event_data` when queuing the next event. This fix improves performance and stability in event-heavy workflows.
