from enum import Enum
from math import ceil, floor
from typing import Sequence


BASE_BORDER = "*"


class Justification(Enum):
    """An enum used for justifying text in the table"""

    LEFT = 0
    CENTRE = 1
    RIGHT = 2


class _Cell:
    """The basic building block of the table that stores the text and the justification of itself"""

    def __init__(self, text: str, justification: Justification | None = None) -> None:
        self.justification: Justification | None = justification
        self.text = text

    def _print_as_left_justified(self, max_length: int, border_character: str) -> str:
        """Prints out the cell left-justified

        Args:
            max_length (int): The max length of a cell in this column
            border_character (str): The character to be used on the border at the sides

        Returns:
            str: The cell as a string with left justification
        """
        return (
            " "
            + self.text
            + (" " * (max_length - len(self.text) + 1))
            + border_character
        )

    def _print_as_centre_justified(self, max_length: int, border_character: str) -> str:
        """Prints out the cell centre-justified

        Args:
            max_length (int): The max length of a cell in this column
            border_character (str): The character to be used on the border at the sides

        Returns:
            str: The cell as a string with centre justification
        """
        return (
            " " * (floor((max_length - len(self.text)) / 2) + 1)
            + self.text
            + " " * (ceil((max_length - len(self.text)) / 2) + 1)
            + border_character
        )

    def _print_as_right_justified(self, max_length: int, border_character: str) -> str:
        """Prints out the cell right-justified

        Args:
            max_length (int): The max length of a cell in this column
            border_character (str): The character to be used on the border at the sides

        Returns:
            str: The cell as a string with right justification
        """
        return (
            " " * (max_length - len(self.text) + 1) + self.text + " " + border_character
        )

    def get_cell_as_string(self, max_length: int, border_character: str) -> str:
        """Returns the cell in string format with the correct formatting.
        Note that a justification of None will default to a left justification.

        Args:
            max_length (int): The max length of a cell in that column.
            border_character (str): The character to be used on the border at the sides

        Raises:
            Exception: If there is an incorrect/unsupported justification

        Returns:
            str: The cell as a string with the correct justicication
        """
        # Edge case if there is nothing for this column
        if max_length == 0:
            return " " + border_character

        match self.justification:
            case None:  # Default to left-justified
                return self._print_as_left_justified(max_length, border_character)

            case Justification.LEFT:
                return self._print_as_left_justified(max_length, border_character)

            case Justification.CENTRE:
                return self._print_as_centre_justified(max_length, border_character)

            case Justification.RIGHT:
                return self._print_as_right_justified(max_length, border_character)

            # TODO: Raise an "Unknown Justification" exception
            case _:
                raise Exception()


class _Column:
    """An abstraction of the column of the table, which holds information about the column"""

    def __init__(self) -> None:
        self.max_length = 0


class _Row:
    """An abstraction of the row of the table, which holds each of the cells in that row of the table"""

    def __init__(self, *row: str, justification: Justification | None = None) -> None:
        self.cells: dict[int, _Cell] = {}

        for i in range(0, len(row)):
            self.cells[i] = _Cell(row[i], justification)

    def set_justification_for_cells(self, justification: Justification | None) -> None:
        """Sets the justification for all of the cells in the row

        Args:
            justification (Justification | None): The justification to set the cells to. If `None` is passed in, it sets the justificaiton to the default,
            which currently is the right justification.
        """
        for cell in self.cells.values():
            cell.justification = justification

    def get_row_as_string(
        self, border_character: str, max_lengths: Sequence[int]
    ) -> str:
        """Returns the row as a string

        Args:
            border_character (str): The border character that is being used for the table
            max_lengths (Sequence[int]): The max lengths for all of the columns

        Returns:
            str: _description_
        """
        final_row = border_character

        for i in range(0, len(max_lengths)):
            if i < len(self.cells):
                final_row += self.cells[i].get_cell_as_string(
                    max_lengths[i], border_character
                )

            else:
                final_row += " " * (max_lengths[i] + 2) + border_character

        return final_row + "\n"


class PrintTable:
    """The table itself. More documentation to come!"""

    def __init__(self) -> None:
        self.has_header_row: bool = True

        # TODO: Is a dictionary whose keys are ints essentially a list in Python?
        self._columns: dict[int, _Column] = {}
        self._rows: dict[int, _Row] = {}
        self._border_character = BASE_BORDER
        self._title: str | None = None
        self._title_justification: Justification = Justification.CENTRE

    def _check_and_increase_max_column_length(self, column_i: int, length: int) -> None:
        """Takes in the column number and the length of the new string at that column and changes the length to the given length if longer, or if that column doesn't yet exist.

        Args:
            column (int): The column at which this is ocurring
            length (int): The length of the new string
        """
        # If there is no column for this yet, add one
        if column_i >= len(self._columns):
            self._columns[column_i] = _Column()
            self._columns[column_i].max_length = length
            return

        # Check if this length is longer than the currently stored one. If so, update
        if length > self._columns[column_i].max_length:
            self._columns[column_i].max_length = length

    def _total_border_length(self) -> int:
        """Figures out the length of the table, including the two border characters themselves

        Returns:
            int: The length of the table
        """
        total_border_length = 0

        for col_len in self._columns.values():
            total_border_length += (
                col_len.max_length + 3 if not col_len.max_length == 0 else 2
            )

        return total_border_length + 1

    def _get_border_row(self) -> str:
        """Creates the 'border row', or the top and botom row of the table made up of only border characters

        Returns:
            str: The border row made up of only border characters
        """
        return BASE_BORDER * self._total_border_length() + "\n"

    def _get_header(self, row: _Row, border_character: str) -> str:
        """Creates the 'header rows', which is the row of text sandwiched between two border rows

        Args:
            row (_Row): The row of text to sandwich between the border rows
            border_character (str): The border character

        Returns:
            str: The row of text sandwiched between two border rows
        """
        return (
            row.get_row_as_string(
                border_character,
                [column.max_length for column in self._columns.values()],
            )
            + self._get_border_row()
        )

    def _get_title_row(self, border_character: str) -> str:
        """Creates the title row for the table if the title is set. Otherwise returns just the top border row

        NOTE: For now, we will cut off the title if it's too long. This will be fixed in GitHub Issue #61
        NOTE: For now, we only support centre-justifying the title. This will be addressed in GitHub Issue #62

        Args:
            border_character (str): The border character

        Returns:
            str: The title of the table, or an empty string if the title is not set
        """

        if not self._title:
            return self._get_border_row()

        # We want to cut off the title (for now) to not go passed the length of the table
        title = self._title[: self._total_border_length() - 4]

        length_without_borders = self._total_border_length() - 4

        match self._title_justification:
            case Justification.LEFT:
                title = (
                    self._border_character
                    + " "
                    + title
                    + (" " * (length_without_borders - len(title) + 1))
                    + border_character
                    + "\n"
                )

            case Justification.CENTRE:
                title = (
                    self._border_character
                    + " " * (floor((length_without_borders - len(title)) / 2) + 1)
                    + title
                    + " " * (ceil((length_without_borders - len(title)) / 2) + 1)
                    + self._border_character
                    + "\n"
                )

            case Justification.RIGHT:
                title = (
                    self._border_character
                    + " " * (length_without_borders - len(title) + 1)
                    + title
                    + " "
                    + self._border_character
                    + "\n"
                )

            case _:
                # TODO: Raise "Unsupported Justification" Exception
                raise Exception()

        return self._get_border_row() + title + self._get_border_row()

    def add_row(self, *row: str) -> None:
        """Adds another row to the bottom of the table

        Args:
            row (*str): The next row of text to be added to the table
        """
        for col_i in range(0, len(row)):
            self._check_and_increase_max_column_length(col_i, len(row[col_i]))

        self._rows[len(self._rows)] = _Row(*row)

    def set_table_justification(self, justification: Justification) -> None:
        """Sets the justification for the whole table. Overrides any previous justification that was set

        Args:
            justification (Justification): The justification for the table
        """
        for row in self._rows.values():
            row.set_justification_for_cells(justification)

    def set_row_justification(self, row_i: int, justification: Justification) -> None:
        """Sets the justification for the row in the table. Overrides any previous justification for that row that was set.

        Args:
            row_i (int): The row to justify.
            justification (Justification): The justification for the row
        """
        self._rows[row_i].set_justification_for_cells(justification)

    def set_column_justification(
        self, col_i: int, justification: Justification
    ) -> None:
        """Sets the justificatoin for the column in the table. Overrides any previous justification for that row that was set.

        Args:
            col_i (int): The index of the column to change
            justification (Justification): The justification to set for the cells in that column
        """
        for row_i in range(0, len(self._rows)):
            self.set_cell_justification(row_i, col_i, justification)

    def set_cell_justification(
        self, row_i: int, column_i: int, justificaiton: Justification
    ) -> None:
        """Sets the justification for a specific cell in the table. Overrides any previous justification for that cell that was set.

        Args:
            row_i (int): The row that the cell is in.
            column_i (int): The column that the cell is in.
            justificaiton (Justification): The justification for the cell.
        """
        self._rows[row_i].cells[column_i].justification = justificaiton

    def get_table(self) -> str:
        """Returns the current table as a string

        Raises:
            Exception: If there is no data, throws an exception. TODO: Create the actual exception for this.

        Returns:
            str: The currenct table
        """
        if len(self._rows) == 0:
            # TODO: Throw a "table has no length" exception
            raise Exception

        table = self._get_title_row(self._border_character)

        if self.has_header_row:
            table += self._get_header(self._rows[0], self._border_character)

        for i in range(1 if self.has_header_row else 0, len(self._rows)):
            table += self._rows[i].get_row_as_string(
                self._border_character,
                [column.max_length for column in self._columns.values()],
            )

        return table + self._get_border_row()

    def clear_title(self) -> None:
        """Clears the title, by setting it to None"""
        self._title = None
        self._title_justification = Justification.CENTRE

    def set_title(
        self, title: str, title_justification: Justification = Justification.CENTRE
    ) -> None:
        """Sets the title and the title's justification.

        Args:
            title (str): The new title of the table
            justification (Justification): The justification for the title of the table. If no justification is passed in, defaults to Justification.CENTRE
        """
        self._title = title
        self._title_justification = title_justification

    def set_title_justification(self, title_justification: Justification) -> None:
        """Sets the title's justification

        Args:
            title_justification (Justification): The justification for the title
        """
        self._title_justification = title_justification
