"""This module contains vulnerability types and helpers.

It is only used in vulnerabilities.py
"""
from enum import Enum


class VulnerabilityType(Enum):
    FALSE = 0
    SANITISED = 1
    TRUE = 2
    UNKNOWN = 3


def vuln_factory(vulnerability_type):
    if vulnerability_type == VulnerabilityType.UNKNOWN:
        return UnknownVulnerability
    elif vulnerability_type == VulnerabilityType.SANITISED:
        return SanitisedVulnerability
    else:
        return Vulnerability


def _get_reassignment_str(reassignment_nodes):
    reassignments = ''
    if reassignment_nodes:
        reassignments += '\nReassigned in:\n\t'
        reassignments += '\n\t'.join([
            'File: ' + node.path + '\n' +
            '\t > Line ' + str(node.line_number) + ': ' + node.label
            for node in reassignment_nodes
        ])
    return reassignments


class Vulnerability():
    def __init__(
        self,
        source,
        source_trigger_word,
        sink,
        sink_trigger_word,
        reassignment_nodes
    ):
        """Set source and sink information."""
        self.source = source
        self.source_trigger_word = source_trigger_word
        self.sink = sink
        self.sink_trigger_word = sink_trigger_word

        self.reassignment_nodes = reassignment_nodes
        self._remove_sink_from_secondary_nodes()

    def _remove_sink_from_secondary_nodes(self):
        try:
            self.reassignment_nodes.remove(self.sink)
        except ValueError:  # pragma: no cover
            pass

    def __str__(self):
        """Pretty printing of a vulnerability."""
        reassigned_str = _get_reassignment_str(self.reassignment_nodes)
        return (
            'File: {}\n'
            ' > User input at line {}, trigger word "{}":\n'
            '\t {}{}\nFile: {}\n'
            ' > reaches line {}, trigger word "{}":\n'
            '\t{}'.format(
                self.source.path,
                self.source.line_number, self.source_trigger_word,
                self.source.label, reassigned_str, self.sink.path,
                self.sink.line_number, self.sink_trigger_word,
                self.sink.label
            )
        )

    def as_dict(self):
        return {
            'source': self.source.as_dict(),
            'source_trigger_word': self.source_trigger_word,
            'sink': self.sink.as_dict(),
            'sink_trigger_word': self.sink_trigger_word,
            'type': self.__class__.__name__,
            'reassignment_nodes': [node.as_dict() for node in self.reassignment_nodes]
        }


class SanitisedVulnerability(Vulnerability):
    def __init__(
        self,
        confident,
        sanitiser,
        **kwargs
    ):
        super().__init__(**kwargs)
        self.confident = confident
        self.sanitiser = sanitiser

    def __str__(self):
        """Pretty printing of a vulnerability."""
        return (
            super().__str__() +
            '\nThis vulnerability is ' +
            ('' if self.confident else 'potentially ') +
            'sanitised by: ' +
            str(self.sanitiser)
        )

    def as_dict(self):
        output = super().as_dict()
        output['sanitiser'] = self.sanitiser.as_dict()
        output['confident'] = self.confident
        return output


class UnknownVulnerability(Vulnerability):
    def __init__(
        self,
        unknown_assignment,
        **kwargs
    ):
        super().__init__(**kwargs)
        self.unknown_assignment = unknown_assignment

    def as_dict(self):
        output = super().as_dict()
        output['unknown_assignment'] = self.unknown_assignment.as_dict()
        return output

    def __str__(self):
        """Pretty printing of a vulnerability."""
        return (
            super().__str__() +
            '\nThis vulnerability is unknown due to: ' +
            str(self.unknown_assignment)
        )
