#!/usr/bin/env python
#
# A library that provides a Python interface to the Telegram Bot API
# Copyright (C) 2015-2018
# Leandro Toledo de Souza <devs@python-telegram-bot.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser Public License for more details.
#
# You should have received a copy of the GNU Lesser Public License
# along with this program.  If not, see [http://www.gnu.org/licenses/].
"""This module contains the CallbackContext class."""

from telegram import Update


class CallbackContext(object):
    """
    This is a context object passed to the callback called by :class:`telegram.ext.Handler`
    or by the :class:`telegram.ext.Dispatcher` in an error handler added by
    :attr:`telegram.ext.Dispatcher.add_error_handler` or to the callback of a
    :class:`telegram.ext.Job`.

    Note:
        :class:`telegram.ext.Dispatcher` will create a single context for an entire update. This
        means that if you got 2 handlers in different groups and they both get called, they will
        get passed the same `CallbackContext` object (of course with proper attributes like
        `.matches` differing). This allows you to add custom attributes in a lower handler group
        callback, and then subsequently access those attributes in a higher handler group callback.
        Note that the attributes on `CallbackContext` might change in the future, so make sure to
        use a fairly unique name for the attributes.

    Warning:
         Do not combine custom attributes and @run_async. Due to how @run_async works, it will
         almost certainly execute the callbacks for an update out of order, and the attributes
         that you think you added will not be present.

    Attributes:
        chat_data (:obj:`dict`, optional): A dict that can be used to keep any data in. For each
            update from the same chat it will be the same ``dict``.
        user_data (:obj:`dict`, optional): A dict that can be used to keep any data in. For each
            update from the same user it will be the same ``dict``.
        matches (List[:obj:`re match object`], optional): If the associated update originated from
            a regex-supported handler or had a :class:`Filters.regex`, this will contain a list of
            match objects for every pattern where ``re.search(pattern, string)`` returned a match.
        args (List[:obj:`str`], optional): Arguments passed to a command if the associated update
            is handled by :class:`telegram.ext.CommandHandler`, :class:`telegram.ext.PrefixHandler`
            or :class:`telegram.ext.StringCommandHandler`. It contains a list of the words in the
            text after the command, using any whitespace string as a delimiter.
        error (:class:`telegram.TelegramError`, optional): The Telegram error that was raised.
            Only present when passed to a error handler registered with
            :attr:`telegram.ext.Dispatcher.add_error_handler`.
        job (:class:`telegram.ext.Job`): The job that that originated this callback.
            Only present when passed to the callback of :class:`telegram.ext.Job`.

    """

    def __init__(self, dispatcher):
        """
        Args:
            dispatcher (:class:`telegram.ext.Dispatcher`):
        """
        if not dispatcher.use_context:
            raise ValueError('CallbackContext should not be used with a non context aware '
                             'dispatcher!')
        self._dispatcher = dispatcher
        self.chat_data = None
        self.user_data = None
        self.args = None
        self.matches = None
        self.error = None
        self.job = None

    @classmethod
    def from_error(cls, update, error, dispatcher):
        self = cls.from_update(update, dispatcher)
        self.error = error
        return self

    @classmethod
    def from_update(cls, update, dispatcher):
        self = cls(dispatcher)
        if update is not None and isinstance(update, Update):
            chat = update.effective_chat
            user = update.effective_user

            if chat:
                self.chat_data = dispatcher.chat_data[chat.id]
            if user:
                self.user_data = dispatcher.user_data[user.id]
        return self

    @classmethod
    def from_job(cls, job, dispatcher):
        self = cls(dispatcher)
        self.job = job
        return self

    def update(self, data):
        self.__dict__.update(data)

    @property
    def bot(self):
        """:class:`telegram.Bot`: The bot associated with this context."""
        return self._dispatcher.bot

    @property
    def job_queue(self):
        """
        :class:`telegram.ext.JobQueue`: The ``JobQueue`` used by the
            :class:`telegram.ext.Dispatcher` and (usually) the :class:`telegram.ext.Updater`
            associated with this context.

        """
        return self._dispatcher.job_queue

    @property
    def update_queue(self):
        """
        :class:`queue.Queue`: The ``Queue`` instance used by the
            :class:`telegram.ext.Dispatcher` and (usually) the :class:`telegram.ext.Updater`
            associated with this context.

        """
        return self._dispatcher.update_queue

    @property
    def match(self):
        """
        `Regex match type`: The first match from :attr:`matches`.
            Useful if you are only filtering using a single regex filter.
            Returns `None` if :attr:`matches` is empty.
        """
        try:
            return self.matches[0]  # pylint: disable=unsubscriptable-object
        except (IndexError, TypeError):
            return None
