#!/usr/bin/env python
#
# A library that provides a Python interface to the Telegram Bot API
# Copyright (C) 2015-2022
# Leandro Toledo de Souza <devs@python-telegram-bot.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser Public License for more details.
#
# You should have received a copy of the GNU Lesser Public License
# along with this program.  If not, see [http://www.gnu.org/licenses/].
"""This module contains an object that represents a Telegram KeyboardButton."""

from typing import TYPE_CHECKING, Any, Optional

from telegram._keyboardbuttonpolltype import KeyboardButtonPollType
from telegram._telegramobject import TelegramObject
from telegram._utils.types import JSONDict
from telegram._webappinfo import WebAppInfo

if TYPE_CHECKING:
    from telegram import Bot


class KeyboardButton(TelegramObject):
    """
    This object represents one button of the reply keyboard. For simple text buttons String can be
    used instead of this object to specify text of the button.

    Objects of this class are comparable in terms of equality. Two objects of this class are
    considered equal, if their :attr:`text`, :attr:`request_contact`, :attr:`request_location`,
    :attr:`request_poll` and :attr:`web_app` are equal.

    Note:
        * Optional fields are mutually exclusive.
        * :attr:`request_contact` and :attr:`request_location` options will only work in Telegram
          versions released after 9 April, 2016. Older clients will display unsupported message.
        * :attr:`request_poll` option will only work in Telegram versions released after 23
          January, 2020. Older clients will display unsupported message.
        * :attr:`web_app` option will only work in Telegram versions released after 16 April, 2022.
          Older clients will display unsupported message.

    .. versionchanged:: 20.0
       :attr:`web_app` is considered as well when comparing objects of this type in terms of
       equality.

    Args:
        text (:obj:`str`): Text of the button. If none of the optional fields are used, it will be
            sent to the bot as a message when the button is pressed.
        request_contact (:obj:`bool`, optional): If :obj:`True`, the user's phone number will be
            sent as a contact when the button is pressed. Available in private chats only.
        request_location (:obj:`bool`, optional): If :obj:`True`, the user's current location will
            be sent when the button is pressed. Available in private chats only.
        request_poll (:class:`KeyboardButtonPollType`, optional): If specified, the user will be
            asked to create a poll and send it to the bot when the button is pressed. Available in
            private chats only.
        web_app (:class:`WebAppInfo`, optional): If specified, the described `Web App
            <https://core.telegram.org/bots/webapps>`_ will be launched when the button is pressed.
            The Web App will be able to send a :attr:`Message.web_app_data` service message.
            Available in private chats only.

            .. versionadded:: 20.0

    Attributes:
        text (:obj:`str`): Text of the button.
        request_contact (:obj:`bool`): Optional. The user's phone number will be sent.
        request_location (:obj:`bool`): Optional. The user's current location will be sent.
        request_poll (:class:`KeyboardButtonPollType`): Optional. If the user should create a poll.
        web_app (:class:`WebAppInfo`): Optional. If the described Web App will be launched when the
            button is pressed.

            .. versionadded:: 20.0
    """

    __slots__ = ("request_location", "request_contact", "request_poll", "text", "web_app")

    def __init__(
        self,
        text: str,
        request_contact: bool = None,
        request_location: bool = None,
        request_poll: KeyboardButtonPollType = None,
        web_app: WebAppInfo = None,
        **_kwargs: Any,
    ):
        # Required
        self.text = text
        # Optionals
        self.request_contact = request_contact
        self.request_location = request_location
        self.request_poll = request_poll
        self.web_app = web_app

        self._id_attrs = (
            self.text,
            self.request_contact,
            self.request_location,
            self.request_poll,
            self.web_app,
        )

    @classmethod
    def de_json(cls, data: Optional[JSONDict], bot: "Bot") -> Optional["KeyboardButton"]:
        """See :meth:`telegram.TelegramObject.de_json`."""
        data = cls._parse_data(data)

        if not data:
            return None

        data["request_poll"] = KeyboardButtonPollType.de_json(data.get("request_poll"), bot)
        data["web_app"] = WebAppInfo.de_json(data.get("web_app"), bot)

        return cls(**data)
