import numpy as np
import pandas as pd

from tide.meteo import sun_position, beam_component, ground_diffuse, sky_diffuse

SURFACE_TILT = 35
SURFACE_AZIMUTH = 154
LATITUDE = 44.844
LONGITUDE = -0.564

# Source Elevation, Azimuth  and projection: pvlib
# https://pvlib-python.readthedocs.io/en/stable/
# Source radiations :
# https://www.soda-pro.com/web-services/radiation/cams-radiation-service
TEST_DF = pd.DataFrame(
    {
        "Elevation": [
            -23.15501,
            -22.13249,
            -18.45929,
            -12.53057,
            -4.84902,
            4.10789,
            13.93921,
            24.31203,
            34.91997,
            45.40703,
            55.20735,
            63.16619,
            67.00673,
            64.68676,
            57.53711,
            48.06597,
            37.68394,
            27.06107,
            16.58682,
            6.57083,
            -2.66638,
            -10.74418,
            -17.20259,
            -21.53409,
        ],
        "Azimuth": [
            358.02192,
            13.0521,
            27.35737,
            40.41785,
            52.15285,
            62.80058,
            72.77237,
            82.59633,
            92.97471,
            104.99125,
            120.57558,
            143.12468,
            175.30016,
            209.27264,
            234.30403,
            251.31949,
            264.03145,
            274.70758,
            284.59487,
            294.46803,
            304.8913,
            316.30955,
            329.02074,
            343.04711,
        ],
        "BHI": [
            0.0,
            0.0,
            0.0,
            0.0,
            1.2421,
            51.4441,
            177.8372,
            303.2679,
            479.3531,
            455.9864,
            182.644,
            179.1913,
            185.5595,
            260.3788,
            206.3438,
            331.3794,
            283.1507,
            213.5861,
            80.4939,
            5.5889,
            0.0,
            0.0,
            0.0,
            0.0,
        ],
        "DHI": [
            0.0,
            0.0,
            0.0,
            0.0,
            4.9005,
            47.5719,
            85.0891,
            121.5737,
            128.3813,
            222.8379,
            373.4821,
            430.0031,
            437.3183,
            402.0775,
            368.7997,
            227.0156,
            152.701,
            92.2861,
            57.7285,
            12.5657,
            0.0,
            0.0,
            0.0,
            0.0,
        ],
        "BNI": [
            0.0,
            0.0,
            0.0,
            0.0,
            23.3832,
            302.5096,
            537.0757,
            610.9364,
            742.788,
            593.7701,
            216.0035,
            196.5999,
            202.4893,
            297.6538,
            257.1175,
            492.3087,
            521.9771,
            573.8196,
            383.0992,
            68.5959,
            0.0,
            0.0,
            0.0,
            0.0,
        ],
        "proj_tilt_35_az_154_alb_025": [
            0.0,
            0.0,
            0.0,
            0.0,
            4.59624,
            59.63713,
            234.91605,
            428.0604,
            648.03782,
            721.2353,
            556.59217,
            598.58417,
            606.8154,
            642.69273,
            539.49189,
            495.07234,
            328.96706,
            155.0263,
            55.63311,
            11.83986,
            0.0,
            0.0,
            0.0,
            0.0,
        ],
    },
    index=pd.date_range("2009-07-12 00:00:00", freq="h", periods=24, tz="UTC"),
)


class TestScience:
    def test_sun_position(self):
        np.testing.assert_allclose(
            TEST_DF[["Elevation", "Azimuth"]].to_numpy(),
            np.array(
                [sun_position(date, LATITUDE, LONGITUDE) for date in TEST_DF.index]
            ),
            atol=1,
        )

        np.testing.assert_allclose(
            TEST_DF["proj_tilt_35_az_154_alb_025"],
            beam_component(
                SURFACE_TILT,
                SURFACE_AZIMUTH,
                90 - TEST_DF["Elevation"],
                TEST_DF["Azimuth"],
                TEST_DF["BNI"],
            )
            + sky_diffuse(SURFACE_TILT, TEST_DF["DHI"])
            + ground_diffuse(SURFACE_TILT, TEST_DF["DHI"] + TEST_DF["BHI"]),
            rtol=0.01,
        )
