# Copyright 2009-2017 Ram Rachum.
# This program is distributed under the MIT license.

try:
    import enum
except ImportError:
    import imp
    try:
        imp.find_module('enum')
    except ImportError:
        raise Exception(
            'You don\'t have the standard library Python package `enum`, '
            'which I guess means you\'re running Python 3.3 or earlier. '
            'Please either install the backported `enum34` module by running '
            '`pip install enum34` or upgrade your Python version to 3.4 or '
            'later.'
        )
    else:
        raise 
import functools

from python_toolbox import caching


class EnumType(enum.EnumMeta):
    '''Metaclass for our kickass enum type.'''
    def __dir__(cls):
        # working around Python bug 22506 that would be fixed in Python 3.5.
        return type.__dir__(cls) + cls._member_names_
    
    __getitem__ = lambda self, i: self._values_tuple[i]
    # This `__getitem__` is important, so we could feed enum types straight
    # into `ProductSpace`.
    
    _values_tuple = caching.CachedProperty(tuple)
    
    
    
@functools.total_ordering
class _OrderableEnumMixin:
    '''
    Mixin for an enum that has an order between items.
    
    We're defining a mixin rather than defining these things on `CuteEnum`
    because we can't use `functools.total_ordering` on `Enum`, because `Enum`
    has exception-raising comparison methods, so `functools.total_ordering`
    doesn't override them.
    '''
    number = caching.CachedProperty(
        lambda self: type(self)._values_tuple.index(self)
    )
    __lt__ = lambda self, other: isinstance(other, CuteEnum) and \
                                                   (self.number < other.number)
    
    
class CuteEnum(_OrderableEnumMixin, enum.Enum, metaclass=EnumType):
    '''
    An improved version of Python's builtin `enum.Enum` type.
    
    `CuteEnum` provides the following benefits:
    
      - Each item has a property `number` which is its serial number in the
        enum.
        
      - Items are comparable with each other based on that serial number. So
        sequences of enum items can be sorted.
        
      - The enum type itself can be accessed as a sequence, and you can access
        its items like this: `MyEnum[7]`.
      
    '''