---
title: Обработка запросов
icon: 
---

# Обработка запросов

## Роутер

При любом взаимодействии с ботом — например, при личном сообщении, добавлении в групповой чат или канал — бот получает **обновление (update)** от сервера.

Для обработки этих событий используется `Router`. Он определяет, какие функции нужно вызвать при получении конкретного типа апдейта. Это позволяет централизованно управлять логикой обработки различных типов сообщений:

```python
from trueconf import Router

r = Router()
```

Чтобы отловить обновление с методом [`SendMessage`](https://trueconf.ru/docs/chatbot-connector/ru/server-requests/#sendMessage), функцию-обработчик оборачивают в декоратор:

```python
@r.message()
async def on_message(message): ...
```

## Поддержка фильтров

Роутеры поддерживают фильтры на основе библиотеки [magic-filter](https://github.com/aiogram/magic-filter). Для этого используется объект `F`:

```python
from trueconf import F
```

Фильтры позволяют обрабатывать только те апдейты, которые соответствуют заданным условиям. Например:

```python
# Сообщения только с текстом
@r.message(F.text)
async def on_message(message): ...

# Сообщения только с изображениями
@r.message(F.photo)
async def on_photo(message): ...

# Сообщения от конкретного пользователя
@r.message(F.from_user.id == "elisa")
async def on_elisa(message): ...
```

!!! Tip
    Более подробные примеры использования фильтров вы можете найти в [разделе Фильтры](filters.md).

## Диспетчер

Все созданные роутеры необходимо добавить в диспетчер (Dispatcher).
Именно он объединяет обработчики и управляет маршрутизацией апдейтов:

```python
from trueconf import Dispatcher

dp = Dispatcher()
dp.include_router(r)
```

## Приоритеты обработчиков

* Роутеры и их обработчики проверяются в порядке подключения через Dispatcher.include_router().
* Внутри одного роутера обработчики также идут по порядку объявления.
* При первом совпадении фильтров обработчик выполняется, и дальнейшие совпадения не проверяются (поведение по умолчанию).

Это означает, что если у вас есть несколько обработчиков с одинаковыми фильтрами:

```python
@r.message(F.text == "Hello")
async def handler1(message):
    await message.answer("Первый")

@r.message(F.text == "Hello")
async def handler2(message):
    await message.answer("Второй")
```

То сработает **только handler1**, а до handler2 исполнение уже не дойдёт.

Чтобы задействовать оба обработчика, используйте разные фильтры или объедините их в один с дополнительной логикой внутри функции.

!!! Tip
    Для разделения логики рекомендуется создавать несколько роутеров (например, commands_router, messages_router, admin_router) и подключать их в диспетчере в нужном порядке. Такой подход помогает структурировать код и упрощает поддержку бота.

## Рекомендации по организации кода

- Обычно роутеры выносят в отдельные модули (например, handlers/messages.py), а затем подключают их в главном модуле бота через include_router.
- Это позволяет разделять обработчики по областям ответственности: сообщения, фото, командыа и т. д.
- Диспетчер (Dispatcher) можно рассматривать как центральный управляющий компонент, объединяющий логику обработки всех событий.


