"""
Tests for AccountingSupplierParty and AccountingCustomerParty wrappers.

Tests the wrapper components that contain Party elements, used to
represent the supplier (seller) and customer (buyer) in UBL documents.
"""

import unittest

from lxml import etree

from ubl.models.aggregate_components import (
    AccountingCustomerParty,
    AccountingSupplierParty,
    Party,
    PostalAddress,
)


class TestAccountingPartyWrappers(unittest.TestCase):
    """Test AccountingSupplierParty and AccountingCustomerParty wrappers."""

    def setUp(self):
        """Set up test fixtures."""
        self.address = PostalAddress(
            street_name="Rue de Test",
            city_name="Brussels",
            postal_zone="1000",
            country="BE",
        )

        self.party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
        )

    def test_create_accounting_supplier_party(self):
        """Create AccountingSupplierParty wrapper."""
        supplier = AccountingSupplierParty(party=self.party)

        self.assertIsInstance(supplier.party, Party)
        self.assertEqual(supplier.party.name, "P4X SA")

    def test_create_accounting_customer_party(self):
        """Create AccountingCustomerParty wrapper."""
        customer = AccountingCustomerParty(party=self.party)

        self.assertIsInstance(customer.party, Party)
        self.assertEqual(customer.party.name, "P4X SA")

    def test_supplier_and_customer_are_separate_types(self):
        """Supplier and customer are distinct types."""
        supplier = AccountingSupplierParty(party=self.party)
        customer = AccountingCustomerParty(party=self.party)

        # Different types
        self.assertNotEqual(type(supplier), type(customer))

    def test_accounting_supplier_party_to_xml(self):
        """AccountingSupplierParty generates correct XML structure."""
        supplier = AccountingSupplierParty(party=self.party)

        xml = supplier.to_xml()
        self.assertEqual(
            xml.tag, "{urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2}AccountingSupplierParty",
        )

        # Should have Party child
        party_elements = xml.findall("{urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2}Party")
        self.assertEqual(len(party_elements), 1)

    def test_accounting_customer_party_to_xml(self):
        """AccountingCustomerParty generates correct XML structure."""
        customer = AccountingCustomerParty(party=self.party)

        xml = customer.to_xml()
        self.assertEqual(
            xml.tag, "{urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2}AccountingCustomerParty",
        )

        # Should have Party child
        party_elements = xml.findall("{urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2}Party")
        self.assertEqual(len(party_elements), 1)

    def test_accounting_supplier_party_xml_roundtrip(self):
        """AccountingSupplierParty round-trip preservation."""
        supplier = AccountingSupplierParty(party=self.party)

        # Generate XML
        xml = supplier.to_xml()
        xml_str = etree.tostring(xml, encoding="unicode")

        # Parse back
        parsed_xml = etree.fromstring(xml_str)
        reconstructed = AccountingSupplierParty.from_xml(parsed_xml)

        # Should preserve party name
        self.assertEqual(str(reconstructed.party.name), "P4X SA")
        self.assertEqual(reconstructed.party.country_code, "BE")

    def test_accounting_customer_party_xml_roundtrip(self):
        """AccountingCustomerParty round-trip preservation."""
        customer = AccountingCustomerParty(party=self.party)

        # Generate XML
        xml = customer.to_xml()
        xml_str = etree.tostring(xml, encoding="unicode")

        # Parse back
        parsed_xml = etree.fromstring(xml_str)
        reconstructed = AccountingCustomerParty.from_xml(parsed_xml)

        # Should preserve party details
        self.assertEqual(str(reconstructed.party.name), "P4X SA")
        self.assertEqual(reconstructed.party.country_code, "BE")

    def test_accounting_party_with_complex_party(self):
        """Wrapper with complex party (VAT + registration + contact)."""
        from ubl.models.aggregate_components import Contact

        contact = Contact(electronic_mail='test@p4x.be', telephone='+32123456789')

        complex_party = Party(
            name="LevIT SC",
            country_code="BE",
            postal_address=self.address,
            vat="BE0597601756",
            registration="0597601756",
            contact=contact,
            website_uri="https://www.levit.be",
            peppol_participant_ids=["9925:be0597601756"],
        )

        supplier = AccountingSupplierParty(party=complex_party)

        # Should preserve all party details
        self.assertEqual(supplier.party.name, "LevIT SC")
        self.assertEqual(supplier.party.vat, "BE0597601756")
        self.assertEqual(supplier.party.registration, "0597601756")
        self.assertIsNotNone(supplier.party.contact)
        self.assertEqual(supplier.party.website_uri, "https://www.levit.be")
        self.assertEqual(len(supplier.party.peppol_participant_ids), 1)

    def test_xml_contains_party_identifications(self):
        """XML contains all party identifications."""
        supplier = AccountingSupplierParty(party=self.party)

        xml = supplier.to_xml()
        xml_str = etree.tostring(xml, encoding="unicode")

        # Should contain PartyIdentification elements
        self.assertIn("PartyIdentification", xml_str)

        # Should have endpoint ID
        self.assertIn("EndpointID", xml_str)

    def test_xml_namespace_collection(self):
        """Wrapper XML contains expected namespaces."""
        supplier = AccountingSupplierParty(party=self.party)

        xml = supplier.to_xml()
        xml_str = etree.tostring(xml, encoding='unicode')

        # Should have CAC namespace
        self.assertIn('xmlns:cac=', xml_str)

        # Should also have CBC namespace (from Identifier, etc.)
        self.assertIn('xmlns:cbc=', xml_str)


if __name__ == "__main__":
    unittest.main()
