import os
import re
import datetime
import sys

def is_valid_macro_name(macro_name):
    """
    Determine whether a macro name is valid using Python's standard library methods.
    
    Args:
        macro_name: The macro name to check.
        
    Returns:
        bool: True if it's a valid Python identifier, False otherwise.
    """
    # Empty string is invalid
    if not macro_name:
        return False
    
    # Use str.isidentifier() to check for valid identifier syntax
    return macro_name.isidentifier()

def extract_macro_name(line):
    """Extract the macro name from a #define line (handles spaces between # and define)."""
    line = line.strip()

    # Match '#', optional spaces, 'define', spaces, and the macro name
    match = re.match(r'^#\s*define\s+([A-Za-z_][A-Za-z0-9_]*)', line)
    if not match:
        return None
    
    candidate = match.group(1)
    
    # Validate with standard identifier rules
    if candidate and is_valid_macro_name(candidate):
        return candidate
    return None

def is_standard_python_macro(macro_name):
    """
    Check whether a macro follows Python's standard naming conventions.
    Rules: Starts with Py, PY, _Py, _PY, or ends with _H.
    """
    standard_prefixes = ('Py', 'PY', '_Py', '_PY')
    return macro_name.startswith(standard_prefixes) or macro_name.endswith('_H')

def generate_undef_code(macro_name):
    """Generate the code to undefine a macro."""
    return f"""#ifndef DONOTUNDEF_{macro_name}
#ifdef {macro_name}
#undef {macro_name}
#endif
#endif

"""

def generate_python_undef_header(pyconfig_path, output_path=None):
    """
    Generate the Python_undef.h header file.
    
    Args:
        pyconfig_path: Path to pyconfig.h
        output_path: Output file path, defaults to Python_undef.h in the current directory.
    """
    if output_path is None:
        file_dir = os.path.dirname(os.path.abspath(__file__))
        output_path = f'{file_dir}/include/Python_undef.h'
        if not os.path.exists(f'{file_dir}/include'):
            os.makedirs(f'{file_dir}/include')
    
    # Read pyconfig.h
    try:
        with open(pyconfig_path, 'r', encoding='utf-8') as f:
            lines = f.readlines()
    except FileNotFoundError:
        print(f"Error: File not found {pyconfig_path}")
        return False
    except Exception as e:
        print(f"Error reading file: {e}")
        return False
    
    # Collect macros
    macros_to_undef = []
    all_macros = []
    invalid_macros = []
    
    print("Analyzing pyconfig.h...")
    
    for i, line in enumerate(lines, 1):
        macro_name = extract_macro_name(line)
        if macro_name:
            all_macros.append(macro_name)
            
            # New rule: any macro not starting with Py/PY/_Py/_PY and not ending with _H is considered non-standard
            if not is_standard_python_macro(macro_name):
                macros_to_undef.append(macro_name)
                print(f"Line {i:4d}: Found non-standard macro '{macro_name}'")
        else:
            # Check if line looks like a define but has invalid name
            line = line.strip()
            if line.startswith('#'):
                m = re.match(r'^#\s*define\s+(\S+)', line)
                if m:
                    candidate = m.group(1)
                    if candidate and not is_valid_macro_name(candidate):
                        invalid_macros.append((i, candidate))
    
    # Deduplicate and sort
    macros_to_undef = sorted(set(macros_to_undef))
    
    # Header section
    header = f"""/*
 * Python_undef.h - Automatically generated macro undefinition header
 * 
 * This file is automatically generated from {os.path.basename(pyconfig_path)}
 * Contains macros that may need to be undefined to avoid conflicts with other libraries.
 * 
 * WARNING: This is an automatically generated file. Do not edit manually.
 * 
 * Usage:
 *   #include <Python.h>
 *   #include <Python_undef.h>
 *   #include <other_library_headers.h>
 * 
 * To preserve specific macros, define before including this header:
 *   #define DONOTUNDEF_MACRO_NAME
 * 
 * Generation rules:
 *   - Macros starting with Py_, PY_, _Py, _PY are preserved (Python standard)
 *   - Macros ending with _H are preserved (header guards)
 *   - All other macros are undefined
 *   - Macro name validation uses Python's standard identifier checking
 * 
 * Generated from: {os.path.abspath(pyconfig_path)}
 * Generated at: {datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
 * Total valid macros found: {len(all_macros)}
 * Macros to undef: {len(macros_to_undef)}
 * Invalid macro names skipped: {len(invalid_macros)}
 */

#ifndef PYTHON_UNDEF_H
#define PYTHON_UNDEF_H

#ifndef Py_PYTHON_H
#  error "Python_undef.h must be included *after* Python.h"
#endif

/*
 * Platform Note:
 * - The COMPILER macro is primarily defined in pyconfig.h on Windows
 * - Other platforms define compiler info in Python/getcompiler.c
 * - This macro and others can conflict with libraries such as V8
 */

"""
    
    # Generate undef code sections
    undef_sections = []
    for macro_name in macros_to_undef:
        undef_sections.append(generate_undef_code(macro_name))
    
    # Footer
    footer = """#endif /* PYTHON_UNDEF_H */
"""
    
    # Write output
    try:
        with open(output_path, 'w', encoding='utf-8', newline='\n') as f:
            f.write(header)
            f.writelines(undef_sections)
            f.write(footer)
        
        print(f"\n{'='*60}")
        print(f"Successfully generated: {output_path}")
        print(f"{'='*60}")
        print("Summary:")
        print(f"  - Total valid macro definitions: {len(all_macros)}")
        print(f"  - Macros to undefine: {len(macros_to_undef)}")
        print(f"  - Preserved standard macros: {len(all_macros) - len(macros_to_undef)}")
        print(f"  - Invalid macro names skipped: {len(invalid_macros)}")
        
        if invalid_macros:
            print(f"\nSkipped invalid macro names:")
            for line_num, invalid_macro in invalid_macros[:10]:  # show only first 10
                print(f"  Line {line_num:4d}: '{invalid_macro}'")
            if len(invalid_macros) > 10:
                print(f"  ... and {len(invalid_macros) - 10} more")
        
        if macros_to_undef:
            print(f"\nMacros to undefine (first 50):")
            for i, macro in enumerate(macros_to_undef[:50], 1):
                print(f"  {i:3d}. {macro}")
            if len(macros_to_undef) > 50:
                print(f"  ... and {len(macros_to_undef) - 50} more")
        
        print(f"\nUsage Notes:")
        print(f"  1. Include this file before including other library headers but must be after '<Python.h>'.")
        print(f"  2. Use DONOTUNDEF_XXX to protect macros that must be kept.")
        print(f"  3. Regenerate this file whenever rebuilding Python.")
        
        return True
        
    except Exception as e:
        print(f"Error writing file: {e}")
        return False

def main():
    import sysconfig
    from pathlib import Path
    if sys.argv[1:]:
        if sys.argv[1] in ('-h', '--help'):
            print("""Usage:
python -m python_undef --generate
    Generate Python_undef.h based on the system's pyconfig.h.
python -m python_undef --include
    Print the include path where Python_undef.h is located.""")
            sys.exit(0)
        elif sys.argv[1] == '--generate': 
            include_dir = Path(sysconfig.get_path('include'))
            print(f"\n{'='*60}")
            print("Note: Python keywords are not excluded since they are valid macro names in C/C++.")
            print(f"{'='*60}")
            
            pyconfig_path = include_dir / "pyconfig.h"
            
            if os.path.exists(pyconfig_path):
                success = generate_python_undef_header(pyconfig_path)
                
                if success:
                    print(f"\n✅ Generation complete!")
                    print(f"💡 Tip: Place Python_undef.h inside module 'python_def' include search path.")
                    sys.exit(0)
                else:
                    print(f"\n❌ Generation failed!")
                    sys.exit(1)
                    
            else:
                print(f"File {pyconfig_path} not found.")
                print("Please update the pyconfig_path variable to the actual pyconfig.h path.")
                print("\nTypical paths on Windows:")
                print("  C:\\\\Python3x\\\\include\\\\pyconfig.h")
                print("\nTypical paths on Unix/Linux:")
                print("  /usr/include/python3.x/pyconfig.h")
                print("  /usr/local/include/python3.x/pyconfig.h")
                sys.exit(1)
        elif sys.argv[1] == "--include":
            file_dir = os.path.dirname(os.path.abspath(__file__))
            if not Path(file_dir).exists():
                print("File not found. Use 'python -m python_undef --generate' to generate the header first.")
                sys.exit(1)
            include_path = os.path.abspath(os.path.join(file_dir, 'include'))
            print(include_path)
            sys.exit(0)
        else:
            print("Unknown argument. Use --help for usage information.")
            sys.exit(1)
    else:
        print("No arguments provided. Use --help for usage information.")
        sys.exit(1)
