# -*- encoding: utf-8 -*-
# Copyright (c) 2015 b<>com
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import unicode_literals

import datetime
import platform

from dateutil.tz import tzlocal
import pytz


class Measurement(object):

    def __init__(self, name, unit, type_, value, resource_id, host=None,
                 timestamp=None, resource_metadata=None):
        """
        :param name: The name of the measurement (Should the Probe ID)
        :type name: str
        :param unit: The measurement unit sign (International System of Units)
        :type unit: str
        :param type_: Should be either 'gauge', 'cumulative', 'delta' or ''
        :type type_: str
        :param value:
        :type value: int, float or long
        :param resource_id: The ID of the resource
        :type resource_id: str
        :param host: The IP address / FQDN of the host from which the
            measurement was obtained (By default, takes the current hostname)
        :type host: str
        :param timestamp: The datetime at which the measurement was realized,
            generated by default with the current datetime (Should be UTC)
        :type timestamp: str (ISO 8601)
        :param resource_metadata: A mapping of metadata values
        :type resource_metadata: dict which may contain 'host', 'ttl', 'state'
            as well as some other custom values
        """
        self.name = name
        self.unit = unit
        self.type = type_
        self.value = value
        self.resource_id = resource_id
        self.host = host or platform.node()
        self.timestamp = timestamp or Measurement.utc_now().isoformat()
        self.resource_metadata = resource_metadata or {}

    @classmethod
    def utc_now(cls):
        """Gets current datetime and converts it as UTC-based datetime"""
        local_now = datetime.datetime.now(tzlocal())
        utc_now = pytz.utc.normalize(local_now)
        return utc_now

    def as_dict(self):

        return {
            "name": self.name,  # self.probe_id
            "timestamp": self.timestamp,  # datetime.datetime.now().isoformat()
            "unit": self.unit,  # self.unit,
            "type": self.type,  # "gauge"
            "value": self.value,  # value
            "resource_id": self.resource_id,  # ""
            "host": self.host,  # platform.node()
            "resource_metadata": self.resource_metadata,  # resource_metadata
        }
