# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# jvm.py
# Copyright (C) 2014-2024 Fracpete (pythonwekawrapper at gmail dot com)

import jpype
from jpype import JClass
import os
import glob
import logging


started = None
""" whether the JVM has been started """

with_package_support = None
""" whether JVM was started with package support """

automatically_install_packages = None
""" whether to automatically install missing packages """

# logging setup
logging.basicConfig()
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)


def lib_dir():
    """
    Returns the "lib" directory path.

    :return: the path to the "lib" directory
    :rtype: str
    """

    rootdir = os.path.split(os.path.dirname(__file__))[0]
    return rootdir + os.sep + "lib"


def add_bundled_jars(cp):
    """
    Adds the bundled jars to the JVM's classpath.

    :param cp: the list to append the classpath to
    :type cp: list
    """
    # determine lib directory with jars
    libdir = lib_dir()

    # add jars from lib directory
    for l in glob.glob(libdir + os.sep + "*.jar"):
        if l.lower().find("-src.") == -1:
            cp.append(str(l))


def add_system_classpath(cp):
    """
    Adds the system's classpath to the JVM's classpath.

    :param cp: the list to append the classpath to
    :type cp: list
    """
    if 'CLASSPATH' in os.environ:
        parts = os.environ['CLASSPATH'].split(os.pathsep)
        for part in parts:
            cp.append(part)
    else:
        logger.warning("Cannot add system's classpath, as environment variable CLASSPATH not set.")


def start(class_path=None, bundled=True, packages=False, system_cp=False, max_heap_size=None, system_info=False,
          auto_install=False, logging_level=logging.DEBUG):
    """
    Initializes the javabridge connection (starts up the JVM).

    :param class_path: the additional classpath elements to add
    :type class_path: list
    :param bundled: whether to add jars from the "lib" directory
    :type bundled: bool
    :param packages: whether to add jars from Weka packages as well (bool) or an alternative Weka home directory (str)
    :type packages: bool or str
    :param system_cp: whether to add the system classpath as well
    :type system_cp: bool
    :param max_heap_size: the maximum heap size (-Xmx parameter, eg 512m or 4g)
    :type max_heap_size: str
    :param system_info: whether to print the system info (generated by weka.core.SystemInfo)
    :type system_info: bool
    :param auto_install: whether to automatically install missing Weka packages (based on suggestions); in conjunction with package support
    :type auto_install: bool
    :param logging_level: the logging level to use for this module, e.g., logging.DEBUG or logging.INFO
    :type logging_level: int
    """
    global started
    global with_package_support
    global automatically_install_packages
    global logger

    logger.setLevel(logging_level)

    if started is not None:
        logger.info("JVM already running, call jvm.stop() first")
        return

    full_cp = []

    # add user-defined jars first
    if class_path is not None:
        for cp in class_path:
            logger.debug("Adding user-supplied classpath=" + cp)
            full_cp.append(cp)

    if bundled:
        logger.debug("Adding bundled jars")
        add_bundled_jars(full_cp)

    if system_cp:
        logger.debug("Adding system classpath")
        add_system_classpath(full_cp)

    logger.debug("Classpath=" + str(full_cp))
    logger.debug("MaxHeapSize=" + ("default" if (max_heap_size is None) else max_heap_size))

    args = []
    weka_home = None
    if packages is not None:
        if isinstance(packages, bool):
            if packages:
                with_package_support = True
                logger.debug("Package support enabled")
            else:
                logger.debug("Package support disabled")
                args.append("-Dweka.packageManager.loadPackages=false")
        if isinstance(packages, str):
            if os.path.exists(packages) and os.path.isdir(packages):
                logger.debug("Using alternative Weka home directory: " + packages)
                weka_home = packages
                with_package_support = True
            else:
                logger.warning("Invalid Weka home: " + packages)

    if with_package_support and auto_install:
        logger.debug("Automatically installing missing Weka packages (based on suggestions).")
        automatically_install_packages = True

    # headless mode
    args.append("-Djava.awt.headless=true")

    jpype.startJVM(*args, classpath=full_cp, convertStrings=True)
    started = True

    if weka_home is not None:
        from weka.core.classes import Environment
        env = Environment.system_wide()
        logger.debug("Using alternative Weka home directory: " + packages)
        env.add_variable("WEKA_HOME", weka_home)

    # initialize package manager
    JClass("weka.core.WekaPackageManager").loadPackages(False)

    # output system info
    if system_info:
        logger.debug("System info:")
        info = JClass("weka.core.SystemInfo")().getSystemInfo()
        for k in info.keys():
            logger.debug(k + "=" + info[k])


def stop():
    """
    Kills the JVM.
    """
    global started
    if started is not None:
        started = None
        jpype.shutdownJVM()
