from python2latex import TexFile, build
from python2latex.utils import open_file_with_default_program
"""
TODO:
    - Parsing of same packages, and colors in preamble but with different options
    - Parsing for 'include' files.
"""


class Template:
    """
    Class that allows to insert TeX commands inside an already existing tex file and then compile it. Useful to make figures and tables with python2latex and insert them into your project without needing to copy and paste.

    To tell python2latex where to insert an object, write the line
        %! python2latex-anchor = *the_name_of_your_object_here*
    The script will automatically insert the tex under this anchor and add a closing statement as
        %! python2latex-end-anchor = *the_name_of_your_object_here*

    The preamble will also be automatically updated, where you can find added commands right under the anchor
        %! python2latex-preamble

    See the examples for a more complete example.
    """
    def __init__(self, filename, filepath='.', output_filename=None, output_filepath=None):
        """
        Args:
            filename (str): Name of the input tex file without extension.
            filepath (str): Path where the input file is.
            output_filename (str or None): Name of the output file without the extension. If None, the name of the input file appended with '_rendered' will be used. If the output filename is the same as the input filename, the input filename will be overwrited, which can be useful but also dangerous if there is a problem in the code.
            output_filepath (str): Path where the rendered files will be placed. If None, the path of the input file will be used.
        """
        self.input_file = TexFile(filename, filepath)
        if output_filename is None:
            output_filename = filename + '_rendered'
        if output_filepath is None:
            output_filepath = filepath
        self.output_file = TexFile(output_filename, output_filepath)
        self.anchors = {}

    def _load_tex_file(self):
        """
        Returns the loaded tex file as a list of strings.
        """
        with open(self.input_file.path, 'r', encoding='utf8') as file:
            return [line.strip() for line in file]

    def _split_preamble(self, text):
        """
        Returns the preamble and the rest of the document.
        """
        for i, line in enumerate(text):
            if r'\begin{document}' in line:
                begin_document_line = i
                break
        return text[:begin_document_line], text[begin_document_line:]

    def _insert_tex_at_anchors(self, doc):
        anchors_position = {}
        for i, line in enumerate(doc):
            if line.startswith('%! python2latex-anchor'):
                _, anchor_name = line.replace(' ', '').split('=')
                anchors_position[anchor_name] = (i, None)

            if line.startswith('%! python2latex-end-anchor'):
                anchors_position[anchor_name] = (anchors_position[anchor_name][0], i)

        for i, (anchor_name, (start, end)) in enumerate(anchors_position.items()):
            if end:
                del doc[start + 1:end + 1]
            doc.insert(start + 1 + i, self.anchors[anchor_name])
            doc.insert(start + 2 + i, f'%! python2latex-end-anchor = {anchor_name}')

    def _update_preamble(self, preamble):
        # Removing old python2latex preamble
        for i, line in enumerate(preamble):
            if line == '%! python2latex-preamble':
                break
        else:
            preamble[i:]

        # Adding only new lines to preamble
        anchor_preambles = set(line for obj in self.anchors.values()
                               for line in obj.build_preamble().split('\n'))
        lines_to_add = []
        for line in anchor_preambles:
            if line not in preamble and line:
                lines_to_add.append(line)

        if lines_to_add:
            preamble.extend(['%! python2latex-preamble'] + lines_to_add)

    def render(self, compile_to_pdf=True, show_pdf=True):
        """
        Loads the input files, parses the tex to find the anchors, inserts the code generated by python2latex then saves it to disk.

        Args:
            compile_to_pdf (bool): If True, automatically call pdflatex to compile the generated tex file to pdf.

            show_pdf (bool): If True, the default pdf reader will be called to show the compiled pdf. This may not work well with non-read-only pdf viewer such as Acrobat Reader or Foxit Reader. Suggested readers are SumatraPDF on Windows and Okular or Evince on Linux.
        """
        tex = self._load_tex_file()
        preamble, doc = self._split_preamble(tex)
        self._insert_tex_at_anchors(doc)
        self._update_preamble(preamble)
        tex = '\n'.join(build(line) for line in preamble + doc)

        self.output_file.save(tex)

        if compile_to_pdf:
            self.output_file.compile_to_pdf()

            if show_pdf:
                open_file_with_default_program(self.output_file.filename, self.output_file.filepath)
