import logging

from typing_extensions import Self

from py2ts.data import (
    DerivedType,
    TSComplex,
    TSInterface,
    TypescriptType,
    ts_reference_str,
)
from py2ts.generate import generate_ts

log = logging.getLogger("py2ts")

prefix = """/*
 * This file is automatically generated by py2ts!
 * For more information, see:
 * https://github.com/semohr/py2ts
 */
"""


class TSBuilder:
    """Build a ts file from a list of types.

    This is a helper class to simplify the process of building a ts file.

    Usage:
    ```
    ts_builder = TSBuilder()

    # You may add a typeddict, dataclass or the internal tsinterface
    # to the builder.
    ts_builder
        .add(TypedDict)
        .add(TSInterface)

    # Save
    ts_builder.save_file("output.ts")
    ```
    """

    # Mapping of types to ts types _elements[n] <-> _ts_elements[n]
    _elements: list[type]
    _exclude: list[set[str]]
    _ts_elements: list[TypescriptType] | None = None

    def __init__(self):
        self._elements = []
        self._exclude = []

    def add(
        self,
        t: type,
        exclude: set[str] | None = None,
    ) -> Self:
        """Add a type to the builder.

        This will add the type to the list of types to be converted to TypeScript.

        Parameters
        ----------
        type : type
            The type to be added. This can be a dataclass, TypedDict or TSInterface.
        exclude : set[str] | None, optional
            A set of field names to exclude from the TypeScript interface. This is only used for
            TypedDicts and dataclasses. If None, all fields will be included.
        """
        self._elements.append(t)
        self._exclude.append(exclude or set())
        self._all_ts_elements = None
        return self

    @property
    def ts_types(self) -> set[TypescriptType]:
        """Get the types in the builder.

        As a unique set, walks all recursive types and removes duplicates.
        """
        elements: set[TypescriptType] = set()

        def resolve_recursive(e: TypescriptType) -> None:
            """Resolve the recursive types in the builder."""
            nonlocal elements
            elements.add(e)
            if isinstance(e, DerivedType):
                for t in e:
                    resolve_recursive(t)
            if isinstance(e, TSInterface):
                for t in e.elements.values():
                    resolve_recursive(t)
                if e.inheritance:
                    resolve_recursive(e.inheritance)
            return

        m = []
        for t, e in zip(self._elements, self._exclude):
            # Resolve the types in the builder
            ts_type = generate_ts(t)

            # Remove excluded fields
            if isinstance(ts_type, TSComplex):
                ts_type = ts_type.exclude(e)
            elif len(e) > 0:
                log.warning(
                    f"Excluding fields {e} from {type(ts_type)} is not supported, skipping."
                )

            resolve_recursive(ts_type)
            m.append(ts_type)

        self._ts_elements = m
        return elements

    @property
    def ts_elements(self) -> list[TypescriptType]:
        """Get the types in the builder as a list."""
        if self._ts_elements is None:
            self.ts_types  # Trigger the generation of ts_types
        return self._ts_elements  # type: ignore[return-value]

    def to_str(self) -> str:
        """Convert the builder to a string.

        This will convert the types in the builder to TypeScript and return a string
        representation of the TypeScript code.
        """
        return ts_reference_str(
            self.ts_types,
        )

    def save_file(self, filename: str) -> None:
        """Convert the types in the builder to TypeScript and save them to a file."""
        with open(filename, "w") as f:
            f.write(prefix)
            f.write(self.to_str())
