"""
To run the basic conversion as a script
"""

import argparse
import os
import ast
from .frontend import Generator2Ast
from .backend.codegen import Verilog

parser = argparse.ArgumentParser(
    description=__doc__, formatter_class=argparse.RawTextHelpFormatter
)

parser.add_argument(
    "input_file",
    type=str,
    help="Input file containing a python generator function",
)


parser.add_argument(
    "-o",
    "--output",
    type=str,
    help="Output file for System Verilog module",
    default="",
)

parser.add_argument(
    "-t",
    "--testbench",
    type=str,
    help="Output file for System Verilog testbench",
    default="",
)

parser.add_argument(
    "-c",
    "--test-case",
    type=str,
    help='Test case for testbench. Required to output testbench. E.g. `-c "(1, 2, 3, 4)"`',
    default="",
)

args = parser.parse_args()
input_file_path = parser.parse_args().input_file
input_file_stem = os.path.splitext(input_file_path)[0]
if args.output == "":
    args.output = input_file_stem + ".sv"

if args.testbench == "":
    args.testbench = input_file_stem + "_tb.sv"

with open(os.path.abspath(input_file_path), mode="r", encoding="utf8") as python_file:
    python = python_file.read()
    _locals = {}
    exec(python, None, _locals)  # grab's exec's populated scoped variables

    tree = ast.parse(python)
    function = tree.body[0]
    ir_generator = Generator2Ast(function)
    output = ir_generator.parse_statements(function.body, "")
    verilog = Verilog()
    verilog.from_ir(output, ir_generator.global_vars)
    verilog.setup_from_python(function)

    with open(os.path.abspath(args.output), mode="w+", encoding="utf8") as module_file:
        module_file.write(verilog.get_module().to_string())

    if args.test_case != "":
        with open(
            os.path.abspath(args.testbench), mode="w+", encoding="utf8"
        ) as tb_file:
            tb_file.write(verilog.get_testbench(ast.literal_eval(args.test_case)))
