import os
import subprocess
import sys
import tempfile
import urllib
import shutil
from pathlib import Path


PY_VERSION = "3.7.9"
WINDOWS    = 1
LINUX      = 2
MACOS      = 3
WIN_URL    = f'https://www.python.org/ftp/python/{PY_VERSION}/python-{PY_VERSION}-amd64.exe'
LINUX_URL  = f"https://www.python.org/ftp/python/{PY_VERSION}/Python-{PY_VERSION}.tgz"
TEMP_PATH  = os.path.join(tempfile.gettempdir(), f"python-{PY_VERSION}-amd64.exe")
SRC_DIR    = Path("/usr/src")
TAR_PATH   = SRC_DIR / f"Python-{PY_VERSION}.tgz"
BUILD_DIR  = SRC_DIR / f"Python-{PY_VERSION}"

def get_platform():
    if sys.platform.startswith("win"):
        return WINDOWS
    elif sys.platform.startswith("linux"):
        return LINUX
    elif sys.platform.startswith("darwin"):
        return MACOS
    else:
        raise EnvironmentError('Unknown platform. Use official download instructions: https://www.python.org/download/other/')

def detect_package_manager():
    if shutil.which("apt-get"):
        return "apt"
    elif shutil.which("dnf"):
        return "dnf"
    elif shutil.which("yum"):
        return "yum"
    elif shutil.which("pacman"):
        return "pacman"
    elif shutil.which("apk"):
        return "apk"
    else:
        return None


def run(cmd, check=True):
    print(f"[RUN] {' '.join(cmd)}")
    subprocess.run(cmd, check=check)

def install_deps():
    pm = detect_package_manager()
    if not pm:
        raise RuntimeError("Cannot detect supported package manager!")

    print(f"📦 Installing dependencies using {pm} ...")
    if pm == "apt":
        deps = [
            "build-essential", "libssl-dev", "zlib1g-dev", "libncurses5-dev",
            "libncursesw5-dev", "libreadline-dev", "libsqlite3-dev",
            "libgdbm-dev", "libdb5.3-dev", "libbz2-dev", "libexpat1-dev",
            "liblzma-dev", "tk-dev", "uuid-dev", "wget"
        ]
        run(["apt-get", "update"], check=False)
        run(["apt-get", "install", "-y"] + deps)
    elif pm in ["dnf", "yum"]:
        deps = [
            "gcc", "make", "zlib-devel", "bzip2-devel", "xz-devel", "libffi-devel",
            "readline-devel", "sqlite-devel", "tk-devel", "uuid-devel", "wget", "openssl-devel"
        ]
        run([pm, "install", "-y"] + deps)
    elif pm == "pacman":
        deps = ["base-devel", "zlib", "bzip2", "xz", "libffi", "readline", "sqlite", "tk", "wget", "openssl"]
        run(["pacman", "-Syu", "--noconfirm"] + deps)
    elif pm == "apk":
        deps = ["build-base", "zlib-dev", "bzip2-dev", "xz-dev", "libffi-dev",
                "readline-dev", "sqlite-dev", "tk-dev", "wget", "openssl-dev"]
        run(["apk", "add", "--no-cache"] + deps)
    else:
        raise RuntimeError(f"Unsupported package manager: {pm}")

def install_linux():
    if os.geteuid() != 0:
        raise EnvironmentError('Run installation with sudo: sudo pip install python3.7 . WARNING: python3.7 package will be installed as a global package.')

    pm = detect_package_manager()
    if not pm:
        raise RuntimeError("Cannot detect supported package manager!")

    print(f"📦 Installing dependencies using {pm} ...")
    install_deps()
    download_python()
    extract_python()
    build_python()

    print("\n✅ Python installation complete!")
    run(["/usr/local/bin/python3.7", "--version"], check=False)


def download_python():
    print(f"⬇️ Downloading Python {PY_VERSION} ...")
    SRC_DIR.mkdir(parents=True, exist_ok=True)
    BUILD_DIR.mkdir(parents=True, exist_ok=True)
    run(["wget", "-O", str(TAR_PATH), LINUX_URL])

def extract_python():
    print("📂 Extracting archive ...")
    run(["tar", "xzf", str(TAR_PATH), "-C", str(BUILD_DIR), "--strip-components=1"])

def build_python():
    print("⚙️ Configuring build ...")
    os.makedirs(BUILD_DIR, exist_ok=True)
    os.chdir(BUILD_DIR)
    run(["./configure", "--enable-optimizations"])

    print("🧱 Building Python (this may take a while)...")
    cpu_count = os.cpu_count() or 2
    run(["make", f"-j{cpu_count}"])
    run(["make", "altinstall"])  # don't overwrite system python3


def install_windows():
    with urllib.request.urlopen(WIN_URL) as response, open(TEMP_PATH, "wb") as out_file:
        # читаем по кусочкам (8 КБ)
        while chunk := response.read(8192):
            out_file.write(chunk)

    if os.path.exists(TEMP_PATH):
        subprocess.run(
            [TEMP_PATH, "/quiet", "InstallAllUsers=1", "PrependPath=1", "Include_pip=1"],
            check=True
        )

def install():
    _platform = get_platform()
    if _platform == WINDOWS:
        install_windows()
    if _platform == LINUX:
        install_linux()