#!/usr/bin/env python3
import shutil
from pathlib import Path
import urllib.request
import sys
import threading
from queue import Queue
import argparse

class MediaHelper:
    IMAGE_EXTENSIONS = ['.jpeg', '.jpg', '.png', '.gif', '.bmp', '.tiff']
    VIDEO_EXTENSIONS = ['.mov', '.mp4', '.avi', '.mkv']
    AUDIO_EXTENSIONS = ['.mp3', '.wav', '.ogg', '.flac']

    def __init__(self, base_dir='media'):
        self.base_dir = Path(base_dir)
        self.base_dir.mkdir(exist_ok=True)
        for folder in ['image', 'video', 'audio', 'other']:
            (self.base_dir / folder).mkdir(exist_ok=True)

    def detect_media_type(self, filepath):
        ext = Path(filepath).suffix.lower()
        if ext in self.IMAGE_EXTENSIONS:
            return 'image'
        elif ext in self.VIDEO_EXTENSIONS:
            return 'video'
        elif ext in self.AUDIO_EXTENSIONS:
            return 'audio'
        else:
            return 'other'

    def save_to_folder(self, filepath):
        media_type = self.detect_media_type(filepath)
        folder = self.base_dir / f"{media_type}s"
        dest = folder / Path(filepath).name
        shutil.copy(filepath, dest)
        return str(dest.resolve())

    def download_to_folder(self, url, filename=None):
        if not filename:
            filename = url.split('/')[-1]

        media_type = self.detect_media_type(filename)
        folder = self.base_dir / f"{media_type}"
        folder.mkdir(exist_ok=True)
        dest_path = folder / filename

        try:
            with urllib.request.urlopen(url) as response:
                total_size = response.getheader('Content-Length')
                total_size = int(total_size) if total_size else 0
                downloaded = 0
                chunk_size = 8192  # 8 KB

                with open(dest_path, 'wb') as out_file:
                    while True:
                        chunk = response.read(chunk_size)
                        if not chunk:
                            break
                        out_file.write(chunk)
                        downloaded += len(chunk)

                        # Progress calculation
                        downloaded_mb = downloaded / (1024 * 1024)
                        total_mb = total_size / (1024 * 1024) if total_size else 0
                        percent = min(100, downloaded * 100 / total_size) if total_size else 0
                        bar_len = 40
                        filled_len = int(bar_len * percent // 100)
                        bar = '=' * filled_len + '-' * (bar_len - filled_len)
                        print(f'\rDownloading {filename}: [{bar}] {percent:.2f}% ({downloaded_mb:.2f} MB / {total_mb:.2f} MB)', end='')

                print()  # newline after complete

        except Exception as e:
            print(f"\nFailed to download {url}: {e}")

        return str(dest_path.resolve())

    def download_multiple(self, urls, threads=10):
        """
        Download multiple URLs in parallel using threads.
        """
        queue = Queue()
        for url in urls:
            queue.put(url)

        def worker():
            while not queue.empty():
                url = queue.get()
                try:
                    self.download_to_folder(url)
                finally:
                    queue.task_done()

        thread_list = []
        for _ in range(min(threads, len(urls))):
            t = threading.Thread(target=worker)
            t.start()
            thread_list.append(t)

        queue.join()
        for t in thread_list:
            t.join()


def main():
    parser = argparse.ArgumentParser(description="Download media files by URL.")
    parser.add_argument('base_dir', help='Base directory to save media files')
    parser.add_argument('url', nargs='?', help='Single URL to download')
    parser.add_argument('--media-file', help='File containing URLs (one per line)')
    parser.add_argument('--threads', type=int, default=10, help='Number of parallel downloads (default 10)')
    args = parser.parse_args()

    helper = MediaHelper(args.base_dir)

    urls = []
    if args.url:
        urls.append(args.url)
    if args.media_file:
        with open(args.media_file, 'r') as f:
            file_urls = [line.strip() for line in f if line.strip()]
            urls.extend(file_urls)

    if not urls:
        print("No URLs provided. Use a URL argument or --media-file.")
        sys.exit(1)

    if len(urls) == 1:
        helper.download_to_folder(urls[0])
    else:
        helper.download_multiple(urls, threads=args.threads)


if __name__ == "__main__":
    main()
