import os
import logging
import threading
import platform
from pathlib import Path
from datetime import datetime
from concurrent_log_handler import ConcurrentRotatingFileHandler


def get_logger():
    console_log_level = logging.getLevelName(os.environ.get('CONSOLE_LOG_LEVEL', 'INFO').upper())
    file_log_level = logging.getLevelName(os.environ.get('FILE_LOG_LEVEL', 'INFO').upper())
    log_dir = os.environ.get('LOG_DIR', "/var/log/my_app")
    log_name = os.environ.get('LOG_NAME', 'my_app')
    Path(log_dir).mkdir(parents=True, exist_ok=True)
    sys_name = platform.system()
    if sys_name == "Windows":
        log_path = f"{log_name}.log"
    else:
        log_path = str(Path(log_dir) / f"{log_name}.log")
    file_log_format = (
        '%(asctime)s | %(name)s | %(levelname)s | thread-%(threadName)s | %(filename)s:%(lineno)s | %(message)s'
    )
    console_log_format = (
        '%(asctime)s | %(levelname)s | thread-%(threadName)s | %(filename)s:%(lineno)s | %(message)s'
    )
    file_handler = ConcurrentRotatingFileHandler(
        log_path,
        mode="a",
        maxBytes=1024 * 1024 * 1024,
        backupCount=10,
        encoding="utf-8"
    )
    file_handler.setFormatter(logging.Formatter(file_log_format))
    file_handler.setLevel(file_log_level)
    console_handler = logging.StreamHandler()
    console_handler.setLevel(level=console_log_level)
    console_handler.setFormatter(logging.Formatter(console_log_format))
    logger = logging.getLogger(log_name)

    if logger.handlers:
        logger.handlers.clear()

    logger.addHandler(console_handler)
    logger.addHandler(file_handler)
    logger.setLevel(logging.DEBUG)
    logger.propagate = False

    class ThreadNameFormatter(logging.Formatter):
        def format(self, record):
            if not hasattr(record, 'threadName') or not record.threadName:
                record.threadName = threading.current_thread().name or "unknown"
            return super().format(record)

    file_handler.setFormatter(ThreadNameFormatter(file_log_format))
    console_handler.setFormatter(ThreadNameFormatter(console_log_format))

    return logger