import os
import shutil
import json
import urllib3
import platform
import logging
import subprocess
from typing import List, Optional, Set, Union, Any
from git import Repo,GitCommandError

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

log=logging.getLogger(__name__)
log.setLevel(logging.WARNING)
formatter = logging.Formatter(
    fmt="%(asctime)s | %(levelname)s | %(name)s | %(filename)s:%(lineno)d | %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S"
)
console_handler = logging.StreamHandler()
console_handler.setFormatter(formatter)
log.addHandler(console_handler)


class ExecResult:
    def __init__(self,stdout,stderr,exit_code):
        log.info(f"exit_code:{exit_code},stdout: {stdout},stderr:{stderr}")
        self.stdout=stdout
        self.stderr=stderr
        self.exit_code=exit_code

class Server():
    def __init__(self,home="/home/my_home"):
        """
        初始化类实例，检查操作系统兼容性并创建指定的主目录

        参数:
            home: 主目录路径，默认为"/home/my_home"

        异常:
            RuntimeError: 当操作系统不支持或目录创建失败时抛出
        """
        # 检查操作系统平台
        self.__platform = platform.system()
        self.__validate_platform()

        # 初始化并验证主目录
        self.__home = self.__initialize_home_directory(home)

    def __initialize_home_directory(self, home: str) -> str:
        """
        初始化主目录，确保目录存在

        参数:
            home: 要创建的主目录路径

        返回:
            成功创建的主目录路径

        异常:
            RuntimeError: 当目录创建失败时抛出
        """
        try:
            # 优先使用Python内置方法创建目录（更跨平台且安全）
            if not self.__directory_exists(home):
                self.__create_directory(home)

            # 验证目录是否成功创建
            if not self.__directory_exists(home):
                raise RuntimeError(f"主目录创建后验证失败: {home}不存在")

            return home

        except Exception as e:
            error_msg = f"主目录创建失败: {str(e)}"
            log.error(error_msg)
            raise RuntimeError(error_msg) from e

    def __validate_platform(self) -> None:
        """验证操作系统是否为Linux，不支持则抛出异常"""
        if self.__platform == "Linux":
            pass
        else:
            raise RuntimeError(
                f"不支持的操作系统: {self.__platform}，仅支持Linux"
            )

    def __create_directory(self, path: str) -> None:
        """使用Python内置方法创建目录（含父目录）"""
        try:
            # 使用exist_ok=True避免重复创建错误
            os.makedirs(path, exist_ok=True)
        except PermissionError:
            raise RuntimeError(f"权限不足，无法创建目录: {path}")
        except OSError as e:
            raise RuntimeError(f"系统错误，创建目录失败: {path}, 错误: {str(e)}")

    def __directory_exists(self, path: str) -> bool:
        """检查目录是否存在且为有效目录"""
        return shutil.os.path.exists(path) and shutil.os.path.isdir(path)

    def exec_command(self, command: Union[str, List[str]]) -> ExecResult:
        log.info(f"执行命令: {command}")
        try:
            shell = isinstance(command, str)
            result = subprocess.run(
                command,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                shell=shell,
                encoding='utf-8',
                errors='replace'
            )
            returncode=result.returncode
            stderr=result.stderr
            stdout=result.stdout
            if returncode != 0:
                log.warning(f"命令执行失败: command : {command},exit code is {returncode}, stderr is {result.stderr}, stdout is {stdout}")
            return ExecResult(
                stdout=stdout,
                stderr=stderr,
                exit_code=returncode
            )

        except Exception as e:
            return ExecResult(
                stdout="",
                stderr=f"执行命令时发生异常: {str(e)}",
                exit_code=-1
            )

    def git_clone(self, repo_url: str, branch: str, dest_path: str) -> bool:
        log.info(f"开始克隆仓库: {repo_url} branch {branch} 到 {dest_path}")
        try:
            if os.path.exists(dest_path):
                git_dir = os.path.join(dest_path, '.git')
                if os.path.isdir(git_dir):
                    log.warning(f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},目标路径 '{dest_path}' 已存在且为Git仓库，无法克隆")
                    return False
                else:
                    log.warning(f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},目标路径 '{dest_path}' 已存在但不是Git仓库，无法克隆")
                    return False

            parent_dir = os.path.dirname(dest_path)
            if parent_dir and not os.path.exists(parent_dir):
                os.makedirs(parent_dir, exist_ok=True)

            repo = Repo.clone_from(
                url=repo_url,
                to_path=dest_path,
                branch=branch
            )

            if repo.bare:
                log.warning(f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},克隆失败，创建了裸仓库: {dest_path}")
                return False

            current_branch = repo.active_branch.name
            if current_branch != branch:
                log.warning(
                    f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},克隆成功，但分支不匹配",
                    f"实际分支: {current_branch}, 预期分支: {branch}"
                )
                return False
            log.info(f"克隆成功: {dest_path}")
            return True

        except GitCommandError as e:
            log.warning(f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},Git命令执行失败: {str(e)}")
            return False
        except Exception as e:
            log.warning(f"克隆仓库: {repo_url} branch {branch} 到 {dest_path},克隆过程中发生未知错误: {str(e)}")
            return False

    def find_all_file(
            self,
            root_dir: str,
            suffix: Optional[str] = None,
            with_suffix: bool = True,
            recursive: bool = True,
            exclude_dirs: Optional[List[str]] = None,
            exclude_files: Optional[List[str]] = None,
            absolute_path: bool = True,
            only_file_name: bool = False
    ) -> List[str]:
        log.info(f"开始查找文件: root_dir:{root_dir},suffix:{suffix},with_suffix:{with_suffix},recursive:{recursive},exclude_dirs:{exclude_dirs},exclude_files:{exclude_files},absolute_path:{absolute_path},only_file_name:{only_file_name}.")
        exclude_dirs_set: Set[str] = set(exclude_dirs) if exclude_dirs else set()
        exclude_files_set: Set[str] = set(exclude_files) if exclude_files else set()

        root_dir = os.path.abspath(root_dir)
        if not os.path.isdir(root_dir):
            log.warning(f"查找文件: root_dir:{root_dir},suffix:{suffix},with_suffix:{with_suffix},recursive:{recursive},exclude_dirs:{exclude_dirs},exclude_files:{exclude_files},absolute_path:{absolute_path},only_file_name:{only_file_name},根目录不存在或不是目录: {root_dir}")
            return None

        if suffix is not None and not suffix.startswith('.'):
            suffix = f'.{suffix}'

        result: List[str] = []
        seen: Set[str] = set()

        for dirpath, dirnames, filenames in os.walk(root_dir):
            dirnames[:] = [d for d in dirnames if d not in exclude_dirs_set]
            for filename in filenames:
                if filename in exclude_files_set:
                    log.warning(f"查找文件: root_dir:{root_dir},suffix:{suffix},with_suffix:{with_suffix},recursive:{recursive},exclude_dirs:{exclude_dirs},exclude_files:{exclude_files},absolute_path:{absolute_path},only_file_name:{only_file_name},文件被排除: {filename}")
                    continue
                if suffix is not None and not filename.endswith(suffix):
                    log.warning(f"查找文件: root_dir:{root_dir},suffix:{suffix},with_suffix:{with_suffix},recursive:{recursive},exclude_dirs:{exclude_dirs},exclude_files:{exclude_files},absolute_path:{absolute_path},only_file_name:{only_file_name},文件后缀不匹配: {filename}")
                    continue
                processed_name = filename if with_suffix else os.path.splitext(filename)[0]
                if only_file_name:
                    if processed_name not in seen:
                        seen.add(processed_name)
                        result.append(processed_name)
                else:
                    full_path = os.path.join(dirpath, filename)
                    if absolute_path:
                        display_path = os.path.abspath(full_path)
                    else:
                        display_path = os.path.relpath(full_path, root_dir)

                    if not with_suffix:
                        dir_part, file_part = os.path.split(display_path)
                        display_path = os.path.join(dir_part, os.path.splitext(file_part)[0])

                    if display_path not in seen:
                        seen.add(display_path)
                        result.append(display_path)

            # 非递归模式：只处理顶层目录
            if not recursive:
                break
        if not result:
            log.warning(f"未找到匹配的文件 - "
                     f"根目录: {root_dir}, 后缀: {suffix}, 递归: {recursive}, "
                     f"排除目录: {exclude_dirs}, 排除文件: {exclude_files}, "
                     f"绝对路径: {absolute_path}, 纯文件名: {only_file_name}")
        log.info(f"已找到文件 - {result}")
        return result

    def read_json(self, file_path: str) -> Optional[Any]:
        log.info(f"正在读取JSON文件 - {file_path}")
        if not os.path.exists(file_path):
            log.warning(f"读取JSON文件 - {file_path},错误: 文件不存在 - {file_path}")
            return None
        if not os.path.isfile(file_path):
            log.warning(f"读取JSON文件 - {file_path},错误: 不是有效的文件 - {file_path}")
            return None
        try:
            with open(file_path, 'r', encoding='utf-8') as file:
                try:
                    json_data = json.load(file)
                    log.info(f"成功读取JSON文件 - {file_path}, data is {json_data}")
                    return json_data
                except json.JSONDecodeError as e:
                    log.warning("读取JSON文件 - {file_path},错误: JSON解析失败 - {str(e)}，文件: {file_path}")
                    return None
                except Exception as e:
                    log.warning(f"读取JSON文件 - {file_path},错误: 读取文件内容时发生意外 - {str(e)}，文件: {file_path}")
                    return None
        except PermissionError:
            log.warning(f"读取JSON文件 - {file_path},错误: 没有权限打开文件 - {file_path}")
            return None
        except Exception as e:
            log.warning(f"读取JSON文件 - {file_path},错误: 打开文件时发生意外 - {str(e)}，文件: {file_path}")
            return None

    def get_abs_path(self, file_name: str, dest_dir: str) -> Optional[str]:
        log.info(f"正在获取绝对路径 - 文件名: {file_name}，目标目录: {dest_dir}")
        if not os.path.isdir(dest_dir):
            log.warning(f"获取绝对路径 - 文件名: {file_name}，目标目录: {dest_dir} 错误: 目标目录不存在或不是目录 - {dest_dir}")
            return None

        for root, _, files in os.walk(dest_dir):
            if file_name in files:
                abs_path=os.path.abspath(os.path.join(root, file_name))
                log.info(f"成功获取绝对路径 - 文件名: {file_name}，绝对路径: {abs_path}")
                return abs_path

        log.warning(f"获取绝对路径 - 文件名: {file_name}，目标目录: {dest_dir},未找到文件 - {file_name}，目录: {dest_dir}")
        return None