import numpy as np
import string
import inspect
import re
import typing
from typing import Literal
import sys
import math
import time

variable = "A"
global_label = "AA"

xor = "xor"

global_vars = {}
function_code = {}
file_lines = {} # all the lines in the child script
indentations = {} # the indentation (leading whitespace) of each line in the child script
pb_import_name = "" # whatever this module is referred to in the child script 
math_import_name = "" # whatver the math module is referred to in the child script


'''Some useful literals'''
COLORS = Literal["BLUE", "RED", "BLACK", "MAGENTA", "GREEN", "ORANGE", "NAVY", "LTBLUE",
                 "YELLOW", "WHITE", "LTGRAY", "MEDGRAY", "GRAY", "DARKGRAY"]

# just so i can index the colors
COLOR_LIST = ["BLUE", "RED", "BLACK", "MAGENTA", "GREEN", "ORANGE", "NAVY", "LTBLUE",
              "YELLOW", "WHITE", "LTGRAY", "MEDGRAY", "GRAY", "DARKGRAY"]

MARKS = Literal["▫", "•", "·"] #  "⁺" was giving me issues for some reason, sorry :(
TAIL_DIRECTIONS = Literal["LEFT", "CENTER", "RIGHT", "BLANK"] # select blank if your model doesn't support tail direction
ALTERNATIVES = Literal["NOT_EQUAL", "LESS", "GREATER"]

FIXED_MARKS = {
    "Â·" : "·",
    "â€¢" : "•",
    "â–«" : "▫",
}

L1 = "L₁"
L2 = "L₂"
L3 = "L₃"
L4 = "L₄"
L5 = "L₅"
L6 = "L₆"

X = "X"
Y = "Y"
Z = "Z"

i = ""
'''The imaginary number i''' # (it's there - use your imagination)


'''End of literals'''

# These two lines are used to help translate if/while statements
in_indented_block = False
statement_indentation = 0

ti_basic = ""

screen_width = 24 # usually 16 or 24 - how many characters can fit horizontally across the screen
labels = [] # a dictionary list with info on every label in the program
all_menus = []


class Menu():
    title = "" # what is displayed to the user
    options = [] # a list of MenuOption objects
    function = "" # name of function that stores the menu's code
    parameter = "" # the parameter that the function takes
    label = "" # the label we can run GoTo on to get back to this menu

    def __init__(self, title, function, options):
        self.title = title
        self.function = function
        self.parameter = re.search(r"^def " + function.__name__ + r"\s*\((.*)\):.*$", 
                                        str(inspect.getsource(self.function)).splitlines()[0]).groups()[0]
        self.options = options
        self.label = generateLabel()
        for option in self.options:
            option.setMenu(self)

            start_end = getStartEndLines(self, option)
            option.setStartLine(start_end[0])
            option.setEndLine(start_end[1])

        labels.append({
            "title": self.title,
            "label": self.label,
            "type": Menu
        })
        all_menus.append(self)

    def getTitle(self):
        return self.title
    
    def getOptions(self):
        return self.options
    
    def getFunction(self):
        return self.function

    def getLabel(self):
        return self.label

    def setLabel(self, label):
        self.label = label

    def getParameter(self):
        return self.parameter

class MenuOption():
    optionName = "" # name of option
    label = "" # Will be generated by the computer
    menu = Menu # Whatever menu this MenuOption is a part of

    start_line = 0 # whatever line in the menu's function this option's code block begins and ends on
    end_line = 0

    def __init__(self, optionName):
        self.optionName = optionName
        self.label = generateLabel()

        labels.append({
            "title": self.optionName,
            "label": self.label,
            "type": MenuOption
        })

    def getOptionName(self):
        return self.optionName
    
    def getLabel(self):
        return self.label
    
    def getMenu(self):
        return self.menu

    def setLabel(self, label):
        self.labelName = label
    
    def setMenu(self, menu):
        self.menu = menu

    def setStartLine(self, start_line):
        self.start_line = start_line

    def getStartLine(self):
        return self.start_line
    
    def setEndLine(self, end_line):
        self.end_line = end_line

    def getEndLine(self):
        return self.end_line
   
# Used for when a line has a PB function that needs to be translated into Basic
class NestedFunctionCall():
    content = "" # the stuff in parentheses
    start_index = 0
    level = 0 # how nested this function call is (0 is innermost, 1 is next step up, etc)

    def __init__(self, content, start_index, level):
        self.content = content
        self.start_index = start_index
        self.level = level
    
    def get_content(self):
        return self.content
    
    def get_start_index(self):
        return self.start_index
    
    def get_level(self):
        return self.level

# What is output from a line translation. Includes the old string/substring, the translation, and the starting index of the old string.
class TranslationObject():
    old_content = ""
    translation = ""
    old_content_start_index = 0

    def __init__(self, old_content, translation, old_content_start_index):
        self.old_content = old_content
        self.translation = translation
        self.old_content_start_index = old_content_start_index
        
    
    def get_old_content(self):
        return self.old_content
    
    def get_translation(self):
        return self.translation
    
    def get_old_content_start_index(self):
        return self.old_content_start_index
    
    

def generateLabel(): # goes from AB to ZZ, over 600 combinations 
    global global_label
    alphabet = list(string.ascii_uppercase)

    index = alphabet.index(global_label[1]) # Will return 0-25, for A-Z, respectively
    if index == 25: # the second letter in the label is Z, so we increment the first letter
        global_label = alphabet[alphabet.index(global_label[0]) + 1] + "A"
    else:
        global_label = global_label[0] + alphabet[alphabet.index(global_label[1]) + 1]
    
    return global_label

def reformat_equation(equation):
    variable = ""

    equation_divider = re.search(r"^(\w*)\s*\+=|^(\w*)\s*=|^(\w*)\s*-=|^(\w*)\s*\*=|^(\w*)\s*\/=", equation)
    equation = mathConvert(equation) # to replace "math." with calculator commands

    
    negative_sequence = re.findall(r"[-+=/*][\s]*-\d*", equation)
    negatives = []

    if(negative_sequence):
        for sequence in negative_sequence:
            #print(f"sequence {sequence}")
            negatives.append(int(str(sequence[1:]).strip()))

        characters = list(equation)
        for negative in negatives:
            print(f"negative! {negative}")
            i = equation.index(str(negative))
            characters[i] = "­"
        equation = "".join(characters)
    
    if(equation_divider):
        variable = [i for i in equation_divider.groups() if i is not None][0]
        divider = equation_divider.group().split(variable)[1].strip()
        print(f"var: {variable, divider}")
    else:
          # this isn't a valid equation, so we just return the line and use it as is
          return(equation)

    equation = equation.replace(" ", "") # remove spaces

    if(len(divider) == 2): # it's either += -= *= or /=
        equation = f"{variable.upper()}{divider[0]}{equation.split(variable + divider)[1]}→{variable.upper()}"
    else:
        equation = f"{equation.split(variable + divider)[1]}→{variable.upper()}"
    return equation

def mathConvert(line):
    fixed_line = str(line)
    fixed_line=fixed_line.replace("math.sqrt(","√(")
    fixed_line=fixed_line.replace("math.fabs(","abs(")
    fixed_line=fixed_line.replace("math.sin(","sin(")
    fixed_line=fixed_line.replace("math.cos(","cos(")
    fixed_line=fixed_line.replace("math.tan(","tan(")
    fixed_line=fixed_line.replace("math.asin(","asin(")
    fixed_line=fixed_line.replace("math.acos(","acos(")
    fixed_line=fixed_line.replace("math.atan(","atan(")
    fixed_line=fixed_line.replace("math.sinh(","sinh(")
    fixed_line=fixed_line.replace("math.cosh(","cosh(")
    fixed_line=fixed_line.replace("math.tanh(","tanh(")
    fixed_line=fixed_line.replace("math.asinh(","asinh(")
    fixed_line=fixed_line.replace("math.acosh(","acosh(")
    fixed_line=fixed_line.replace("math.atanh(","atanh(")
    fixed_line=fixed_line.replace("math.log(","ln(")
    fixed_line=fixed_line.replace("math.exp(","e^(")
    fixed_line=fixed_line.replace("math.floor(","int(")
    fixed_line=fixed_line.replace("math.log10(","log(")
    
    #same, but without "math." They might use
    #from math import sqrt etc...
    fixed_line=fixed_line.replace("sqrt(","[root]^2(")
    fixed_line=fixed_line.replace("fabs(","abs(")
    #(Redundant fixed_lines deleted)
    fixed_line=fixed_line.replace("log(","ln(")
    fixed_line=fixed_line.replace("exp(","e^(")
    fixed_line=fixed_line.replace("floor(","int(")
    fixed_line=fixed_line.replace("log10(","log(")

    if not (fixed_line == line):
        mathConvert(fixed_line)
    
    return fixed_line

def basicAppend(input):
    '''Appends the input to the existing TI-Basic code'''
    global ti_basic
    ti_basic += str(input) + "\n"

def literal_tibasic(input):
    '''
    Whatever is passed as input will be put into the final output file without any translation.\n
    This may be useful if you need an especially complex line that this module isn't capable of creating automatically.
    '''
    return input.strip('"').strip("'")

def getStartEndLines(menu, option): # automatically finds where each option's code starts and ends in its function
    function_lines = getFunctionCode(menu)
    parameter = menu.getParameter()
    option_name = option.getOptionName()
    start = 0
    end = 0

    for line in function_lines:
        regex_result = re.search("if.*" + str(parameter) + r"\s*==\s*\"" + str(option_name) + "\".*:", line)
        if(regex_result):
            start = function_lines.index(line) + 1
            break
    
    i = start
    while i < len(function_lines):
        if(i == len(function_lines) - 1):
            # this is the end of the function, so this is also the end of the option's code
            end = i
            break
        
        line = function_lines[i + 1]
        regex_result = re.search(r"if.*" + str(parameter) + r".*==.*\".*\".*:", line)
        if(regex_result):
            end = i
            break
        i += 1

    return [start, end]

def getFunctionCode(menu): # returns the code of a menu's function
    try:
        return function_code[menu.getFunction().__name__]
    except KeyError:
        function_code[menu.getFunction().__name__] = str(inspect.getsource(menu.getFunction())).splitlines()
        return function_code[menu.getFunction().__name__]

def getOptionCode(menu, option):
    function_lines = getFunctionCode(menu)
    option_lines = []

    i = option.getStartLine()
    while i <= option.getEndLine():
        option_lines.append(function_lines[i].strip())
        i += 1
    return option_lines

def fix_negatives(line):
    '''
    Gives negative numbers the negation symbol instead of just the - sign, preventing negation errors in the calculator
    '''
    line = str(line)

    # my best attempt at a regex that identifies all negative numbers in a line
    negative_search_regex = r"(?:[-+*=].*?(-\d+))|^\s*(-\d+)|\(\s*(-\d+)|,\s*(-\d+)"
    re_result = re.finditer(negative_search_regex, line)
    re_result_groups = re.findall(negative_search_regex, line)

    negatives = []
    indices = []

    for count, match in enumerate(re_result, 0):
        match_groups = re_result_groups[count]
        for group in match_groups:
            negatives.append(group)
            indices.append(match.group().rfind("-") + match.start())

    #print(f"{line}\n{negatives}\n{indices}\n")

    line_chars = list(line)
    for index in indices:
        line_chars[index] = "­"
    line = "".join(line_chars)

    return line

def string_insert(source_str, insert_str, pos):
    '''
    Inserts the "insert_str" at index "pos" in "source_str"
    '''
    return source_str[:pos] + insert_str + source_str[pos:]

def identify_line(line):
    '''
    Uses regex to look at each line and categorize it. The line is then translated differently depending on its type.
    '''

    global pb_import_name

    # Define the dictionary that will be returned with the line identification
    line_values = {
        "empty_line" : False,
        "pb_function" : False,
        "math_module" : False,
        "variable_set" : False,
        "if_for_while" : False
    }

    # 1. Check if the line is a comment (begins with #) or is blank
    # NOTE: If this is true, we skip over the rest of identification. Because what's the point
    if str(line).startswith("#") or not line: # "not line" checks for a blank line
        line_values["empty_line"] = True
        return line_values

    # 2. Check if the line calls a function from this module
    pb_function_regex = f"{pb_import_name}\."
    if(re.search(pb_function_regex, line)):
        line_values["pb_function"] = True
    
    # 3. Check if the line uses the math module
    math_regex = rf"{math_import_name}\."
    if(re.search(math_regex, line)):
        line_values["math_module"] = True

    # 4. Check if the line sets the value of a variable (i.e., x = 4)
    variable_set_regex = r"([\w\d]+)\s*(?!==)(?:(?:=|\+=|-=|\/=))\s*(.*)"
    if (re.search(variable_set_regex, line)):
        line_values["variable_set"] = True

    # 5. Check if the line initializes an if statement or a for/while loop
    if_for_while_regex = r"^(?:(?:if)|(?:while))"
    if(re.search(if_for_while_regex, line)):
        line_values["if_for_while"] = True

    return line_values

def get_arguments(function_call):
    '''
    Returns the arguments in the outmost function in the string. Nested functions' arguments are ignored.
    '''

    # First, regex is used to determine if the parentheses are empty/contains only whitespace. This means there are no arguments within them
    
    argument_search = rf"^{pb_import_name}\.[\w\d]+\(\s*\)" # returns a match if the outtermost function has NO arguments
    re_result = re.search(argument_search, function_call)
    if re_result:
        return []

    index = function_call.index("(") + 1 # so we start at where the arguments begin as opposed to the beginning of the line
    closing_left = 1 # goes up for each opening parentheses, down for each closing parentheses
    argument_count = 1 # at least one exists because the regex matched
    splitters = [] # where each comma that splits the arguments lies in the string
    splitters.append(index)

    #TODO: realize that parentheses in strings will eff this up
    while index < len(function_call):
        if function_call[index] == "," and closing_left == 1: # because commas separate arguments
            argument_count += 1
            splitters.append(index)
        if function_call[index] == "(":
            closing_left += 1
        if function_call[index] == ")":
            closing_left -= 1
        index += 1

    raw_args = [function_call[i:j] for i,j in zip(splitters, splitters[1:]+[None])] # dunno what the hell this does but thank you stackoverflow
    arguments = []
    for argument in raw_args:
        # clean everything up before returning
        
        if argument.endswith(")") and argument == raw_args[-1]:
            argument = argument[:len(argument) - 1]

        if argument.startswith(","):
            argument = argument.replace(",", "", 1)
        argument = argument.strip()
        arguments.append(argument)
    
    # if len(arguments) == 0:
    #     # This means there is one argument not separated by commas, so the above code didn't catch the argument
    #     # Return everything in the parentheses

    #     argument_grabber_regex = r"\((\s*(?=\S+)\S*\s*)\)"
    #     arguments.append(re_result.groups()[0])

    return arguments

def translate_pb_function(line):
    '''
    Translates the leftmost Python Basic function in the given line into its corresponding calculator command,
    then returns the line with the translation applied
    '''

    pb_function_identifier = f"{pb_import_name}\.(.*?)\("
    re_result = re.search(pb_function_identifier, line)
    if(re_result):
        # Identify the function name (Output, Prompt, normalCdf, etc)
        pb_function_name = re_result.groups()[0]

        start_index = line.index(re_result.group())
        splitters = []
        splitters.append(str(line).index("(", start_index))
        
        # While loop gets the substring containing this specific function within the line
        open_parentheses = 0
        index = start_index
        while index < len(line):
            if line[index] == "," and open_parentheses == 1: # because commas separate arguments
                splitters.append(index)
            if line[index] == "(":
                open_parentheses += 1
            if line[index] == ")":
                open_parentheses -= 1

                if open_parentheses == 0:
                    # We just encountered the closing parentheses that corresponds to the opening parentheses for this function
                    # So, we break the loop
                    break
            index += 1
        end_index = index + 1 # +1 so we actually include the final closing parentheses in the substring

        # NOTE: function_call is the substring within the line that has the specific PB function to translate.
        # Also note that the function_call substring could be the whole line, if the line is just a single PB command
        # Example: function_call = "pb.output(5,1,"Hey there!")"
        function_call = line[start_index:end_index]
        function_arguments = get_arguments(function_call)
        
        print(f"function call is {function_call}\narguments are {function_arguments}\nfunction name is {pb_function_name}")
        
        if len(function_arguments) > 1:
            function_arguments.insert(0, pb_function_name)
            translated_function_call = list_input(function_arguments)
            translated_line = line.replace(function_call, translated_function_call)
            return translated_line
        elif len(function_arguments) == 1:
            translated_function_call = globals()[pb_function_name](function_arguments[0])
            translated_line = line.replace(function_call, translated_function_call)
            return translated_line
        elif len(function_arguments) == 0:
            translated_function_call = globals()[pb_function_name]()
            translated_line = line.replace(function_call, translated_function_call)
            return translated_line
        
    else:
        # no PB functions were found, but this function was still called
        # this must mean a PB literal is being used, so it is translated here

        pb_literal_finder = rf"{pb_import_name}\.([\w\d]+)\s*(?=,|\)|$)"
        re_matches = re.finditer(pb_literal_finder, line)
        re_groups = re.findall(pb_literal_finder, line)

        for count, match in enumerate(re_matches, 0):
            line = line.replace(match.group(), globals()[re_groups[count]])

        return line

def translate_option_code(option):
    '''
    Accepts a menuOption passed from setup(), and returns a translated list of strings
    '''
    global in_indented_block, statement_indentation, indentations, file_lines

    translated_option_code = [] # the translated list of strings to be returned

    # 1. Get the lines that make up this specific menuOption object
    option_code = getOptionCode(option.getMenu(), option)
    
    # 2. This line finds the line number in the child script that the function of this specified option begins on
    function_start_line = list(file_lines.keys())[list(file_lines.values()).index(getFunctionCode(option.getMenu())[0])]
    
    current_index = 0

    while current_index < len(option_code):
        # We start on the first line of this option's code within its function, and current_index ensures we increment by one line each iteration
        master_index = function_start_line + option.getStartLine() + current_index
        
        # In simplier terms, master_index is the line number of this line in the child script

        current_line = file_lines[master_index]
        line_index = master_index
        line_indentation = indentations[master_index]

        #print(f"line: {current_line}, index: {line_index}")
        # Check if this line is part of an indented segment. If not, append the End statement
        if in_indented_block and line_indentation <= statement_indentation:
            if(current_line == "else:"):
                translated_option_code.append("Else")
                current_index += 1
                continue
            else:
                in_indented_block = False
                translated_option_code.append("End")

        translated_line = translateLine(current_line, line_index)
        translated_option_code.append(translated_line)
        
        if translated_line.startswith("If"):
            translated_option_code.append("Then")
        
        current_index += 1
    
    return translated_option_code

def remove_spaces(input):
    '''
    Returns a copy of the input string with spaces removed, except for spaces that are contained within strings
    '''
    input = str(input)

    new_string_chars = []
    index = 0
    in_quotes = False
    opening_quote = "" # either ' or "

    while index < len(input):
        char = input[index]

        if char == "'" or char == '"':
            if not in_quotes:
                opening_quote = char
                in_quotes = True
            else:
                if char == opening_quote:
                    in_quotes = False
        
        if char != " " or in_quotes:
            new_string_chars.append(char)
        
        index += 1

    return "".join(new_string_chars)

def translate_if_while(line):
    line = str(line)
    condition = "" # the entire substring after the if/while/for
    conditions = "" # condition split by and/or

    line = line.replace("!=", "≠")
    line = line.replace(">=", "≥")
    line = line.replace("<=", "≤")
    line = line.replace("==", "=")

    condition_grabber = r"(?:if|while|for)(?:\(|\s+)\(?(.*?)\s*\)?\s*:"
    re_result = re.search(condition_grabber, line)
    if(re_result):
        condition = re_result.groups()[0]

        and_or_regex = r"(.*?)(\s*(?:and|or|$)\s*)"
        conditions = re.findall(and_or_regex, condition)
        
        translated_condition = ""
        for match in conditions:
            for text in match:
                if not " and " in text and not " or " in text:
                    translated_condition += remove_spaces(text)
                else:
                    translated_condition += text

    if line.startswith("if"):
        return f"If {translated_condition}"
    if line.startswith("while"):
        return f"While {translated_condition}"

def translateLine(line, line_index): # actually not so bad anymore
    global pb_import_name, in_indented_block, statement_indentation, indentations

    # 1. Before translating, we call identify_line() to get an overview of the line
    line_info = identify_line(line)

    print(f"Attempting to translate: {line}")
    print(f"Line info: {line_info}")
    
    # 2. Look at each value in the dictionary and translate whatever is needed

        # 2(a): If the line is a comment / empty, an empty string is returned, as there is nothing to be done
    if line_info["empty_line"]:
        return ""
    
        # 2(b): If the line contains a PB function, translate_pb_function is called
    if line_info["pb_function"]:
        line = translate_pb_function(line)

        # 2(c): If the line uses the math module, we convert it into the corresponding calculator commmand
    if line_info["math_module"]:
        line = mathConvert(line)

        # 2(d): A variable is set (i.e., x = 4)
    if line_info["variable_set"]:
        line = reformat_equation(line)

        # 2(e): An if, for, or while statement is defined
    if line_info["if_for_while"]:
        # Declare that we're now in an indented code block
        in_indented_block = True
        statement_indentation = indentations[line_index]
        print(f"Line: {line}\tIndent: {statement_indentation}")

        return translate_if_while(line)
        

    # End of function checks
    done_translating = True
    for attr in line_info.values():
        if attr:
            done_translating = False
            break
    
    if done_translating:
        return line
    else:
        return translateLine(line, line_index) # recursion baby

def splitDisplayIntoLines(text):
    global screen_width

    splitters = ["=","+","-","/","*"] # list of symbols that are generally a good idea to split a line by
    text = str(text)

    letter = screen_width - 1 # we start by looking backwards from the end of the line for a good splitter
    while letter >= 0:
        if text[letter] in splitters: # we split the line here
            line1 = text[:letter + 1]
            line2 = text[letter + 1:]
            return [line1, line2]
        letter -= 1
    
    # if we get here, no valid splitter was found, so we split at the screen width
    line1 = text[:screen_width]
    line2 = text[screen_width:]
    return [line1, line2]

def disp(text): # turns a string into a Disp command, splitting the string into multiple lines as needed
    global tibasic
    text = str(text)
    output = ""
    result = "" # what will be returned
    screen_width = 24 # how many characters can be displayed in one line on the screen (will vary a bit between models but 24 for TI-84 Plus CE)

    if (len(text) > screen_width): # it won't fit into one line, so we have to break it up
        broken_string = text.split()

        line = ""
        for word in broken_string:
            if(len(word) > screen_width): # we must trim it
                trimmed = splitDisplayIntoLines(word)
                broken_string.insert(broken_string.index(word) + 1, trimmed[1])
                word = trimmed[0]

            if(((len(word) + len(line)) > screen_width) and len(line) > 0):
                output += line.strip() + "\n"
                line = word + " "
            else:
                line += word + " "
        output += line.strip()
    
        output_lines = output.split("\n")
        for output_line in output_lines:
            result += "Disp " + str(output_line) + "\n"
    else:
        # text is 24 characters or less, so it'll fit onto one line

        if ("\"" in text):
            # text is a string type
            result += "Disp " + str(text) + "\n"
        else:
            result += "Disp " + str(text).replace(" ", "").replace("+", ",") + "\n"
    result = result.rsplit("\n", 1)[0] # to remove line breaks at the end
    return result

def goToLabel(label):
    global global_vars

    if(".getLabel()" in label): # a literal string was not given
        label = global_vars[f"{label.split(".")[0]}"].getLabel()

    return "Goto " + label

def goToMenuTitle(menu_title):
    global global_vars, all_menus
    
    if(".getTitle()" in menu_title): # a literal string was not given
        menu_title = global_vars[f"{menu_title.split(".")[0]}"].getTitle()

    for menu in all_menus:
        if menu.getTitle() == menu_title:
            return ("Goto " + menu.getLabel())
    # if no match
    print("No label found for " + menu_title)

def goToMenu(menu):
    global global_vars, all_menus

    menu = global_vars[menu]

    for m in all_menus:
        if m == menu:
            try:
                return f"Goto {m.getLabel()}"
            except:
                print(f"No label found for {m}")

def clrHome():
    return "ClrHome"

def pause():
    return "Pause "

def list_input(input): # for when we need to make a call with globals() but we have multiple parameters
    for item in input:
        # loop through each item in list to fix formatting
        index = input.index(item)
        fixed_item = str(item).replace("'", "").strip()
        input[index] = fixed_item
    print("fixed list " + str(input))

    # first item in list is the TI operation to perform
    function_name = input[0]
    if(len(input) == 3):
        return globals()[function_name](input[1], input[2])
    if(len(input) == 4):
        return globals()[function_name](input[1], input[2], input[3])
    if(len(input) == 5):
        return globals()[function_name](input[1], input[2], input[3], input[4])
    if(len(input) == 6):
        return globals()[function_name](input[1], input[2], input[3], input[4], input[5])
    
def output(line, character, output):
    """Takes in a list input with 3 elements\n
    1st - Line number, 2nd - Character number, 3rd - text"""



    if(not "\"" in output):
         # we're not outputting a string, so we remove spaces to help prevent errors
         output = output.replace(" ", "")
    return f"Output({line},{character},{output})"

def Prompt(variable):
    return f"Prompt {variable.replace(" ", "").upper()}"

def For(variable, start, end, step=1):
    '''
    Executes some commands many times, with a variable increasing from start to end by step, with the default value step=1.\n
    NOTE: After you've written the code to be contained in the For loop, put an End() function after it! Otherwise your loop will never close.
    '''
    return f"For({variable},{start},{end},{step})"

def newLine():
    return "\n"

def variableSet(variable, expression):
    translated_line = translateLine(expression)
    #print("translation of " + str(expression) + " is " + str(translated_line))
    if(translated_line == ""):
        return str(expression) + "→" + str(variable)
    else:
        return str(translated_line) + "→" + str(variable)

def Stop():
    return "Stop"

def End():
    return "End"

def read_file_lines(filename):
  """
  Reads a Python file and returns a dictionary mapping line numbers to indentation levels.

  Args:
    filename: The path to the Python file.

  Returns:
    A dictionary mapping line numbers (starting from 1) to indentation levels (number of spaces).
  """
  global file_lines, indentations

  current_indent = 0

  with open(filename, 'r') as file:
    for line_number, line in enumerate(file, 1):
        # Strip the line and put it into the line dictionary
        file_lines[line_number] = line.strip()

        # Remove trailing whitespace
        stripped_line = line.rstrip()

        # Count leading spaces for indentation
        line_indent = 0
        for char in stripped_line:
            if char == "\t": # \t is a tab break
                line_indent += 4
                continue
            if char != ' ' and char != "\t":
                break
            line_indent += 1

        # Update indentation based on braces
        if line.endswith(':'):
            current_indent += line_indent
        elif line.startswith('}') and current_indent > 0:
            current_indent = max(0, current_indent - line_indent)

        indentations[line_number] = line_indent

        #print(f"{line_number}: ({line_indent}) {line.strip()}")

def setup(child_globals, filename): # Called from child script to begin the heavy lifting
    global global_vars, file_lines, pb_import_name, math_import_name, ti_basic, indentations

    # 1. Get all file info for later
    global_vars = child_globals
    read_file_lines(filename)

    # 2. Get the shorthand for the modules in the child script
    for line in file_lines.values():
        if "import pythonbasic" in line:
            if(" as " in line):
                pb_import_name = line.split("import pythonbasic as ")[1].strip()
                print(f"imported as {pb_import_name}")
            else:
                pb_import_name = "pythonbasic"
            
            if math_import_name:
                break
        if "import math" in line:
            if(" as " in line):
                math_import_name = line.split("import math as ")[1].strip()
            else:
                math_import_name = "math"
            
            if pb_import_name:
                break
    
    # 3. Go through each menu that the user created and translate it to Basic
    for menu in all_menus:
        menuOptions = menu.getOptions()

        basicAppend(f"Lbl {menu.getLabel()}")
        menu_syntax = f"Menu(\"{menu.getTitle().upper()}\"" # starting syntax for the menu option in Basic

        for option in menuOptions:
            menu_syntax += f",\"{option.getOptionName().upper()}\",{option.getLabel()}"
        
        basicAppend(menu_syntax) # writes the line in Basic that declares the menu

        for option in menuOptions:
            basicAppend(f"Lbl {option.getLabel()}")
            
            translated_option_code = translate_option_code(option)
            for line in translated_option_code:
                #translated_line = translateLine(line)
                #print(f"TRANSLATION: {line}")
                basicAppend(line)
            
            basicAppend(Stop())

    # 4. Check translated lines for negative numbers and fix them
    split_ti_basic = []
    for line in ti_basic.splitlines():
        split_ti_basic.append(fix_negatives(line))
    ti_basic = "\n".join(split_ti_basic)

    # 5. Write output to file
    filename = input("Enter the output file name. Omit .txt\n")
    filename += ".txt"
    file = open(filename, "w", encoding="utf-8")
    file.writelines(ti_basic)
    file.close()
    print("\nTI-Basic:\n" + ti_basic)

















# now for hell

def abs(value):
	'''Returns the absolute value of a real number, and the complex absolute value of a complex number.'''
	return f"abs({value})"

def angle(z):
	'''Returns the complex argument of a complex number.'''
	return f"angle({z})"

def AsmComp(original,result):
    '''Compresses an assembly program in hexadecimal form into binary form.'''
    return f"Asm(prgm{original.replace('"', "")},prgm{result.replace('"', "")})"

def binomcdf(trials,probability,value=-1):
    '''Calculates the binomial cumulative probability, either at a single value or for all values'''
    if value == -1:
        return f"binomcdf({trials},{probability})"
    else:
        return f"binomcdf({trials},{probability},{value})"

def binompdf(trials,probability,value=-1):
    '''Calculates the binomial probability, either at a single value or for all values'''
    if value == -1:
        return f"binompdf({trials},{probability})"
    else:
        return f"binompdf({trials},{probability},{value})"

def get_color_number(color: COLORS = "BLUE"):
    '''Returns the numeric value of the given color string. Ranges from 10-24.'''
    color = color.strip('"').strip("'")
    if color not in COLOR_LIST:
        print(f"Unaccepted color {color}")
    else:
        return str(10 + COLOR_LIST.index(color))

def PlotScatter(plot_number,x_list,y_list,mark: MARKS = "▫"):
    '''Plot#(Scatter, x-list, y-list, mark) defines a scatter plot. The points defined by x-list and y-list are plotted using mark on the graph screen. x-list and y-list must be the same length.'''
    mark = FIXED_MARKS[mark.strip("'").strip('"')]
    return f"Plot{plot_number}(Scatter,{x_list},{y_list},{mark})"

def PlotxyLine(plot_number,x_list,y_list,mark: MARKS = "▫"):
    '''Plot#(xyLine, x-list, y-list, mark) defines an xyLine plot. Similarly to a scatter plot, the points defined by x-list and y-list are plotted using mark on the graph screen, but with an xyLine plot they are also connected by a line, in the order that they occur in the lists. x-list and y-list must be the same length.'''
    mark = FIXED_MARKS[mark.strip("'").strip('"')]
    return f"Plot{plot_number}(xyLine,{x_list},{y_list},{mark})"

def PlotHistogram(plot_number,x_list,frequency_list):
    '''Plot#(Histogram, x-list, freq list) defines a Histogram plot. The x-axis is divided into intervals that are Xscl wide. A bar is drawn in in each interval whose height corresponds to the number of points in the interval. Points that are not between Xmin and Xmax are not tallied. Xscl must not be too small - it can divide the screen into no more than 47 different bars.'''
    return f"Plot{plot_number}(Histogram,{x_list},{frequency_list})"

def PlotBoxplot(plot_number,x_list,frequency_list):
    '''Plot#(Boxplot, x-list, freq list) defines a box plot. A rectangular box is drawn whose left edge is Q1 (the first quartile) of the data, and whose right edge is Q3 (the third quartile). A vertical segment is drawn within the box at the median, and 'whiskers' are drawn from the box to the minimum and maximum data points. The box plot ignores the Ymax and Ymin dimensions of the screen, and any plots that aren't box plots or modified box plots. Each box plot takes approximately 1/3 of the screen in height, and if more than one are plotted, they will take up different areas of the screen.'''
    return f"Plot{plot_number}(Boxplot,{x_list},{frequency_list})"

def PlotModBoxplot(plot_number,x_list,frequency_list,mark: MARKS = "▫"):
    '''Plot#(ModBoxplot, x-list, freq list, mark) defines a modified box plot. This is almost entirely like the normal box plot, except that it also draws outliers. Whiskers are only drawn to the furthers point within 1.5 times the interquartile range (Q3-Q1) of the box. Beyond this point, data points are drawn individually, using mark. The box plot ignores the Ymax and Ymin dimensions of the screen, and any plots that aren't box plots or modified box plots. Each box plot takes approximately 1/3 of the screen in height, and if more than one are plotted, they will take up different areas of the screen.'''
    mark = FIXED_MARKS[mark.strip("'").strip('"')]
    return f"Plot{plot_number}(ModBoxplot,{x_list},{frequency_list},{mark})"

def NormProbPlot(plot_number,data_list,data_axis,mark: MARKS = "▫"):
    '''Plot#(NormProbPlot, data list, data axis, mark) defines a normal probability plot. The mean and standard deviation of the data are calculated. Then for each point, the number of standard deviations it is from the mean is calculated, and the point is plotted against this number using mark. data axis can be either X or Y: it determines whether the value of a point determines it's x-coordinate or y-coordinate. The point behind this rather convoluted process is to test the extent to which the data is normally distributed. If it follows the normal distribution closely, then the result will be close to a straight line - otherwise it will be curved.'''
    mark = FIXED_MARKS[mark.strip("'").strip('"')]
    return f"Plot{plot_number}(NormProbPlot,{data_list},{data_axis},{mark})"

def checkTmr(Variable):
	'''Returns the number of seconds since the timer was started.'''
	return f"checkTmr({Variable})"

def Circle(X,Y,r):
	'''Draws a circle.'''
	return f"Circle({X},{Y},{r})"

def conj(value):
	'''Calculates the complex conjugate of a complex number.'''
	return f"conj({value})"

def cos(angle):
	'''Returns the cosine of a real number.'''
	return f"cos({angle})"

def cosh(value):
	'''Calculates the hyperbolic cosine of a value.'''
	return f"cosh({value})"

def cumSum(listormatrix):
	'''Calculates cumulative sums of a list or of the columns of a matrix.'''
	return f"cumSum({listormatrix})"

def dayOfWk(year,month,day):
	'''Returns an integer from 1 to 7, each representing a day of the week, given a date.'''
	return f"dayOfWk({year},{month},{day})"

def dbd(date1,date2):
	'''Calculates the number of days between two days.'''
	return f"dbd({date1},{date2})"

def det(matrix):
	'''Calculates the determinant of a square matrix.'''
	return f"det({matrix})"

def expr(string):
	'''Returns the value of a string that contains an expression.'''
	return f"expr({string})"

def Fill(value,matrix):
	'''Fills a list or matrix with one number.'''
	return f"Fill({value},{matrix})"

def fMax(function,var,min,max):
    '''fMax(f(var),var,lo,hi[,tol]) finds the value of var between lo and hi at which the maximum of f(var) occurs. tol controls the accuracy of the maximum value computed. The default value of tol is 10-5. fMax( only works for real numbers and expressions. Brent's method for optimization is used for approximating the maximum value.
    \nfMax(sin(X)cos(X),X,0,3)
    \n.7853995667'''
    return f"fMax({remove_spaces(function.strip('"').strip("'"))},{var},{min},{max})"

def fMin(function,var,min,max):
    '''fMin(f(var),var,lo,hi[,tol]) finds the value of var between lo and hi at which the minimum of f(var) occurs. tol controls the accuracy of the minimum value computed. The default value of tol is 10-5. fMin( only works for real numbers and expressions. Brent's method for optimization is used for approximating the minimum value.
    \nfMin(cos(sin(X)+Xcos(X)),X,0,2)
    \n1.076873875'''
    return f"fMin({remove_spaces(function.strip('"').strip("'"))},{var},{min},{max})"

def fnInt(function,var,a,b):
    '''fnInt(f(var),var,a,b[,tol]) computes an approximation to the definite integral of f with respect to var from a to b. tol controls the accuracy of the integral computed. The default value of tol is 10-5. fnInt( returns exact results for functions that are polynomials of small degree. fnInt( only works for real numbers and expressions. The Gauss-Kronrod method is used for approximating the integral. Tip: Sometimes, to get an answer of acceptable accuracy out of fnInt(, substitution of variables and analytic manipulation may be needed.'''
    return f"fnInt({remove_spaces(function.strip('"').strip("'"))},{var},{a},{b})"

def fPart(value):
	'''Returns the fractional part of a value.'''
	return f"fPart({value})"

def Fpdf(x,numeratordf,denominatordf):
	'''Evaluates the F-distribution probability density function at a point.'''
	return f"Fpdf({x},{numeratordf},{denominatordf})"

def gcd(value1,value2):
	'''Finds the greatest common divisor of two values.'''
	return f"gcd({value1},{value2})"

def geometcdf(probability,trials):
	'''Calculates the cumulative geometric probability for a single value'''
	return f"geometcdf({probability},{trials})"

def geometpdf(probability,trials):
	'''Calculates the geometric probability for a single value'''
	return f"geometpdf({probability},{trials})"

def Get(variable):
	'''Gets a variable's value from a connected calculator or CBL device.'''
	return f"Get({variable})"

def GetCalc(variable):
	'''Gets a variable from another calculator.'''
	return f"GetCalc({variable})"

def getDtStr(value):
	'''Returns the current date of the clock on the TI-84+/SE/CE as a string.'''
	return f"getDtStr({value})"

def getTmStr(value):
	'''Returns the current time of the clock on the TI-84+/SE as a string.'''
	return f"getTmStr({value})"

# def GraphStyle(equation#,style#):
# 	'''Sets the graphing style of a graphing equation in the current mode.'''
# 	return f"GraphStyle({equation#},{style#})"

def identity(n):
	'''Creates an n by n identity matrix.'''
	return f"identity({n})"

def imag(value):
	'''Returns the imaginary part of a complex number.'''
	return f"imag({value})"

def inString(haystack,needle,startingpoint):
	'''Finds the first occurrence of a search string in a larger string.'''
	return f"inString({haystack},{needle},{startingpoint})"

def Int(value):
    '''Rounds a value down to the nearest integer.'''
    return f"int({value})"

def invNorm(probability,mean=0,sd=1,tail_direction: TAIL_DIRECTIONS = "BLANK"):
    '''invNorm( is the inverse of the cumulative normal distribution function: given a probability, it will give you a z-score with that tail probability. The probability argument of invNorm( is between 0 and 1; 0 will give -1E99 instead of negative infinity, and 1 will give 1E99 instead of positive infinity.'''
    if tail_direction != "BLANK":
        return f"invNorm({probability},{mean},{sd},{tail_direction})"
    else:
        return f"invNorm({probability},{mean},{sd})"
    
def invT(probability,ν):
	'''Calculates the inverse of the cumulative Student's t-distribution function with degrees of freedom ν.'''
	return f"invT({probability},{ν})"

def iPart(value):
	'''Returns the integer part of a value.'''
	return f"iPart({value})"

def lcm(value1,value2):
	'''Finds the least common multiple of two values.'''
	return f"lcm({value1},{value2})"

def length(string):
	'''Returns the length of a string.'''
	return f"length({string})"

def Line(x1,y1,x2,y2):
    '''The Line( command is used to draw lines at any angle, as opposed to only drawing vertical or horizontal lines. Line(X1,Y1,X2,Y2) will draw a line from (X1,Y1) to (X2,Y2). Line( is affected by the window settings, although you can use a friendly window so there is no impact on the command.'''
    return f"Line({x1},{y1},{x2},{y2})"

def ln(value):
	'''Computes the (principal branch of the) natural logarithm.'''
	return f"ln({value})"

def log(value,base=-1):
    '''The log( command computes the base 10 logarithm of a value — the exponent to which 10 must be raised, to get that value. This makes it the inverse of the 10^( command. log( is a real number for all positive real values. For negative numbers, log( is an imaginary number (so taking log( of a negative number will cause ERR:NONREAL ANS to be thrown in Real mode), and of course it's a complex number for complex values. log( is not defined at 0, even if you're in a complex mode.'''
    if(base == -1):
        return f"log{value}"
    else:
        return f"log{value},{base}"

def max(value1,value2 = None):
    '''max(X,Y) returns the largest of the two numbers X and Y. max(list) returns the largest element of list. max(list1,list2) returns the pairwise maxima of the two lists. max(list1,X) (equivalently, max(X,list1)) returns a list whose elements are the larger of X or the corresponding element of the original list.'''
    if value2 is None:
        return f"max({remove_spaces(value1)})"
    else:
        return f"max({remove_spaces(value1)},{remove_spaces(value2)})"
    
def mean(list,frequency_list = None):
    '''The mean( command finds the mean, or the average, of a list. It's pretty elementary. It takes a list of real numbers as a parameter.'''
    if frequency_list is None:
        return f"mean({list})"
    else:
        return f"mean({list},{frequency_list})"
    
def median(list,frequency_list = None):
    '''The median( command finds the median of a list. It takes a list of real numbers as a parameter.'''
    if frequency_list is None:
        return f"median({list})"
    else:
        return f"median({list},{frequency_list})"
    
def min(value1,value2 = None):
    '''min(x,y) returns the smallest of the two numbers x and y. min(list) returns the smallest element of list. min(list1,list2) returns the pairwise minima of the two lists. min(list1,x) (equivalently, min(x,list1)) returns a list whose elements are the smaller of x or the corresponding element of the original list.'''
    if value2 is None:
        return f"min({remove_spaces(value1)})"
    else:
        return f"min({remove_spaces(value1)},{remove_spaces(value2)})"
    
def ncr(int1,int2):
    '''nCr is the number of combinations function (or binomial coefficient), defined as a nCr b = a!/(b!*(a-b)!), where a and b are nonnegative integers. The function also works on lists.'''
    return f"{int1} nCr {int2}"

def nDeriv(function,var,value,h=.001):
    '''nDeriv(f(var),var,value[,h]) computes an approximation to the value of the derivative of f(var) with respect to var at var=value. h is the step size used in the approximation of the derivative. The default value of h is 0.001.'''
    return f"nDeriv({remove_spaces(function)},{var},{value},{h})"

def normalcdf(lower,upper,mean=0,sd=1):
    '''normalcdf( is the normal (Gaussian) cumulative density function. If some random variable follows a normal distribution, you can use this command to find the probability that this variable will fall in the interval you supply. There are two ways to use normalcdf(. With two arguments (lower bound and upper bound), the calculator will assume you mean the standard normal distribution, and use that to find the probability corresponding to the interval between "lower bound" and "upper bound". You can also supply two additional arguments to use the normal distribution with a specified mean and standard deviation.'''
    return f"normalcdf({lower},{upper},{mean},{sd})"

def normalpdf(x,mean=0,sd=1):
    '''normalpdf( is the normal (Gaussian) probability density function. Since the normal distribution is continuous, the value of normalpdf( doesn't represent an actual probability - in fact, one of the only uses for this command is to draw a graph of the normal curve. You could also use it for various calculus purposes, such as finding inflection points. The command can be used in two ways: normalpdf(x) will evaluate the standard normal p.d.f. (with mean at 0 and a standard deviation of 1) at x, and normalpdf(x,μ,σ) will work for an arbitrary normal curve, with mean μ and standard deviation σ.'''
    return f"normalpdf({x},{mean},{sd})"

def Not(value):
    '''Flips the truth value of its argument.'''
    return f"not({value})"

def npr(int1,int2):
    '''nPr is the number of permutations function, defined as a nPr b = a!/(a-b)!, where a and b are nonnegative integers. The function also works on lists.'''
    return f"{int1} nPr {int2}"

def npv(interest_rate,cf0,cf_list,cf_freq=None):
    '''The npv( command computes the net present value of money over a specified time period. If a positive value is returned after executing npv(, that means it was a positive cashflow; otherwise it was a negative cashflow. The npv( command takes four arguments, and the fourth one is optional:\n\ninterest rate — the percentage of the money that is paid for the use of that money over each individual period of time.\n\nCF0 — the initial amount of money that you start out with; this number must be a real number, otherwise you will get a ERR:DATA TYPE error.\n\nCFList — the list of cash flows added or subtracted after the initial money.\n\nCFFreq — the list of frequencies of each cash flow added after the initial money; if this is left off, each cash flow in the cash flow list will just appear once by default.'''
    if cf_freq is None:
        return f"npv({interest_rate},{cf0},{cf_list})"
    else:
        return f"npv({interest_rate},{cf0},{cf_list},{cf_freq})"

def OnePropZInt(x,n,c_level):
    '''The 1-PropZInt( command calculates a confidence interval for a proportion, at a specific confidence level: for example, if the confidence level is 95%, you are 95% certain that the proportion lies within the interval you get. The command assumes that the sample is large enough that the normal approximation to binomial distributions is valid: this is true if, in the sample you take, the positive and negative counts are both >5. The 1-PropZInt( command takes 3 arguments. The first, x, is the positive count in the sample. The second, n, is the total size of the sample. (So the sample proportion is equal to x out of n). The third argument is the confidence level, which defaults to 95. The output gives you a confidence interval of the form (a,b), meaning that the true proportion π is most likely in the range a<π<b, and the value of x/n.'''
    return f"1-PropZInt({x},{n},{c_level})"

def OnePropZTest(null,x,n,prop: ALTERNATIVES = "NOT_EQUAL",draw=0):
    '''1-PropZTest performs an z-test to compare a population proportion to a hypothesis value. This test is valid for sufficiently large samples: only when the number of successes (x in the command syntax) and the number of failures (n-x) are both >5.\n\nThe logic behind the test is as follows: we want to test the hypothesis that the true proportion is equal to some value p0 (the null hypothesis). To do this, we assume that this "null hypothesis" is true, and calculate the probability that the (usually, somewhat different) actual proportion occurred, under this assumption. If this probability is sufficiently low (usually, 5% is the cutoff point), we conclude that since it's so unlikely that the data could have occurred under the null hypothesis, the null hypothesis must be false, and therefore the true proportion is not equal to p0. If, on the other hand, the probability is not too low, we conclude that the data may well have occurred under the null hypothesis, and therefore there's no reason to reject it.\n\nCommonly used notation has the letter π being used for the true population proportion (making the null hypothesis be π=p0). TI must have been afraid that this would be confused with the real number π, so on the calculator, "prop" is used everywhere instead.\n\nIn addition to the null hypothesis, we must have an alternative hypothesis as well - usually this is simply that the proportion is not equal to p0. However, in certain cases, our alternative hypothesis may be that the proportion is greater or less than p0.\n\n\nThe arguments to 1-PropZTest( are as follows:\n\np0 - the value for the null hypothesis (the proportion you're testing for)\n\nx - the success count in the sample\n\nn - the total size of the sample (so the sample proportion would be x/n)\n\nalternative (optional if you don't include draw?) - determines the alternative hypothesis\n\n0 (default value) - prop≠p0\n\n-1 (or any negative value) - prop<p0\n\n1 (or any positive value) - prop>p0\n\ndraw? (optional) set this to 1 if you want a graphical rather than numeric result'''

    alternative = 0
    if prop == "NOT_EQUAL": alternative = 0
    if prop == "LESS": alternative = -1
    if prop == "GREATER": alternative = 1

    return f"1-PropZTest({null},{x},{n},{alternative},{draw})"

def OpenLib(library):
	'''Sets up a compatible Flash application library for use with ExecLib'''
	return f"OpenLib({library})"

def Output(row,column,expression):
	'''Displays an expression on the home screen starting at a specified row and column. Wraps around if necessary.'''
	return f"Output({row},{column},{expression})"

def poissoncdf(mean,value):
	'''Calculates the Poisson cumulative probability for a single value'''
	return f"poissoncdf({mean},{value})"

def poissonpdf(mean,value):
	'''Calculates the Poisson probability for a single value'''
	return f"poissonpdf({mean},{value})"

def prod(list,start=-1,end=-1):
    '''The prod( command calculates the product of all or part of a list. When you use it with only one argument, the list, it multiplies all the elements of the list. You can also give it a bound of start and end and it will only multiply the elements starting and ending at those indices (inclusive).'''
    if start == -1 and end == -1:
        return f"prod({list})"
    elif start != -1 and end != -1:
        return f"prod({list},{start},{end})"
    elif start != -1 and end == -1:
        return f"prod({list},{start})"
    else:
        return f"prod({list})"

def PtChange(X,Y):
	'''Toggles a point on the graph screen.'''
	return f"Pt-Change({X},{Y})"

def Pt_Off(x,y):
    '''The Pt-Off( command is used to turn off a point (a pixel on the screen) on the graph screen at the given (X,Y) coordinates. Pt-Off( is affected by the window settings, which means you have to change the window settings accordingly, otherwise the point won't show up correctly on the screen.'''
    return f"Pt-Off({x},{y})"

def Pt_On(x,y):
    '''The Pt-On( command is used to draw a point on the graph screen at the given (X,Y) coordinates. Pt-On( is affected by the window settings Xmin, Xmax, Ymin, and Ymax. Make sure to change these accordingly when using it in a program, otherwise, you don't know where the point will show up.'''
    return f"Pt-On({x},{y})"

def PxlChange(row,column):
	'''Toggles a pixel on the graph screen.'''
	return f"Pxl-Change({row},{column})"

def PxlOff(row,column):
	'''Turns off a pixel on the graph screen.'''
	return f"Pxl-Off({row},{column})"

def PxlOn(row,column):
	'''Turns on a pixel on the graph screen.'''
	return f"Pxl-On({row},{column})"

def pxlTest(Y,X):
	'''Tests a pixel on the graph screen to see if it is on or off.'''
	return f"pxl-Test({Y},{X})"

def randBin(n,p,simulations):
	'''Generates a random number with the binomial distribution.'''
	return f"randBin({n},{p},{simulations})"

def randInt(lower,upper,amount=-1):
    '''randInt(min,max) generates a uniformly-distributed pseudorandom integer between min and max inclusive. randInt(min,max,n) generates a list of n uniformly-distributed pseudorandom integers between min and max.'''
    if amount == -1:
        return f"randInt({lower},{upper})"
    else:
        return f"randInt({lower},{upper},{amount})"

def randM(rows,columns):
	'''Creates a matrix of specified size with the entries random integers from -9 to 9.'''
	return f"randM({rows},{columns})"

def randNorm(mean,sd,amount):
    '''randNorm(µ,σ) generates a normally-distributed pseudorandom number with mean µ and standard deviation σ. The result returned will most probably be within the range µ±3σ. randNorm(µ,σ,n) generates a list of n normally-distributed pseudorandom numbers with mean µ and standard deviation σ.'''
    if amount == -1:
        return f"randInt({mean},{sd})"
    else:
        return f"randInt({mean},{sd},{amount})"

def ref(matrix):
	'''Puts a matrix into row-echelon form.'''
	return f"ref({matrix})"

def Repeat(condition):
    '''A Repeat loop executes a block of commands between the Repeat and End commands until the specified condition is true. The condition is tested at the end of the loop (when the End command is encountered), so the loop will always be executed at least once. This means that you sometimes don't have to declare or initialize the variables in the condition before the loop. After each time the Repeat loop is executed, the condition is checked to see if it is true. If it is true, then the loop is exited and program execution continues after the End command. If the condition is false, the loop is executed again.\n\n NOTE: After you've written all the lines to be contained in the repeat loop, make sure to end the loop with the End() command!'''
    return f"Repeat {remove_spaces(condition)}"

def rowSwap(matrix,row1,row2):
	'''Swaps two rows of a matrix.'''
	return f"rowSwap({matrix},{row1},{row2})"

def rref(matrix):
	'''Puts a matrix into reduced row-echelon form.'''
	return f"rref({matrix})"

def Select(xlistname,ylistname):
	'''Allows the user to select a subinterval of any enabled Scatter or xyLine plots.'''
	return f"Select({xlistname},{ylistname})"

def Send(variable):
	'''Sends data or a variable to a connected CBL device.'''
	return f"Send({variable})"

def setDate(year,month,day):
	'''Sets the date of the clock on the TI-84+/SE.'''
	return f"setDate({year},{month},{day})"

def setDtFmt(value):
	'''Sets the date format of the clock on the TI-84+/SE.'''
	return f"setDtFmt({value})"

def setTime(hour,minute,second):
	'''Sets the time of the clock on the TI-84+/SE.'''
	return f"setTime({hour},{minute},{second})"

def setTmFmt(value):
	'''Sets the time format of the clock on the TI-84+/SE.'''
	return f"setTmFmt({value})"

def ShadeF(lower,upper,numeratordf,denominatordf):
	'''Finds the probability of an interval of the <em>F</em>-distribution, and graphs the distribution with the interval's area shaded.'''
	return f"ShadeF({lower},{upper},{numeratordf},{denominatordf})"

def sin(angle):
	'''Returns the sine of a real number.'''
	return f"sin({angle})"

def sinh(value):
	'''Calculates the hyperbolic sine of a value.'''
	return f"sinh({value})"

def SortA(list):
    '''The SortA( command sorts a list in ascending order. It does not return it, but instead edits the original list variable (so it takes only list variables as arguments).'''
    return f"SortA({list})"

def SortD(list):
    '''The SortD( command sorts a list in descending order. It does not return it, but instead edits the original list variable (so it takes only list variables as arguments).'''
    return f"SortD({list})"

def stdDev(list,frequency_list = None):
    '''The stdDev( command finds the sample standard deviation of a list, a measure of the spread of a distribution. It takes a list of real numbers as a parameter.'''
    if(frequency_list is None):
        return f"stdDev({list})"
    else:
        return f"stdDev({list},{frequency_list})"
        
def sub(string,start,length):
	'''Returns a specific part of a given string, or divides by 100.'''
	return f"sub({string},{start},{length})"

def tan(angle):
	'''Returns the tangent of a real number.'''
	return f"tan({angle})"

def Tangent(expression,value):
	'''Draws a line tangent to an expression at the specified value.'''
	return f"Tangent({expression},{value})"

def tanh(value):
	'''Calculates the hyperbolic tangent of a value.'''
	return f"tanh({value})"

def tcdf(lower,upper,ν):
	'''Calculates the Student's t probability betwen lower and upper for degrees of freedom ν.'''
	return f"tcdf({lower},{upper},{ν})"

def timeCnv(value):
	'''Converts seconds into the equivalent days, hours, minutes, and seconds.'''
	return f"timeCnv({value})"

def tpdf(t,ν):
	'''Evaluates the Student's t probability density function with degrees of freedom ν.'''
	return f"tpdf({t},{ν})"
