import inspect
from typing import Any, Callable, Dict, List, Optional, Tuple
from hypothesis import strategies as st
from hypothesis.strategies._internal.core import SearchStrategy


def generate_input_values(func: Callable) -> Tuple:
    """
    This function takes an input function and returns a tuple of randomly generated
    values that could be passed as arguments to the input function.
    """

    # Get the parameter names and types of the function
    signature = inspect.signature(func)
    parameters = list(signature.parameters.values())

    # Create a dictionary to hold the generated values for each parameter
    generated_values: Dict[str, Any] = {}

    # Generate a value for each parameter
    for parameter in parameters:
        # If the parameter has a default value, use it
        if parameter.default is not inspect.Parameter.empty:
            generated_values[parameter.name] = parameter.default
        # Otherwise, generate a random value for the parameter type
        else:
            parameter_type = parameter.annotation
            if parameter_type is inspect.Parameter.empty:
                raise ValueError(f"Missing type annotation for parameter {parameter.name}")
            generated_values[parameter.name] = generate_random_value(parameter_type)

    # Return a tuple of the generated values
    if inspect.ismethod(func):
        # If the input function is a method, skip the first parameter (self)
        return tuple(generated_values.values())[1:]
    else:
        return tuple(generated_values.values())


def generate_random_value(parameter_type: Any) -> Any:
    """
    This function generates a random value for a given type using Hypothesis.
    """

    # Define the Hypothesis strategy for the given type
    if parameter_type is int:
        return st.integers().example()
    elif parameter_type is float:
        return st.floats().example()
    elif parameter_type is str:
        return st.text().example()
    elif parameter_type is bool:
        return st.booleans().example()
    elif parameter_type is List:
        inner_type = Any
        if hasattr(parameter_type, '__args__'):
            inner_type = parameter_type.__args__[0]
        return st.lists(elements=generate_random_value(inner_type)).example()
    elif parameter_type is Tuple:
        inner_types = Any, ...
        if hasattr(parameter_type, '__args__'):
            inner_types = parameter_type.__args__
        elements = [generate_random_value(inner_type) for inner_type in inner_types]
        return tuple(elements)
    elif parameter_type is Dict:
        key_type, value_type = Any, Any
        if hasattr(parameter_type, '__args__'):
            key_type, value_type = parameter_type.__args__
        keys = st.text().example()
        values = generate_random_value(value_type)
        return {keys: values}
    else:
        # For other types, use the Hypothesis strategy for generating values of any type
        if isinstance(parameter_type, type):
            return st.from_type(parameter_type).example()
        # for generics, such as Union or Optional, take one of the argument types
        elif hasattr(parameter_type, '__args__'):
            for inner_type in parameter_type.__args__:
                try:
                    return generate_random_value(inner_type)
                except ValueError:
                    pass
            raise ValueError(f"No matching type found for {parameter_type}")
        else:
            raise ValueError(f"Don't know how to generate random value for type {parameter_type}")


def test_function(func: Callable, n: int, tests: List[str]) -> Optional[Tuple[str, Tuple]]:
    """
    This function takes an input function, a number of iterations, and a list of tests,
    and executes the function n times with random inputs across all of the tests,
    replacing any values like ${1} with the corresponding entry of the tuple
    returned by generate_input_values. If any of the tests fail, the function returns
    the failing test and the tuple of arguments; otherwise, it returns None.
    """

    # Iterate over the tests n times, replacing any ${} placeholders with the
    # corresponding argument value generated by generate_input_values
    for _ in range(n):
        args = generate_input_values(func)
        for test in tests:
            for i, arg in enumerate(args):
                test = test.replace(f"${i+1}", str(arg))
            try:
                # Call the input function with the generated arguments
                func(*args)
            except Exception as e:
                print(e)
                # If the function call raises an exception, return the failing test
                # and the tuple of arguments that caused the exception
                return test, args

    # If all tests pass, return None
    return None

def add_numbers(x : int, y: int) -> int:
    print(f"X = {x}, Y = {y}\n")
    if x < 1000:
        return x + y
    else:
        return x + y + 1

print(test_function(add_numbers, 100, ["add_numbers(${1}, ${2}) == add_numbers(${2}, ${1})"]))
#print(generate_input_values(add_numbers))
