# -*- coding: utf-8 -*-

"""
Base model implementation for Gmail.

This module provides the foundational data models for the Gmail API client,
implementing common patterns for representing and interacting with Gmail.
The models follow three key design patterns:

1. **Raw Data Storage Pattern**:

All models store the original API response data in a `_data` attribute, treating the
API response schema as potentially unstable. Properties provide a stable interface
for accessing the underlying data, making the code more resilient to API changes.

2. **Property-Based Access Pattern**:

All attributes are exposed through properties rather than direct instance attributes.
This approach allows for lazy loading, data validation, and type conversion while
maintaining a clean public interface.

3. **Core Data Extraction Pattern**:

Each model implements a `core_data` property that returns a standardized, minimal
representation of the object. This provides a consistent way to access essential
information across different model types.

These models are designed to be instantiated by the API client methods, not directly
by users of the library. They provide a Pythonic interface to the JSON data returned
by the native boto3 AWS Lambda API.

This file is automatically generated from
``googleapiclient-stubs/_apis/gmail/v1/schemas.pyi``
"""

try:
    import typing_extensions as T
except ImportError:  # pragma: no cover
    import typing as T
import dataclasses
from functools import cached_property
from datetime import datetime, timezone

from func_args.api import REQ, BaseFrozenModel

from .type_hint import T_KWARGS, T_RESPONSE
from .utils import (
    extract_email_name,
    extract_email_address,
    create_email_deeplink,
    b64decode_with_auto_padding,
    html_to_text,
)
from .custom_model import Email

if T.TYPE_CHECKING:  # pragma: no cover
    from googleapiclient._apis.gmail.v1 import schemas


@dataclasses.dataclass(frozen=True)
class Base(BaseFrozenModel):
    _data: dict[str, T.Any] = dataclasses.field(default=REQ)

    @classmethod
    def new(cls, data: dict[str, T.Any]):
        return cls(_data=data)

    @property
    def core_data(self) -> T_KWARGS:
        raise NotImplementedError

    @property
    def full_data(self) -> T_RESPONSE:
        return self._data


T_BASE = T.TypeVar("T_BASE", bound=Base)


@dataclasses.dataclass(frozen=True)
class AutoForwarding(Base):
    _data: "schemas.AutoForwarding" = dataclasses.field(default=REQ)

    @cached_property
    def disposition(self):
        return self._data["disposition"]

    @cached_property
    def emailAddress(self):
        return self._data["emailAddress"]

    @cached_property
    def enabled(self):
        return self._data["enabled"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"emailAddress": self.emailAddress, "enabled": self.enabled}


@dataclasses.dataclass(frozen=True)
class BatchDeleteMessagesRequest(Base):
    _data: "schemas.BatchDeleteMessagesRequest" = dataclasses.field(default=REQ)

    @cached_property
    def ids(self):
        return self._data.get("ids", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"ids": self.ids}


@dataclasses.dataclass(frozen=True)
class BatchModifyMessagesRequest(Base):
    _data: "schemas.BatchModifyMessagesRequest" = dataclasses.field(default=REQ)

    @cached_property
    def addLabelIds(self):
        return self._data.get("addLabelIds", [])

    @cached_property
    def ids(self):
        return self._data.get("ids", [])

    @cached_property
    def removeLabelIds(self):
        return self._data.get("removeLabelIds", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"ids": self.ids}


@dataclasses.dataclass(frozen=True)
class CseIdentity(Base):
    _data: "schemas.CseIdentity" = dataclasses.field(default=REQ)

    @cached_property
    def emailAddress(self):
        return self._data["emailAddress"]

    @cached_property
    def primaryKeyPairId(self):
        return self._data["primaryKeyPairId"]

    @cached_property
    def signAndEncryptKeyPairs(self) -> "SignAndEncryptKeyPairs":
        return SignAndEncryptKeyPairs(_data=self._data["signAndEncryptKeyPairs"])

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "emailAddress": self.emailAddress,
            "primaryKeyPairId": self.primaryKeyPairId,
        }


@dataclasses.dataclass(frozen=True)
class CseKeyPair(Base):
    _data: "schemas.CseKeyPair" = dataclasses.field(default=REQ)

    @cached_property
    def disableTime(self):
        return self._data["disableTime"]

    @cached_property
    def enablementState(self):
        return self._data["enablementState"]

    @cached_property
    def keyPairId(self):
        return self._data["keyPairId"]

    @cached_property
    def pem(self):
        return self._data["pem"]

    @cached_property
    def pkcs7(self):
        return self._data["pkcs7"]

    @cached_property
    def privateKeyMetadata(self) -> list["CsePrivateKeyMetadata"]:
        return [
            CsePrivateKeyMetadata(_data=dct)
            for dct in self._data.get("privateKeyMetadata", [])
        ]

    @cached_property
    def subjectEmailAddresses(self):
        return self._data.get("subjectEmailAddresses", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"keyPairId": self.keyPairId, "enablementState": self.enablementState}


@dataclasses.dataclass(frozen=True)
class CsePrivateKeyMetadata(Base):
    _data: "schemas.CsePrivateKeyMetadata" = dataclasses.field(default=REQ)

    @cached_property
    def hardwareKeyMetadata(self) -> "HardwareKeyMetadata":
        return HardwareKeyMetadata(_data=self._data["hardwareKeyMetadata"])

    @cached_property
    def kaclsKeyMetadata(self) -> "KaclsKeyMetadata":
        return KaclsKeyMetadata(_data=self._data["kaclsKeyMetadata"])

    @cached_property
    def privateKeyMetadataId(self):
        return self._data["privateKeyMetadataId"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"privateKeyMetadataId": self.privateKeyMetadataId}


@dataclasses.dataclass(frozen=True)
class Delegate(Base):
    _data: "schemas.Delegate" = dataclasses.field(default=REQ)

    @cached_property
    def delegateEmail(self):
        return self._data["delegateEmail"]

    @cached_property
    def verificationStatus(self):
        return self._data["verificationStatus"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "delegateEmail": self.delegateEmail,
            "verificationStatus": self.verificationStatus,
        }


@dataclasses.dataclass(frozen=True)
class DisableCseKeyPairRequest(Base):
    _data: "schemas.DisableCseKeyPairRequest" = dataclasses.field(default=REQ)

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class Draft(Base):
    _data: "schemas.Draft" = dataclasses.field(default=REQ)

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def message(self) -> "Message":
        return Message(_data=self._data["message"])

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id}


@dataclasses.dataclass(frozen=True)
class EnableCseKeyPairRequest(Base):
    _data: "schemas.EnableCseKeyPairRequest" = dataclasses.field(default=REQ)

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class Filter(Base):
    _data: "schemas.Filter" = dataclasses.field(default=REQ)

    @cached_property
    def action(self) -> "FilterAction":
        return FilterAction(_data=self._data["action"])

    @cached_property
    def criteria(self) -> "FilterCriteria":
        return FilterCriteria(_data=self._data["criteria"])

    @cached_property
    def id(self):
        return self._data["id"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id}


@dataclasses.dataclass(frozen=True)
class FilterAction(Base):
    _data: "schemas.FilterAction" = dataclasses.field(default=REQ)

    @cached_property
    def addLabelIds(self):
        return self._data.get("addLabelIds", [])

    @cached_property
    def forward(self):
        return self._data["forward"]

    @cached_property
    def removeLabelIds(self):
        return self._data.get("removeLabelIds", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"addLabelIds": self.addLabelIds, "removeLabelIds": self.removeLabelIds}


@dataclasses.dataclass(frozen=True)
class FilterCriteria(Base):
    _data: "schemas.FilterCriteria" = dataclasses.field(default=REQ)

    @cached_property
    def excludeChats(self):
        return self._data["excludeChats"]

    @cached_property
    def from_(self):
        return self._data["from"]

    @cached_property
    def hasAttachment(self):
        return self._data["hasAttachment"]

    @cached_property
    def negatedQuery(self):
        return self._data["negatedQuery"]

    @cached_property
    def query(self):
        return self._data["query"]

    @cached_property
    def size(self):
        return self._data["size"]

    @cached_property
    def sizeComparison(self):
        return self._data["sizeComparison"]

    @cached_property
    def subject(self):
        return self._data["subject"]

    @cached_property
    def to(self):
        return self._data["to"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"query": self.query, "subject": self.subject}


@dataclasses.dataclass(frozen=True)
class ForwardingAddress(Base):
    _data: "schemas.ForwardingAddress" = dataclasses.field(default=REQ)

    @cached_property
    def forwardingEmail(self):
        return self._data["forwardingEmail"]

    @cached_property
    def verificationStatus(self):
        return self._data["verificationStatus"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "forwardingEmail": self.forwardingEmail,
            "verificationStatus": self.verificationStatus,
        }


@dataclasses.dataclass(frozen=True)
class HardwareKeyMetadata(Base):
    _data: "schemas.HardwareKeyMetadata" = dataclasses.field(default=REQ)

    @cached_property
    def description(self):
        return self._data["description"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"description": self.description}


@dataclasses.dataclass(frozen=True)
class History(Base):
    _data: "schemas.History" = dataclasses.field(default=REQ)

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def labelsAdded(self) -> list["HistoryLabelAdded"]:
        return [
            HistoryLabelAdded(_data=dct) for dct in self._data.get("labelsAdded", [])
        ]

    @cached_property
    def labelsRemoved(self) -> list["HistoryLabelRemoved"]:
        return [
            HistoryLabelRemoved(_data=dct)
            for dct in self._data.get("labelsRemoved", [])
        ]

    @cached_property
    def messages(self) -> list["Message"]:
        return [Message(_data=dct) for dct in self._data.get("messages", [])]

    @cached_property
    def messagesAdded(self) -> list["HistoryMessageAdded"]:
        return [
            HistoryMessageAdded(_data=dct)
            for dct in self._data.get("messagesAdded", [])
        ]

    @cached_property
    def messagesDeleted(self) -> list["HistoryMessageDeleted"]:
        return [
            HistoryMessageDeleted(_data=dct)
            for dct in self._data.get("messagesDeleted", [])
        ]

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id}


@dataclasses.dataclass(frozen=True)
class HistoryLabelAdded(Base):
    _data: "schemas.HistoryLabelAdded" = dataclasses.field(default=REQ)

    @cached_property
    def labelIds(self):
        return self._data.get("labelIds", [])

    @cached_property
    def message(self) -> "Message":
        return Message(_data=self._data["message"])

    @property
    def core_data(self) -> T_KWARGS:
        return {"labelIds": self.labelIds}


@dataclasses.dataclass(frozen=True)
class HistoryLabelRemoved(Base):
    _data: "schemas.HistoryLabelRemoved" = dataclasses.field(default=REQ)

    @cached_property
    def labelIds(self):
        return self._data.get("labelIds", [])

    @cached_property
    def message(self) -> "Message":
        return Message(_data=self._data["message"])

    @property
    def core_data(self) -> T_KWARGS:
        return {"labelIds": self.labelIds}


@dataclasses.dataclass(frozen=True)
class HistoryMessageAdded(Base):
    _data: "schemas.HistoryMessageAdded" = dataclasses.field(default=REQ)

    @cached_property
    def message(self) -> "Message":
        return Message(_data=self._data["message"])

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class HistoryMessageDeleted(Base):
    _data: "schemas.HistoryMessageDeleted" = dataclasses.field(default=REQ)

    @cached_property
    def message(self) -> "Message":
        return Message(_data=self._data["message"])

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ImapSettings(Base):
    _data: "schemas.ImapSettings" = dataclasses.field(default=REQ)

    @cached_property
    def autoExpunge(self):
        return self._data["autoExpunge"]

    @cached_property
    def enabled(self):
        return self._data["enabled"]

    @cached_property
    def expungeBehavior(self):
        return self._data["expungeBehavior"]

    @cached_property
    def maxFolderSize(self):
        return self._data["maxFolderSize"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"enabled": self.enabled, "expungeBehavior": self.expungeBehavior}


@dataclasses.dataclass(frozen=True)
class KaclsKeyMetadata(Base):
    _data: "schemas.KaclsKeyMetadata" = dataclasses.field(default=REQ)

    @cached_property
    def kaclsData(self):
        return self._data["kaclsData"]

    @cached_property
    def kaclsUri(self):
        return self._data["kaclsUri"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"kaclsUri": self.kaclsUri}


@dataclasses.dataclass(frozen=True)
class Label(Base):
    _data: "schemas.Label" = dataclasses.field(default=REQ)

    @cached_property
    def color(self) -> "LabelColor":
        return LabelColor(_data=self._data["color"])

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def labelListVisibility(self):
        return self._data["labelListVisibility"]

    @cached_property
    def messageListVisibility(self):
        return self._data["messageListVisibility"]

    @cached_property
    def messagesTotal(self):
        return self._data["messagesTotal"]

    @cached_property
    def messagesUnread(self):
        return self._data["messagesUnread"]

    @cached_property
    def name(self):
        return self._data["name"]

    @cached_property
    def threadsTotal(self):
        return self._data["threadsTotal"]

    @cached_property
    def threadsUnread(self):
        return self._data["threadsUnread"]

    @cached_property
    def type(self):
        return self._data["type"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id, "name": self.name, "type": self.type}


@dataclasses.dataclass(frozen=True)
class LabelColor(Base):
    _data: "schemas.LabelColor" = dataclasses.field(default=REQ)

    @cached_property
    def backgroundColor(self):
        return self._data["backgroundColor"]

    @cached_property
    def textColor(self):
        return self._data["textColor"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"backgroundColor": self.backgroundColor, "textColor": self.textColor}


@dataclasses.dataclass(frozen=True)
class LanguageSettings(Base):
    _data: "schemas.LanguageSettings" = dataclasses.field(default=REQ)

    @cached_property
    def displayLanguage(self):
        return self._data["displayLanguage"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"displayLanguage": self.displayLanguage}


@dataclasses.dataclass(frozen=True)
class ListCseIdentitiesResponse(Base):
    _data: "schemas.ListCseIdentitiesResponse" = dataclasses.field(default=REQ)

    @cached_property
    def cseIdentities(self) -> list["CseIdentity"]:
        return [CseIdentity(_data=dct) for dct in self._data.get("cseIdentities", [])]

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"nextPageToken": self.nextPageToken}


@dataclasses.dataclass(frozen=True)
class ListCseKeyPairsResponse(Base):
    _data: "schemas.ListCseKeyPairsResponse" = dataclasses.field(default=REQ)

    @cached_property
    def cseKeyPairs(self) -> list["CseKeyPair"]:
        return [CseKeyPair(_data=dct) for dct in self._data.get("cseKeyPairs", [])]

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"nextPageToken": self.nextPageToken}


@dataclasses.dataclass(frozen=True)
class ListDelegatesResponse(Base):
    _data: "schemas.ListDelegatesResponse" = dataclasses.field(default=REQ)

    @cached_property
    def delegates(self) -> list["Delegate"]:
        return [Delegate(_data=dct) for dct in self._data.get("delegates", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListDraftsResponse(Base):
    _data: "schemas.ListDraftsResponse" = dataclasses.field(default=REQ)

    @cached_property
    def drafts(self) -> list["Draft"]:
        return [Draft(_data=dct) for dct in self._data.get("drafts", [])]

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @cached_property
    def resultSizeEstimate(self):
        return self._data["resultSizeEstimate"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "nextPageToken": self.nextPageToken,
            "resultSizeEstimate": self.resultSizeEstimate,
        }


@dataclasses.dataclass(frozen=True)
class ListFiltersResponse(Base):
    _data: "schemas.ListFiltersResponse" = dataclasses.field(default=REQ)

    @cached_property
    def filter(self) -> list["Filter"]:
        return [Filter(_data=dct) for dct in self._data.get("filter", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListForwardingAddressesResponse(Base):
    _data: "schemas.ListForwardingAddressesResponse" = dataclasses.field(default=REQ)

    @cached_property
    def forwardingAddresses(self) -> list["ForwardingAddress"]:
        return [
            ForwardingAddress(_data=dct)
            for dct in self._data.get("forwardingAddresses", [])
        ]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListHistoryResponse(Base):
    _data: "schemas.ListHistoryResponse" = dataclasses.field(default=REQ)

    @cached_property
    def history(self) -> list["History"]:
        return [History(_data=dct) for dct in self._data.get("history", [])]

    @cached_property
    def historyId(self):
        return self._data["historyId"]

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"historyId": self.historyId, "nextPageToken": self.nextPageToken}


@dataclasses.dataclass(frozen=True)
class ListLabelsResponse(Base):
    _data: "schemas.ListLabelsResponse" = dataclasses.field(default=REQ)

    @cached_property
    def labels(self) -> list["Label"]:
        return [Label(_data=dct) for dct in self._data.get("labels", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListMessagesResponse(Base):
    _data: "schemas.ListMessagesResponse" = dataclasses.field(default=REQ)

    @cached_property
    def messages(self) -> list["Message"]:
        return [Message(_data=dct) for dct in self._data.get("messages", [])]

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @cached_property
    def resultSizeEstimate(self):
        return self._data["resultSizeEstimate"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "nextPageToken": self.nextPageToken,
            "resultSizeEstimate": self.resultSizeEstimate,
        }


@dataclasses.dataclass(frozen=True)
class ListSendAsResponse(Base):
    _data: "schemas.ListSendAsResponse" = dataclasses.field(default=REQ)

    @cached_property
    def sendAs(self) -> list["SendAs"]:
        return [SendAs(_data=dct) for dct in self._data.get("sendAs", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListSmimeInfoResponse(Base):
    _data: "schemas.ListSmimeInfoResponse" = dataclasses.field(default=REQ)

    @cached_property
    def smimeInfo(self) -> list["SmimeInfo"]:
        return [SmimeInfo(_data=dct) for dct in self._data.get("smimeInfo", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class ListThreadsResponse(Base):
    _data: "schemas.ListThreadsResponse" = dataclasses.field(default=REQ)

    @cached_property
    def nextPageToken(self):
        return self._data["nextPageToken"]

    @cached_property
    def resultSizeEstimate(self):
        return self._data["resultSizeEstimate"]

    @cached_property
    def threads(self) -> list["Thread"]:
        return [Thread(_data=dct) for dct in self._data.get("threads", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "nextPageToken": self.nextPageToken,
            "resultSizeEstimate": self.resultSizeEstimate,
        }


@dataclasses.dataclass(frozen=True)
class Message(Base):
    _data: "schemas.Message" = dataclasses.field(default=REQ)

    @cached_property
    def historyId(self):
        return self._data["historyId"]

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def internalDate(self):
        return self._data["internalDate"]

    @cached_property
    def labelIds(self):
        return self._data.get("labelIds", [])

    @cached_property
    def payload(self) -> "MessagePart":
        return MessagePart(_data=self._data["payload"])

    @cached_property
    def raw(self):
        return self._data["raw"]

    @cached_property
    def sizeEstimate(self):
        return self._data["sizeEstimate"]

    @cached_property
    def snippet(self):
        return self._data["snippet"]

    @cached_property
    def threadId(self):
        return self._data["threadId"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id, "threadId": self.threadId, "snippet": self.snippet}

    @cached_property
    def internal_date_datetime(self) -> datetime:
        return datetime.fromtimestamp(int(self.internalDate) / 1000, tz=timezone.utc)

    def get_deeplink(self, acc: str | None = None) -> str:
        return create_email_deeplink(
            thread_id_or_message_id=self.id,
            acc=acc,
        )


@dataclasses.dataclass(frozen=True)
class MessagePart(Base):
    _data: "schemas.MessagePart" = dataclasses.field(default=REQ)

    @cached_property
    def body(self) -> "MessagePartBody":
        return MessagePartBody(_data=self._data["body"])

    @cached_property
    def filename(self):
        return self._data["filename"]

    @cached_property
    def headers(self) -> list["MessagePartHeader"]:
        return [MessagePartHeader(_data=dct) for dct in self._data.get("headers", [])]

    @cached_property
    def mimeType(self):
        return self._data["mimeType"]

    @cached_property
    def partId(self):
        return self._data["partId"]

    @cached_property
    def parts(self) -> list["MessagePart"]:
        return [MessagePart(_data=dct) for dct in self._data.get("parts", [])]

    @property
    def core_data(self) -> T_KWARGS:
        return {"partId": self.partId, "mimeType": self.mimeType}

    @cached_property
    def headers_mapping(self) -> dict[str, "MessagePartHeader"]:
        return {header.name: header for header in self.headers}

    @cached_property
    def from_(self) -> "MessagePartHeader":
        return self.headers_mapping["From"]

    @cached_property
    def from_email(self) -> "Email":
        """
        Extracts and returns an Email object from the From field of the email
        """
        return Email(
            name=extract_email_name(self.from_.value),
            address=extract_email_address(self.from_.value),
        )

    @cached_property
    def to(self) -> "MessagePartHeader":
        return self.headers_mapping["To"]

    @cached_property
    def to_email(self) -> "Email":
        """
        Extracts and returns an Email object from the To field of the email
        """
        return Email(
            name=extract_email_name(self.to.value),
            address=extract_email_address(self.to.value),
        )

    @cached_property
    def cc(self) -> T.Optional["MessagePartHeader"]:
        return self.headers_mapping.get("Cc")

    @cached_property
    def cc_emails(self) -> list["Email"]:
        """
        Extracts and returns a list of Email objects from the CC field of the email
        """
        return [
            Email(
                name=extract_email_name(text.strip()),
                address=extract_email_address(text.strip()),
            )
            for text in self.cc.value.split(",")
        ]

    @cached_property
    def subject_text(self) -> str:
        """
        Extracts and returns the subject of the email message.
        """
        return self.headers_mapping["Subject"].value

    @cached_property
    def sent_on_datetime(self) -> datetime:
        """
        Extracts and returns the sent date and time of the email message as a
        """
        return datetime.fromisoformat(self.headers_mapping["sent-on"].value)

    @cached_property
    def text_body(self) -> str:
        """
        Extracts and returns the text body of the email message.

        .. note::

            This implementation may not right, we may need to inspect the
            payload.mimeType first and then decide how to extract the text body
            from tree of parts.
        """
        part = self.parts[-1]
        if part.mimeType == "text/plain":
            return b64decode_with_auto_padding(part.body.data)
        elif part.mimeType == "text/html":
            return html_to_text(b64decode_with_auto_padding(part.body.data))
        else:
            raise ValueError(f"Unknown mimeType: {part.mimeType}")


@dataclasses.dataclass(frozen=True)
class MessagePartBody(Base):
    _data: "schemas.MessagePartBody" = dataclasses.field(default=REQ)

    @cached_property
    def attachmentId(self):
        return self._data["attachmentId"]

    @cached_property
    def data(self):
        return self._data["data"]

    @cached_property
    def size(self):
        return self._data["size"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"size": self.size}


@dataclasses.dataclass(frozen=True)
class MessagePartHeader(Base):
    _data: "schemas.MessagePartHeader" = dataclasses.field(default=REQ)

    @cached_property
    def name(self):
        return self._data["name"]

    @cached_property
    def value(self):
        return self._data["value"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"name": self.name, "value": self.value}


@dataclasses.dataclass(frozen=True)
class ModifyMessageRequest(Base):
    _data: "schemas.ModifyMessageRequest" = dataclasses.field(default=REQ)

    @cached_property
    def addLabelIds(self):
        return self._data.get("addLabelIds", [])

    @cached_property
    def removeLabelIds(self):
        return self._data.get("removeLabelIds", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"addLabelIds": self.addLabelIds, "removeLabelIds": self.removeLabelIds}


@dataclasses.dataclass(frozen=True)
class ModifyThreadRequest(Base):
    _data: "schemas.ModifyThreadRequest" = dataclasses.field(default=REQ)

    @cached_property
    def addLabelIds(self):
        return self._data.get("addLabelIds", [])

    @cached_property
    def removeLabelIds(self):
        return self._data.get("removeLabelIds", [])

    @property
    def core_data(self) -> T_KWARGS:
        return {"addLabelIds": self.addLabelIds, "removeLabelIds": self.removeLabelIds}


@dataclasses.dataclass(frozen=True)
class ObliterateCseKeyPairRequest(Base):
    _data: "schemas.ObliterateCseKeyPairRequest" = dataclasses.field(default=REQ)

    @property
    def core_data(self) -> T_KWARGS:
        return {}


@dataclasses.dataclass(frozen=True)
class PopSettings(Base):
    _data: "schemas.PopSettings" = dataclasses.field(default=REQ)

    @cached_property
    def accessWindow(self):
        return self._data["accessWindow"]

    @cached_property
    def disposition(self):
        return self._data["disposition"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"accessWindow": self.accessWindow, "disposition": self.disposition}


@dataclasses.dataclass(frozen=True)
class Profile(Base):
    _data: "schemas.Profile" = dataclasses.field(default=REQ)

    @cached_property
    def emailAddress(self):
        return self._data["emailAddress"]

    @cached_property
    def historyId(self):
        return self._data["historyId"]

    @cached_property
    def messagesTotal(self):
        return self._data["messagesTotal"]

    @cached_property
    def threadsTotal(self):
        return self._data["threadsTotal"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "emailAddress": self.emailAddress,
            "messagesTotal": self.messagesTotal,
            "threadsTotal": self.threadsTotal,
        }


@dataclasses.dataclass(frozen=True)
class SendAs(Base):
    _data: "schemas.SendAs" = dataclasses.field(default=REQ)

    @cached_property
    def displayName(self):
        return self._data["displayName"]

    @cached_property
    def isDefault(self):
        return self._data["isDefault"]

    @cached_property
    def isPrimary(self):
        return self._data["isPrimary"]

    @cached_property
    def replyToAddress(self):
        return self._data["replyToAddress"]

    @cached_property
    def sendAsEmail(self):
        return self._data["sendAsEmail"]

    @cached_property
    def signature(self):
        return self._data["signature"]

    @cached_property
    def smtpMsa(self) -> "SmtpMsa":
        return SmtpMsa(_data=self._data["smtpMsa"])

    @cached_property
    def treatAsAlias(self):
        return self._data["treatAsAlias"]

    @cached_property
    def verificationStatus(self):
        return self._data["verificationStatus"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "sendAsEmail": self.sendAsEmail,
            "isDefault": self.isDefault,
            "verificationStatus": self.verificationStatus,
        }


@dataclasses.dataclass(frozen=True)
class SignAndEncryptKeyPairs(Base):
    _data: "schemas.SignAndEncryptKeyPairs" = dataclasses.field(default=REQ)

    @cached_property
    def encryptionKeyPairId(self):
        return self._data["encryptionKeyPairId"]

    @cached_property
    def signingKeyPairId(self):
        return self._data["signingKeyPairId"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "encryptionKeyPairId": self.encryptionKeyPairId,
            "signingKeyPairId": self.signingKeyPairId,
        }


@dataclasses.dataclass(frozen=True)
class SmimeInfo(Base):
    _data: "schemas.SmimeInfo" = dataclasses.field(default=REQ)

    @cached_property
    def encryptedKeyPassword(self):
        return self._data["encryptedKeyPassword"]

    @cached_property
    def expiration(self):
        return self._data["expiration"]

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def isDefault(self):
        return self._data["isDefault"]

    @cached_property
    def issuerCn(self):
        return self._data["issuerCn"]

    @cached_property
    def pem(self):
        return self._data["pem"]

    @cached_property
    def pkcs12(self):
        return self._data["pkcs12"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "id": self.id,
            "isDefault": self.isDefault,
            "expiration": self.expiration,
        }


@dataclasses.dataclass(frozen=True)
class SmtpMsa(Base):
    _data: "schemas.SmtpMsa" = dataclasses.field(default=REQ)

    @cached_property
    def host(self):
        return self._data["host"]

    @cached_property
    def password(self):
        return self._data["password"]

    @cached_property
    def port(self):
        return self._data["port"]

    @cached_property
    def securityMode(self):
        return self._data["securityMode"]

    @cached_property
    def username(self):
        return self._data["username"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"host": self.host, "port": self.port, "securityMode": self.securityMode}


@dataclasses.dataclass(frozen=True)
class Thread(Base):
    _data: "schemas.Thread" = dataclasses.field(default=REQ)

    @cached_property
    def historyId(self):
        return self._data["historyId"]

    @cached_property
    def id(self):
        return self._data["id"]

    @cached_property
    def messages(self) -> list["Message"]:
        return [Message(_data=dct) for dct in self._data.get("messages", [])]

    @cached_property
    def snippet(self):
        return self._data["snippet"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"id": self.id, "snippet": self.snippet}

    def get_deeplink(self, acc: str | None = None) -> str:
        return create_email_deeplink(
            thread_id_or_message_id=self.id,
            acc=acc,
        )


@dataclasses.dataclass(frozen=True)
class VacationSettings(Base):
    _data: "schemas.VacationSettings" = dataclasses.field(default=REQ)

    @cached_property
    def enableAutoReply(self):
        return self._data["enableAutoReply"]

    @cached_property
    def endTime(self):
        return self._data["endTime"]

    @cached_property
    def responseBodyHtml(self):
        return self._data["responseBodyHtml"]

    @cached_property
    def responseBodyPlainText(self):
        return self._data["responseBodyPlainText"]

    @cached_property
    def responseSubject(self):
        return self._data["responseSubject"]

    @cached_property
    def restrictToContacts(self):
        return self._data["restrictToContacts"]

    @cached_property
    def restrictToDomain(self):
        return self._data["restrictToDomain"]

    @cached_property
    def startTime(self):
        return self._data["startTime"]

    @property
    def core_data(self) -> T_KWARGS:
        return {
            "enableAutoReply": self.enableAutoReply,
            "responseSubject": self.responseSubject,
        }


@dataclasses.dataclass(frozen=True)
class WatchRequest(Base):
    _data: "schemas.WatchRequest" = dataclasses.field(default=REQ)

    @cached_property
    def labelFilterAction(self):
        return self._data["labelFilterAction"]

    @cached_property
    def labelFilterBehavior(self):
        return self._data["labelFilterBehavior"]

    @cached_property
    def labelIds(self):
        return self._data.get("labelIds", [])

    @cached_property
    def topicName(self):
        return self._data["topicName"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"topicName": self.topicName, "labelIds": self.labelIds}


@dataclasses.dataclass(frozen=True)
class WatchResponse(Base):
    _data: "schemas.WatchResponse" = dataclasses.field(default=REQ)

    @cached_property
    def expiration(self):
        return self._data["expiration"]

    @cached_property
    def historyId(self):
        return self._data["historyId"]

    @property
    def core_data(self) -> T_KWARGS:
        return {"expiration": self.expiration, "historyId": self.historyId}
