"""
test_notification_service.py
Gelişmiş testler: Mock, parametrik, edge case ve performans testleri
"""
import unittest
from unittest.mock import patch, MagicMock
from src.notification_service import NotificationService

class TestNotificationService(unittest.TestCase):
    def setUp(self):
        self.slack_url = "https://hooks.slack.com/services/xxx/yyy/zzz"
        self.email_cfg = {
            "host": "smtp.example.com",
            "port": 465,
            "user": "user",
            "password": "pass",
            "from": "noreply@example.com"
        }
        self.webhook_url = "https://example.com/webhook"
        self.notifier = NotificationService(
            slack_webhook_url=self.slack_url,
            email_config=self.email_cfg,
            webhook_url=self.webhook_url
        )

    @patch("src.notification_service.requests.post")
    def test_send_slack_success(self, mock_post):
        mock_post.return_value.status_code = 200
        result = self.notifier.send_slack("Test mesaj")
        self.assertTrue(result)

    @patch("src.notification_service.requests.post")
    def test_send_slack_failure(self, mock_post):
        mock_post.return_value.status_code = 400
        result = self.notifier.send_slack("Test mesaj")
        self.assertFalse(result)

    @patch("src.notification_service.smtplib.SMTP_SSL")
    def test_send_email_success(self, mock_smtp):
        smtp_instance = MagicMock()
        mock_smtp.return_value.__enter__.return_value = smtp_instance
        result = self.notifier.send_email("Konu", "İçerik", "to@example.com")
        self.assertTrue(result)

    @patch("src.notification_service.smtplib.SMTP_SSL", side_effect=Exception("SMTP error"))
    def test_send_email_failure(self, mock_smtp):
        result = self.notifier.send_email("Konu", "İçerik", "to@example.com")
        self.assertFalse(result)

    @patch("src.notification_service.requests.post")
    def test_send_webhook_success(self, mock_post):
        mock_post.return_value.status_code = 200
        result = self.notifier.send_webhook({"event": "test"})
        self.assertTrue(result)

    @patch("src.notification_service.requests.post")
    def test_send_webhook_failure(self, mock_post):
        mock_post.return_value.status_code = 500
        result = self.notifier.send_webhook({"event": "test"})
        self.assertFalse(result)

    def test_edge_cases(self):
        # Slack URL yok
        notifier = NotificationService()
        self.assertFalse(notifier.send_slack("Mesaj"))
        # E-posta config yok
        self.assertFalse(notifier.send_email("Konu", "İçerik", "to@example.com"))
        # Webhook URL yok
        self.assertFalse(notifier.send_webhook({"event": "test"}))

if __name__ == "__main__":
    unittest.main()
