/*
 * Decompiled with CFR 0.152.
 */
package org.matsim.core.config;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.log4j.Logger;
import org.matsim.core.config.Config;
import org.matsim.core.config.ConfigGroup;

public class CommandLine {
    private static final Logger logger = Logger.getLogger(CommandLine.class);
    private static final String CONFIG_PREFIX = "config";
    private static final String FLAG_VALUE = "true";
    private final Set<String> allowedPrefixes;
    private final Set<String> allowedOptions;
    private final Set<String> requiredOptions;
    private final Map<String, String> options = new HashMap<String, String>();
    private final List<String> positionalArguments = new LinkedList<String>();
    private final boolean positionalArgumentsAllowed;
    private final boolean allowAnyOption;

    private CommandLine(Set<String> allowedOptions, Set<String> requiredOptions, Set<String> allowedPrefixes, boolean positionalArgumentsAllowed, boolean allowAnyOption) {
        this.allowedOptions = allowedOptions;
        this.requiredOptions = requiredOptions;
        this.allowedPrefixes = allowedPrefixes;
        this.positionalArgumentsAllowed = positionalArgumentsAllowed;
        this.allowAnyOption = allowAnyOption;
    }

    public int getNumberOfPositionalArguments() {
        return this.positionalArguments.size();
    }

    public List<String> getPositionalArguments() {
        return Collections.unmodifiableList(this.positionalArguments);
    }

    public Optional<String> getPositionalArgument(int index) {
        return index < this.positionalArguments.size() ? Optional.of(this.positionalArguments.get(index)) : Optional.empty();
    }

    public String getPositionalArgumentStrict(int index) throws ConfigurationException {
        if (index < this.positionalArguments.size()) {
            return this.positionalArguments.get(index);
        }
        throw new ConfigurationException(String.format("Requested positional command line argument with index %d, but only %d arguments are available", index, this.positionalArguments.size()));
    }

    public Collection<String> getAvailableOptions() {
        return Collections.unmodifiableCollection(this.options.keySet());
    }

    public boolean hasOption(String option) {
        return this.options.containsKey(option);
    }

    public Optional<String> getOption(String option) {
        return this.options.containsKey(option) ? Optional.of(this.options.get(option)) : Optional.empty();
    }

    public String getOptionStrict(String option) throws ConfigurationException {
        if (this.options.containsKey(option)) {
            return this.options.get(option);
        }
        throw new ConfigurationException(String.format("Requested command line option '%s' is not available", option));
    }

    private void process(List<String> args) throws ConfigurationException {
        List<String> arguments = this.flattenArguments(args);
        this.positionalArguments.clear();
        String currentOption = null;
        for (String token : arguments) {
            if (token.startsWith("--")) {
                if (currentOption != null) {
                    this.addOption(currentOption, FLAG_VALUE);
                }
                currentOption = token.substring(2);
                continue;
            }
            if (currentOption != null) {
                this.addOption(currentOption, token);
            } else {
                this.addPositionalArgument(token);
            }
            currentOption = null;
        }
        if (currentOption != null) {
            this.addOption(currentOption, FLAG_VALUE);
        }
        this.checkRequiredOptions();
        this.reportOptions();
    }

    private List<String> flattenArguments(List<String> args) {
        LinkedList<String> flatArguments = new LinkedList<String>();
        for (String argument : args) {
            int index = argument.lastIndexOf("=");
            int bracketIndex = argument.lastIndexOf("]");
            if (bracketIndex > index) {
                index = argument.indexOf("=", bracketIndex);
            }
            if (index > -1) {
                flatArguments.add(argument.substring(0, index));
                flatArguments.add(argument.substring(index + 1));
                continue;
            }
            flatArguments.add(argument);
        }
        return flatArguments;
    }

    private void addPositionalArgument(String value) throws ConfigurationException {
        if (!this.positionalArgumentsAllowed) {
            throw new ConfigurationException(String.format("Positional argument '%s' is not allowed.", value));
        }
        this.positionalArguments.add(value);
    }

    private void addOption(String option, String value) throws ConfigurationException {
        String[] parts;
        if (!(this.allowAnyOption || this.allowedOptions.contains(option) || this.allowedPrefixes.contains((parts = option.split(":"))[0]))) {
            throw new ConfigurationException(String.format("Option '%s' is not allowed.", option));
        }
        this.options.put(option, value);
    }

    private void checkRequiredOptions() throws ConfigurationException {
        LinkedList<String> missingOptions = new LinkedList<String>();
        for (String option : this.requiredOptions) {
            if (this.options.containsKey(option)) continue;
            missingOptions.add(option);
        }
        if (missingOptions.size() > 0) {
            throw new ConfigurationException(String.format("The following options are missing: %s", ((Object)missingOptions).toString()));
        }
    }

    public void applyConfiguration(Config config) throws ConfigurationException {
        List configOptions = this.options.keySet().stream().filter(o -> o.startsWith("config:")).collect(Collectors.toList());
        for (String option : configOptions) {
            this.processConfigOption(config, option, option.substring(CONFIG_PREFIX.length() + 1));
        }
    }

    private void reportOptions() {
        logger.info(String.format("Received %d positional command line arguments:", this.positionalArguments.size()));
        logger.info("   " + String.join((CharSequence)" , ", this.positionalArguments));
        HashMap prefixedOptions = new HashMap();
        LinkedList<String> nonPrefixedOptions = new LinkedList<String>();
        for (String option : this.options.keySet()) {
            int separatorIndex = option.indexOf(":");
            if (separatorIndex > -1) {
                String prefix = option.substring(0, separatorIndex);
                option = option.substring(separatorIndex + 1);
                if (!prefixedOptions.containsKey(prefix)) {
                    prefixedOptions.put(prefix, new LinkedList());
                }
                ((List)prefixedOptions.get(prefix)).add(option);
                continue;
            }
            nonPrefixedOptions.add(option);
        }
        logger.info(String.format("Received %d command line options with %d prefixes:", this.options.size(), prefixedOptions.size()));
        Collections.sort(nonPrefixedOptions);
        for (String option : nonPrefixedOptions) {
            logger.info(String.format("   %s = %s", option, this.options.get(option)));
        }
        LinkedList orderedPrefixes = new LinkedList(prefixedOptions.keySet());
        Collections.sort(orderedPrefixes);
        for (String prefix : orderedPrefixes) {
            logger.info(String.format("   Prefix %s:", prefix));
            for (String option : (List)prefixedOptions.get(prefix)) {
                logger.info(String.format("      %s = %s", option, this.options.get(prefix + ":" + option)));
            }
        }
    }

    private void processConfigOption(Config config, String option, String remainder) throws ConfigurationException {
        String newRemainder;
        String module;
        int separatorIndex = remainder.indexOf(".");
        if (separatorIndex > -1) {
            module = remainder.substring(0, separatorIndex);
            newRemainder = remainder.substring(separatorIndex + 1);
            if (!config.getModules().containsKey(module)) {
                throw new ConfigurationException(String.format("Invalid MATSim option: '%s'. Module '%s' does not exist.", remainder, module));
            }
        } else {
            throw new ConfigurationException(String.format("Malformatted MATSim option: '%s'. Expected MODULE.*", remainder));
        }
        this.processParameter(option, module, config.getModules().get(module), newRemainder);
    }

    private void processParameter(String option, String path, ConfigGroup configGroup, String remainder) throws ConfigurationException {
        if (remainder.contains("[")) {
            int selectorStartIndex = remainder.indexOf("[");
            int selectorEndIndex = remainder.indexOf("]");
            int equalIndex = remainder.indexOf("=");
            if (selectorStartIndex > -1 && selectorEndIndex > -1 && equalIndex > -1 && selectorStartIndex < equalIndex && equalIndex < selectorEndIndex) {
                String parameterSetType = remainder.substring(0, selectorStartIndex);
                String selectionParameter = remainder.substring(selectorStartIndex + 1, equalIndex);
                String selectionValue = remainder.substring(equalIndex + 1, selectorEndIndex);
                String newRemainder = remainder.substring(selectorEndIndex + 1);
                if (newRemainder.startsWith(".")) {
                    newRemainder = newRemainder.substring(1);
                    String newPath = String.format("%s.%s[%s=%s]", path, parameterSetType, selectionParameter, selectionValue);
                    Collection<? extends ConfigGroup> parameterSets = configGroup.getParameterSets(parameterSetType);
                    if (parameterSets.size() > 0) {
                        for (ConfigGroup configGroup2 : parameterSets) {
                            if (!configGroup2.getParams().containsKey(selectionParameter)) continue;
                            String comparisonValue = configGroup2.getParams().get(selectionParameter);
                            if (comparisonValue.equals(selectionValue)) {
                                this.processParameter(option, newPath, configGroup2, newRemainder);
                                return;
                            }
                            if (!parameterSetType.equals("scoringParameters") || !selectionParameter.equals("subpopulation") || !selectionValue.equals("null")) continue;
                            this.processParameter(option, newPath, configGroup2, newRemainder);
                            return;
                        }
                        throw new ConfigurationException(String.format("Parameter set '%s' with %s=%s for %s is not available in %s", parameterSetType, selectionParameter, selectionValue, path, option));
                    }
                    throw new ConfigurationException(String.format("Parameter set of type '%s' for %s is not available in %s", parameterSetType, path, option));
                }
            }
            throw new ConfigurationException(String.format("Malformatted parameter set selector: '%s' in %s. Expected %s.SET_TYPE[PARAM=VALUE].*", remainder, option, path));
        }
        if (!configGroup.getParams().containsKey(remainder)) {
            throw new ConfigurationException(String.format("Parameter %s in %s is not available", remainder, path));
        }
        String value = this.options.get(option);
        configGroup.addParam(remainder, value);
        logger.info(String.format("Setting %s to %s", option, value));
    }

    public static class ConfigurationException
    extends Exception {
        private static final long serialVersionUID = 8427111111975754721L;

        public ConfigurationException(String message) {
            super(message);
        }
    }

    public static class Builder {
        private final Set<String> allowedPrefixes = new HashSet<String>(Collections.singleton("config"));
        private final Set<String> allowedOptions = new HashSet<String>();
        private final Set<String> requiredOptions = new HashSet<String>();
        private boolean positionalArgumentsAllowed = true;
        private boolean allowAnyOption = false;
        private final List<String> arguments;

        public Builder(String[] args) {
            this.arguments = Arrays.asList(args);
            for (String argument : this.arguments) {
                if (argument != null) continue;
                throw new RuntimeException("one of the entries in args is null; this will not work ...");
            }
        }

        public Builder allowPositionalArguments(boolean allow) {
            this.positionalArgumentsAllowed = allow;
            return this;
        }

        public Builder allowAnyOption(boolean allow) {
            this.allowAnyOption = allow;
            return this;
        }

        public Builder allowOptions(Collection<String> options) {
            this.allowedOptions.addAll(options);
            return this;
        }

        public Builder allowOptions(String ... options) {
            this.allowOptions(Arrays.asList(options));
            return this;
        }

        public Builder requireOptions(Collection<String> options) {
            this.allowedOptions.addAll(options);
            this.requiredOptions.addAll(options);
            return this;
        }

        public Builder requireOptions(String ... options) {
            this.requireOptions(Arrays.asList(options));
            return this;
        }

        public Builder allowPrefixes(Collection<String> prefixes) {
            this.allowedPrefixes.addAll(prefixes);
            return this;
        }

        public Builder allowPrefixes(String ... prefixes) {
            this.allowPrefixes(Arrays.asList(prefixes));
            return this;
        }

        public CommandLine build() throws ConfigurationException {
            CommandLine commandLine = new CommandLine(this.allowedOptions, this.requiredOptions, this.allowedPrefixes, this.positionalArgumentsAllowed, this.allowAnyOption);
            commandLine.process(this.arguments);
            return commandLine;
        }
    }
}

