/*
 * Decompiled with CFR 0.152.
 */
package org.matsim.pt.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import javax.xml.parsers.ParserConfigurationException;
import org.matsim.api.core.v01.Id;
import org.matsim.api.core.v01.network.Link;
import org.matsim.api.core.v01.network.Network;
import org.matsim.core.config.ConfigUtils;
import org.matsim.core.network.io.MatsimNetworkReader;
import org.matsim.core.population.routes.NetworkRoute;
import org.matsim.core.scenario.MutableScenario;
import org.matsim.core.scenario.ScenarioUtils;
import org.matsim.core.utils.misc.Time;
import org.matsim.pt.transitSchedule.api.MinimalTransferTimes;
import org.matsim.pt.transitSchedule.api.TransitLine;
import org.matsim.pt.transitSchedule.api.TransitRoute;
import org.matsim.pt.transitSchedule.api.TransitRouteStop;
import org.matsim.pt.transitSchedule.api.TransitSchedule;
import org.matsim.pt.transitSchedule.api.TransitScheduleReader;
import org.matsim.pt.transitSchedule.api.TransitStopFacility;
import org.xml.sax.SAXException;

public abstract class TransitScheduleValidator {
    private TransitScheduleValidator() {
    }

    public static ValidationResult validateNetworkRoutes(TransitSchedule schedule, Network network) {
        ValidationResult result = new ValidationResult();
        if (network == null || network.getLinks().size() == 0) {
            result.addWarning("Cannot validate network routes: No network given!");
            return result;
        }
        for (TransitLine line : schedule.getTransitLines().values()) {
            for (TransitRoute route : line.getRoutes().values()) {
                NetworkRoute netRoute = route.getRoute();
                if (netRoute == null) {
                    result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " has no network route.", ValidationResult.Type.OTHER, Collections.singleton(route.getId())));
                    continue;
                }
                Link prevLink = network.getLinks().get(netRoute.getStartLinkId());
                for (Id<Link> linkId : netRoute.getLinkIds()) {
                    Link link = network.getLinks().get(linkId);
                    if (link == null) {
                        result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " contains a link that is not part of the network: " + linkId, ValidationResult.Type.OTHER, Collections.singleton(route.getId())));
                    } else if (prevLink != null && !prevLink.getToNode().equals(link.getFromNode())) {
                        result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " has inconsistent network route, e.g. between link " + prevLink.getId() + " and " + linkId, ValidationResult.Type.OTHER, Collections.singleton(route.getId())));
                    }
                    prevLink = link;
                }
            }
        }
        return result;
    }

    public static ValidationResult validateStopsOnNetworkRoute(TransitSchedule schedule, Network network) {
        ValidationResult result = new ValidationResult();
        if (network == null || network.getLinks().size() == 0) {
            result.addWarning("Cannot validate stops on network route: No network given!");
            return result;
        }
        for (TransitLine line : schedule.getTransitLines().values()) {
            block1: for (TransitRoute route : line.getRoutes().values()) {
                NetworkRoute netRoute = route.getRoute();
                if (netRoute == null) {
                    result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " has no network route.", ValidationResult.Type.OTHER, Collections.singleton(route.getId())));
                    continue;
                }
                ArrayList<Id<Link>> linkIds = new ArrayList<Id<Link>>();
                linkIds.add(netRoute.getStartLinkId());
                linkIds.addAll(netRoute.getLinkIds());
                linkIds.add(netRoute.getEndLinkId());
                Iterator linkIdIterator = linkIds.iterator();
                Id nextLinkId = (Id)linkIdIterator.next();
                boolean error = false;
                for (TransitRouteStop stop : route.getStops()) {
                    Id<Link> linkRefId = stop.getStopFacility().getLinkId();
                    while (!linkRefId.equals(nextLinkId)) {
                        if (linkIdIterator.hasNext()) {
                            nextLinkId = (Id)linkIdIterator.next();
                            continue;
                        }
                        result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + ": Stop " + stop.getStopFacility().getId() + " cannot be reached along network route.", ValidationResult.Type.ROUTE_HAS_UNREACHABLE_STOP, Collections.singletonList(stop.getStopFacility().getId())));
                        error = true;
                        break;
                    }
                    if (!error) continue;
                    continue block1;
                }
            }
        }
        return result;
    }

    public static ValidationResult validateUsedStopsHaveLinkId(TransitSchedule schedule) {
        ValidationResult result = new ValidationResult();
        for (TransitLine line : schedule.getTransitLines().values()) {
            for (TransitRoute route : line.getRoutes().values()) {
                for (TransitRouteStop stop : route.getStops()) {
                    Id<Link> linkId = stop.getStopFacility().getLinkId();
                    if (linkId != null) continue;
                    result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit Stop Facility " + stop.getStopFacility().getId() + " has no linkId, but is used by transit line " + line.getId() + ", route " + route.getId(), ValidationResult.Type.HAS_NO_LINK_REF, Collections.singleton(stop.getStopFacility().getId())));
                }
            }
        }
        return result;
    }

    public static ValidationResult validateAllStopsExist(TransitSchedule schedule) {
        ValidationResult result = new ValidationResult();
        for (TransitLine line : schedule.getTransitLines().values()) {
            for (TransitRoute route : line.getRoutes().values()) {
                for (TransitRouteStop stop : route.getStops()) {
                    if (stop.getStopFacility() == null) {
                        result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " contains a stop (dep-offset=" + stop.getDepartureOffset() + ") without stop-facility. Most likely, a wrong id was specified in the file.", ValidationResult.Type.HAS_MISSING_STOP_FACILITY, Collections.singletonList(route.getId())));
                        continue;
                    }
                    if (schedule.getFacilities().get(stop.getStopFacility().getId()) != null) continue;
                    result.addIssue(new ValidationResult.ValidationIssue(ValidationResult.Severity.ERROR, "Transit line " + line.getId() + ", route " + route.getId() + " contains a stop (stop-facility " + stop.getStopFacility().getId() + ") that is not contained in the list of all stop facilities.", ValidationResult.Type.HAS_MISSING_STOP_FACILITY, Collections.singletonList(route.getId())));
                }
            }
        }
        return result;
    }

    public static ValidationResult validateOffsets(TransitSchedule schedule) {
        ValidationResult result = new ValidationResult();
        for (TransitLine line : schedule.getTransitLines().values()) {
            for (TransitRoute route : line.getRoutes().values()) {
                ArrayList<TransitRouteStop> stops = new ArrayList<TransitRouteStop>(route.getStops());
                int stopCount = stops.size();
                if (stopCount > 0) {
                    TransitRouteStop stop = stops.get(0);
                    if (Time.isUndefinedTime(stop.getDepartureOffset())) {
                        result.addError("Transit line " + line.getId() + ", route " + route.getId() + ": The first stop does not contain any departure offset.");
                    }
                    for (int i = 1; i < stopCount - 1; ++i) {
                        stop = stops.get(i);
                        if (!Time.isUndefinedTime(stop.getDepartureOffset())) continue;
                        result.addError("Transit line " + line.getId() + ", route " + route.getId() + ": Stop " + i + " does not contain any departure offset.");
                    }
                    stop = stops.get(stopCount - 1);
                    if (!Time.isUndefinedTime(stop.getArrivalOffset())) continue;
                    result.addError("Transit line " + line.getId() + ", route " + route.getId() + ": The last stop does not contain any arrival offset.");
                    continue;
                }
                result.addWarning("Transit line " + line.getId() + ", route " + route.getId() + ": The route has not stops assigned, looks suspicious.");
            }
        }
        return result;
    }

    public static ValidationResult validateTransfers(TransitSchedule schedule) {
        ValidationResult result = new ValidationResult();
        MinimalTransferTimes transferTimes = schedule.getMinimalTransferTimes();
        MinimalTransferTimes.MinimalTransferTimesIterator iter = transferTimes.iterator();
        HashSet<Id<TransitStopFacility>> missingFromStops = new HashSet<Id<TransitStopFacility>>();
        HashSet<Id<TransitStopFacility>> missingToStops = new HashSet<Id<TransitStopFacility>>();
        while (iter.hasNext()) {
            iter.next();
            Id<TransitStopFacility> fromStopId = iter.getFromStopId();
            Id<TransitStopFacility> toStopId = iter.getToStopId();
            double transferTime = iter.getSeconds();
            if (fromStopId == null && toStopId == null) {
                result.addError("Minimal Transfer Times: both fromStop and toStop are null.");
            } else if (fromStopId == null) {
                result.addError("Minimal Transfer Times: fromStop = null, toStop " + toStopId + ".");
            } else if (toStopId == null) {
                result.addError("Minimal Transfer Times: fromStop " + fromStopId + ", toStop = null.");
            }
            if (transferTime <= 0.0) {
                result.addWarning("Minimal Transfer Times: fromStop " + fromStopId + " toStop " + toStopId + " with transferTime = " + transferTime);
            }
            if (schedule.getFacilities().get(fromStopId) == null && missingFromStops.add(fromStopId)) {
                result.addError("Minimal Transfer Times: fromStop " + fromStopId + " does not exist in schedule.");
            }
            if (schedule.getFacilities().get(toStopId) != null || !missingToStops.add(toStopId)) continue;
            result.addError("Minimal Transfer Times: toStop " + toStopId + " does not exist in schedule.");
        }
        return result;
    }

    public static ValidationResult validateAll(TransitSchedule schedule, Network network) {
        ValidationResult v = TransitScheduleValidator.validateUsedStopsHaveLinkId(schedule);
        v.add(TransitScheduleValidator.validateNetworkRoutes(schedule, network));
        try {
            v.add(TransitScheduleValidator.validateStopsOnNetworkRoute(schedule, network));
        }
        catch (NullPointerException e) {
            v.addError("Exception during 'validateStopsOnNetworkRoute'. Most likely something is wrong in the file, but it cannot be specified in more detail." + Arrays.toString(e.getStackTrace()));
        }
        v.add(TransitScheduleValidator.validateAllStopsExist(schedule));
        v.add(TransitScheduleValidator.validateOffsets(schedule));
        v.add(TransitScheduleValidator.validateTransfers(schedule));
        return v;
    }

    public static void printResult(ValidationResult result) {
        if (result.isValid()) {
            System.out.println("Schedule appears valid!");
        } else {
            System.out.println("Schedule is NOT valid!");
        }
        if (result.getErrors().size() > 0) {
            System.out.println("Validation errors:");
            for (String e : result.getErrors()) {
                System.out.println(e);
            }
        }
        if (result.getWarnings().size() > 0) {
            System.out.println("Validation warnings:");
            for (String w : result.getWarnings()) {
                System.out.println(w);
            }
        }
    }

    public static void main(String[] args) throws IOException, SAXException, ParserConfigurationException {
        if (args.length > 2 || args.length < 1) {
            System.err.println("Usage: TransitScheduleValidator transitSchedule.xml [network.xml]");
            return;
        }
        MutableScenario s2 = (MutableScenario)ScenarioUtils.createScenario(ConfigUtils.createConfig());
        s2.getConfig().transit().setUseTransit(true);
        TransitSchedule ts = s2.getTransitSchedule();
        Network net = s2.getNetwork();
        if (args.length > 1) {
            new MatsimNetworkReader(s2.getNetwork()).readFile(args[1]);
        }
        new TransitScheduleReader(s2).readFile(args[0]);
        ValidationResult v = TransitScheduleValidator.validateAll(ts, net);
        TransitScheduleValidator.printResult(v);
    }

    public static class ValidationResult {
        private boolean isValid = true;
        private final List<ValidationIssue> issues = new ArrayList<ValidationIssue>();

        public boolean isValid() {
            return this.isValid;
        }

        public List<String> getWarnings() {
            ArrayList<String> result = new ArrayList<String>();
            for (ValidationIssue issue : this.issues) {
                if (issue.severity != Severity.WARNING) continue;
                result.add(issue.getMessage());
            }
            return Collections.unmodifiableList(result);
        }

        public List<String> getErrors() {
            ArrayList<String> result = new ArrayList<String>();
            for (ValidationIssue issue : this.issues) {
                if (issue.severity != Severity.ERROR) continue;
                result.add(issue.getMessage());
            }
            return Collections.unmodifiableList(result);
        }

        public List<ValidationIssue> getIssues() {
            return Collections.unmodifiableList(this.issues);
        }

        public void addWarning(String warning) {
            this.issues.add(new ValidationIssue(Severity.WARNING, warning, Type.OTHER, Collections.emptyList()));
        }

        public void addError(String error) {
            this.issues.add(new ValidationIssue(Severity.ERROR, error, Type.OTHER, Collections.emptyList()));
            this.isValid = false;
        }

        public void addIssue(ValidationIssue issue) {
            this.issues.add(issue);
        }

        public void add(ValidationResult otherResult) {
            this.issues.addAll(otherResult.getIssues());
            this.isValid = this.isValid && otherResult.isValid;
        }

        public static class ValidationIssue<T> {
            private final Severity severity;
            private final String message;
            private final Type errorCode;
            private final Collection<Id<T>> entities;

            public ValidationIssue(Severity severity, String message, Type errorCode, Collection<Id<T>> entities) {
                this.severity = severity;
                this.message = message;
                this.errorCode = errorCode;
                this.entities = entities;
            }

            public Severity getSeverity() {
                return this.severity;
            }

            public String getMessage() {
                return this.message;
            }

            public Type getErrorCode() {
                return this.errorCode;
            }

            public Collection<Id<T>> getEntities() {
                return this.entities;
            }
        }

        public static enum Type {
            HAS_MISSING_STOP_FACILITY,
            HAS_NO_LINK_REF,
            ROUTE_HAS_UNREACHABLE_STOP,
            OTHER;

        }

        public static enum Severity {
            WARNING,
            ERROR;

        }
    }
}

