import time
import uuid

from enum import Enum
from typing import Optional, Union, Dict, List, Type
from .attachment import Attachment
from .basemodel import BaseModel
from .. import QaseUtils


class StepType(Enum):
    TEXT = 'text'
    ASSERT = 'assert'
    GHERKIN = 'gherkin'
    REQUEST = 'request'
    DB_QUERY = 'db_query'
    SLEEP = 'sleep'


class StepTextData(BaseModel):
    def __init__(self, action: str, expected_result: Optional[str] = None):
        self.action = action
        self.expected_result = expected_result
        self.input_data = None


class StepAssertData(BaseModel):
    def __init__(self, expected: str, actual: str, message: str):
        self.expected = expected
        self.actual = actual
        self.message = message


class StepGherkinData(BaseModel):
    def __init__(self, keyword: str, name: str, line: int, data: Optional[str] = None):
        self.keyword = keyword
        self.name = name
        self.line = line
        self.data = data


class StepRequestData(BaseModel):
    def __init__(self, request_body: str, request_headers: Dict[str, str], request_method: str, request_url: str):
        self.response_headers = None
        self.response_body = None
        self.status_code = None
        if isinstance(request_body, bytes):
            try:
                request_body = request_body.decode('utf-8')
            except UnicodeDecodeError:
                # For binary data (like file uploads), keep as base64 encoded string
                import base64
                request_body = base64.b64encode(request_body).decode('ascii')
        self.request_body = request_body
        if isinstance(request_headers, bytes):
            try:
                request_headers = request_headers.decode('utf-8')
            except UnicodeDecodeError:
                # For binary headers, keep as base64 encoded string
                import base64
                request_headers = base64.b64encode(request_headers).decode('ascii')
        self.request_headers = request_headers
        self.request_method = request_method
        self.request_url = request_url

    def add_response(self, status_code: int, response_body: Optional[str] = None,
                     response_headers: Optional[Dict[str, str]] = None):
        self.status_code = status_code

        if isinstance(response_body, bytes):
            try:
                response_body = response_body.decode('utf-8')
            except UnicodeDecodeError:
                # For binary data (like file downloads), keep as base64 encoded string
                import base64
                response_body = base64.b64encode(response_body).decode('ascii')
        self.response_body = response_body
        if isinstance(response_headers, bytes):
            try:
                response_headers = response_headers.decode('utf-8')
            except UnicodeDecodeError:
                # For binary headers, keep as base64 encoded string
                import base64
                response_headers = base64.b64encode(response_headers).decode('ascii')
        self.response_headers = response_headers


class StepDbQueryData(BaseModel):
    def __init__(self, query: str, expected_result: str = None, 
                 database_type: str = None, execution_time: float = None,
                 rows_affected: int = None, connection_info: str = None):
        self.query = query
        self.expected_result = expected_result
        self.database_type = database_type
        self.execution_time = execution_time
        self.rows_affected = rows_affected
        self.connection_info = connection_info


class StepSleepData(BaseModel):
    def __init__(self, duration: int):
        self.duration = duration


class StepExecution(BaseModel):
    def __init__(self, status: Optional[str] = 'untested', end_time: int = 0, duration: int = 0):
        self.start_time = QaseUtils.get_real_time()
        self.status = status
        self.end_time = end_time
        self.duration = duration
        self.attachments = []

    def set_status(self, status: Optional[str]):
        if status in ['passed', 'failed', 'skipped', 'blocked', 'untested', 'invalid']:
            self.status = status
        else:
            raise ValueError('Step status must be one of: passed, failed, skipped, blocked, untested, invalid')

    def complete(self):
        self.end_time = QaseUtils.get_real_time()
        self.duration = int((self.end_time - self.start_time) * 1000)

    def add_attachment(self, attachment: Attachment):
        self.attachments.append(attachment)


class Step(BaseModel):
    def __init__(self,
                 step_type: StepType,
                 id: Optional[str],
                 data: Optional[
                     Union[StepTextData, StepAssertData, StepGherkinData, StepRequestData, StepSleepData]] = None,
                 parent_id: Optional[str] = None
                 ):
        if id:
            self.id = id
        else:
            self.id = str(uuid.uuid4())

        self.step_type = step_type
        self.data = data
        self.parent_id = parent_id
        self.execution = StepExecution()
        self.steps = []

    def set_parent_id(self, parent_id: Optional[str]):
        self.parent_id = parent_id

    def get_parent_id(self) -> Optional[str]:
        return self.parent_id

    def get_steps(self, as_dict: bool = False):
        if as_dict:
            return {step.id: step for step in self.steps}
        else:
            return self.steps

    def set_data(self, data: Union[StepTextData, StepAssertData, StepGherkinData, StepRequestData, StepSleepData]):
        self.data = data

    def add_step(self, step: Type['Step']):
        self.steps.append(step)

    def set_steps(self, steps: List[Type['Step']]):
        self.steps = steps

    def add_attachment(self, attachment: Attachment):
        self.execution.add_attachment(attachment)
