"""
BuySellInfo class - информация о возможности покупки/продажи инструмента
"""

from dataclasses import dataclass
from typing import Optional
from .base import BaseDataStructure


@dataclass
class BuySellInfo(BaseDataStructure):
    """
    Функция getBuySellInfo возвращает таблицу Lua с параметрами из таблицы QUIK «Купить/Продать», 
    означающими возможность купить либо продать указанный инструмент «sec_code» класса «class_code», 
    указанным клиентом «client_code» фирмы «firmid», по указанной цене «price». 
    Если цена равна «0», то используются лучшие значения спроса/предложения.
    """
    
    # Признак маржинальности инструмента. Возможные значения:
    # «0» – не маржинальная
    # «1» – маржинальная
    is_margin_sec: Optional[str] = None
    
    # Принадлежность инструмента к списку инструментов, принимаемых в обеспечение. Возможные значения: 
    # «0» – не принимается в обеспечение; 
    # «1» – принимается в обеспечение.
    is_asset_sec: Optional[str] = None
    
    # Текущая позиция по инструменту, в лотах
    balance: Optional[str] = None
    
    # Оценка количества лотов, доступных на покупку по указанной цене
    # В зависимости от настроек сервера QUIK, величина может выражаться в лотах или в штуках
    can_buy: Optional[str] = None
    
    # Оценка количества лотов, доступных на продажу по указанной цене
    # В зависимости от настроек сервера QUIK, величина может выражаться в лотах или в штуках
    can_sell: Optional[str] = None
    
    # Денежная оценка позиции по инструменту по ценам спроса/предложения
    position_valuation: Optional[str] = None
    
    # Оценка стоимости позиции по цене последней сделки
    value: Optional[str] = None
    
    # Оценка стоимости позиции клиента, рассчитанная по цене закрытия предыдущей торговой сессии
    open_value: Optional[str] = None
    
    # Предельный размер позиции по данному инструменту, принимаемый в обеспечение длинных позиций
    lim_long: Optional[str] = None
    
    # Коэффициент дисконтирования, применяемый для длинных позиций по данному инструменту
    long_coef: Optional[str] = None
    
    # Предельный размер короткой позиции по данному инструменту
    lim_short: Optional[str] = None
    
    # Коэффициент дисконтирования, применяемый для коротких позиций по данному инструменту
    short_coef: Optional[str] = None
    
    # Оценка стоимости позиции по цене последней сделки, с учетом дисконтирующих коэффициентов
    value_coef: Optional[str] = None
    
    # Оценка стоимости позиции клиента, рассчитанная по цене закрытия предыдущей торговой сессии с учетом дисконтирующих коэффициентов
    open_value_coef: Optional[str] = None
    
    # Процентное отношение стоимости позиции по данному инструменту к стоимости всех активов клиента, рассчитанное по текущим ценам
    share: Optional[str] = None
    
    # Средневзвешенная стоимость коротких позиций по инструментам
    short_wa_price: Optional[str] = None
    
    # Средневзвешенная стоимость длинных позиций по инструментам
    long_wa_price: Optional[str] = None
    
    # Разница между средневзвешенной ценой приобретения инструментов и их рыночной оценки
    profit_loss: Optional[str] = None
    
    # Коэффициент корреляции между инструментами
    spread_hc: Optional[str] = None
    
    # Максимально возможное количество инструментов в заявке на покупку этого инструмента на этом классе на собственные средства клиента, исходя из цены лучшего предложения
    can_buy_own: Optional[str] = None
    
    # Срок расчётов. Возможные значения: положительные целые числа, начиная с «0», соответствующие срокам расчётов из таблицы «Позиции по инструментам»:
    # «0» – T0, «1» – T1, «2» – T2 и т.д.
    limit_kind: Optional[str] = None
    
    # Эффективный начальный дисконт для длинной позиции. Заполняется для клиентов типа «МД»
    d_long: Optional[str] = None
    
    # Эффективный минимальный дисконт для длинной позиции. Заполняется для клиентов типа «МД»
    d_min_long: Optional[str] = None
    
    # Эффективный начальный дисконт для короткой позиции. Заполняется для клиентов типа «МД»
    d_short: Optional[str] = None
    
    # Эффективный минимальный дисконт для короткой позиции. Заполняется для клиентов типа «МД»
    d_min_short: Optional[str] = None
    
    # Тип клиента. Возможные значения: 
    # «1» – «МЛ»; 
    # «2» – «МП»; 
    # «3» – «МОП»; 
    # «4» – «МД»
    client_type: Optional[str] = None
    
    # Признак того, является ли инструмент разрешенным для покупки на заемные средства. Возможные значения: 
    # «0» – не разрешен; 
    # «1» – разрешен
    # Заполняется для клиентов типа «МД»
    is_long_allowed: Optional[str] = None
    
    # Признак того, является ли инструмент разрешенным для продажи на заемные средства. Возможные значения: 
    # «0» – не разрешен; 
    # «1» – разрешен
    # Заполняется для клиентов типа «МД»
    is_short_allowed: Optional[str] = None
    
    def get_client_type_description(self) -> str:
        """
        Возвращает описание типа клиента
        """
        client_types = {
            "1": "МЛ (Маржинальное кредитование)",
            "2": "МП (Маржинальное РЕПО)",
            "3": "МОП (Маржинальные операции)",
            "4": "МД (Маржинальные деривативы)"
        }
        return client_types.get(self.client_type, f"Неизвестный тип: {self.client_type}")
    
    def get_settlement_term_description(self) -> str:
        """
        Возвращает описание срока расчётов
        """
        if self.limit_kind is None:
            return "Не указан"
        
        try:
            term = int(self.limit_kind)
            return f"T{term}"
        except (ValueError, TypeError):
            return f"Некорректный срок: {self.limit_kind}"
    
