from __future__ import annotations

from dataclasses import dataclass, field
from time import time
from typing import Any, Dict, Optional, Tuple
import json
import urllib.error
import urllib.parse
import urllib.request

from .origin import Hostname
from .request import Request
from .support.exceptions import StopExecutionRequested


@dataclass
class Client:
    port_number: int = 23517
    host: str = "localhost"

    _cache: Dict[str, Tuple[bool, float]] = field(default_factory=dict, init=False, repr=False)
    _fingerprint: str = field(init=False)
    _network_host: str = field(init=False)

    def __post_init__(self) -> None:
        # For DSN-style hosts like "ray@127.0.0.1" (Buggregator Ray), we want
        # to preserve the full value in the URL so that "ray" is treated as
        # the HTTP userinfo and the host is still resolvable (e.g. 127.0.0.1).
        host = self.host
        if "@" in host:
            host = host.split("@", 1)[1]
        self._network_host = host
        self._fingerprint = f"{self._network_host}:{self.port_number}"

    # --- availability -----------------------------------------------------------

    def server_is_available(self) -> bool:
        # If using a DSN-style host (e.g. "ray@host" for Buggregator),
        # skip the availability check and assume the server is reachable.
        # The send() call will still handle any real network errors.
        if "@" in self.host:
            return True

        # purge expired entries
        now = time()
        self._cache = {
            k: v for k, v in self._cache.items() if now < v[1]
        }

        if self._fingerprint not in self._cache:
            self._perform_availability_check()

        return self._cache.get(self._fingerprint, (True, 0.0))[0]

    def _perform_availability_check(self) -> bool:
        success = False
        try:
            # Probe the root endpoint instead of a dedicated health check path.
            # The original Ray desktop server responds on "/" but may not
            # implement "_availability_check", while Buggregator/Ray servers
            # are generally fine with a GET to the root as well.
            url = self._build_url("")
            req = urllib.request.Request(url, method="GET")
            try:
                with urllib.request.urlopen(req, timeout=2) as resp:  # noqa: S310
                    # Any 2xx/3xx HTTP response means the server is reachable.
                    success = True
            except urllib.error.HTTPError:
                # Non-2xx responses still mean the server is up and accepting
                # connections, which is all we care about for availability.
                success = True
        except Exception:
            success = False
        finally:
            expires_at = time() + 30.0
            self._cache[self._fingerprint] = (success, expires_at)
            return success

    # --- sending ----------------------------------------------------------------

    def send(self, request: Request) -> None:
        if not self.server_is_available():
            return

        url = self._build_url("")
        data = request.to_json().encode("utf-8")
        req = urllib.request.Request(
            url,
            data=data,
            method="POST",
            headers={
                "Accept": "application/json",
                "Content-Type": "application/json",
                "User-Agent": "Ray Python 0.1.0",
            },
        )

        try:
            with urllib.request.urlopen(req, timeout=2):  # noqa: S310
                pass
        except Exception:
            # Mirror PHP behavior: swallow errors for now.
            return

    # --- locks ------------------------------------------------------------------

    def lock_exists(self, lock_name: str, project_name: str) -> bool:
        if not self.server_is_available():
            return False

        query = urllib.parse.urlencode(
            {
                "hostname": Hostname.get(),
                "project_name": project_name,
            }
        )

        url = self._build_url(f"locks/{lock_name}?{query}")
        req = urllib.request.Request(url, method="GET")

        try:
            with urllib.request.urlopen(req, timeout=2) as resp:  # noqa: S310
                body = resp.read().decode("utf-8")
        except Exception:
            return False

        if not body:
            return False

        try:
            parsed = json.loads(body)
        except json.JSONDecodeError:
            return False

        if parsed.get("stop_execution"):
            raise StopExecutionRequested()

        return bool(parsed.get("active", False))

    # --- helpers ----------------------------------------------------------------

    def _build_url(self, path: str) -> str:
        path = path.lstrip("/")
        return f"http://{self._network_host}:{self.port_number}/{path}"
