from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Dict, Optional
import inspect
import os
import socket


@dataclass
class Origin:
    file: Optional[str]
    line_number: Optional[int]
    hostname: str

    def to_dict(self) -> Dict[str, Any]:
        return {
            "file": self.file,
            "line_number": self.line_number,
            "hostname": self.hostname,
        }

    def fingerprint(self) -> str:
        return str(hash((self.file, self.line_number, self.hostname)))


class Hostname:
    _hostname: Optional[str] = None

    @classmethod
    def get(cls) -> str:
        if cls._hostname is None:
            cls._hostname = socket.gethostname()
        return cls._hostname

    @classmethod
    def set(cls, hostname: str) -> None:
        cls._hostname = hostname


class DefaultOriginFactory:
    """Rudimentary origin factory using Python's inspect.stack()."""

    def get_origin(self) -> Origin:
        frame_info = self._get_frame()
        if frame_info is None:
            return Origin(None, None, Hostname.get())
        filename = os.path.abspath(frame_info.filename)
        line = frame_info.lineno
        return Origin(filename, line, Hostname.get())

    def _get_frame(self) -> Optional[inspect.FrameInfo]:
        frames = inspect.stack()[1:]
        for fi in frames:
            filename = os.path.abspath(fi.filename)
            # Skip frames inside this package
            if os.sep + "python_ray" + os.sep in filename:
                continue
            return fi
        return None
