from __future__ import annotations

from typing import Any, Dict, List
import os
import traceback
import inspect

from .base import Payload


class ExceptionPayload(Payload):
    def __init__(self, exc: BaseException, meta: Dict[str, Any] | None = None) -> None:
        super().__init__()
        self._exc = exc
        self._meta = meta or {}

    def get_type(self) -> str:
        return "exception"

    def get_content(self) -> Dict[str, Any]:
        tb = self._exc.__traceback__
        frames: List[Dict[str, Any]] = []

        # Walk the traceback so we have access to real frame objects (for
        # inferring class names via self/cls), similar to Ray.trace().
        while tb is not None:
            frame = tb.tb_frame
            lineno = tb.tb_lineno
            filename = os.path.abspath(frame.f_code.co_filename)

            # Best-effort class detection: look for self/cls in locals.
            cls_name = ""
            if "self" in frame.f_locals:
                cls_name = frame.f_locals["self"].__class__.__name__
            elif "cls" in frame.f_locals:
                cls_obj = frame.f_locals["cls"]
                if isinstance(cls_obj, type):
                    cls_name = cls_obj.__name__

            frames.append(
                {
                    "file_name": self.replace_remote_path_with_local_path(filename),
                    "line_number": lineno,
                    "class": cls_name,
                    "method": frame.f_code.co_name,
                    "vendor_frame": False,
                    "snippet": [],
                }
            )

            tb = tb.tb_next

        return {
            "class": self._exc.__class__.__name__,
            "message": str(self._exc),
            "frames": frames,
            "meta": self._meta,
        }
