from __future__ import annotations

from typing import Any, Dict, Iterable, List

from .base import Payload


def _to_clipboard_data(value: Any) -> str:
    if isinstance(value, (str, int, float, bool)) or value is None:
        return str(value)
    try:
        return repr(value)
    except Exception:
        return ""


class LogPayload(Payload):
    def __init__(self, values: Iterable[Any], raw_values: Iterable[Any] | None = None) -> None:
        super().__init__()
        self._values = list(values)
        raw_values = list(raw_values or [])
        clipboard_data = "" if not raw_values else _to_clipboard_data(raw_values)
        self._meta = [
            {
                "clipboard_data": clipboard_data[:20000],
            }
        ]

    @classmethod
    def from_arguments(cls, arguments: Iterable[Any]) -> "LogPayload":
        # In the PHP version, arguments are converted via ArgumentConverter; here
        # we keep them as-is and rely on Ray to display them.
        return cls(list(arguments), list(arguments))

    def get_type(self) -> str:
        return "log"

    def get_content(self) -> Dict[str, Any]:
        return {
            "values": self._values,
            "meta": self._meta,
        }
