from __future__ import annotations

from dataclasses import dataclass, field
from time import sleep
from types import TracebackType
from typing import Any, Callable, ClassVar, Dict, Iterable, List, Optional, Sequence, Tuple, Type
import sys
import uuid as _uuid

from .client import Client
from .request import Request
from .settings import Settings
from .origin import DefaultOriginFactory
from .support.counters import Counters
from .support.limiters import Limiters
from .support.rate_limiter import RateLimiter
from .support.exceptions import StopExecutionRequested
from .support.ignored_value import IgnoredValue
from .support.invador import Invador
from .payloads import (
    Payload,
    PayloadFactory,
    TextPayload,
    NotifyPayload,
    TablePayload,
    CustomPayload,
    ClearAllPayload,
    ColorPayload,
    ScreenColorPayload,
    SizePayload,
    LabelPayload,
    HidePayload,
    RemovePayload,
    NewScreenPayload,
    SeparatorPayload,
    ShowAppPayload,
    HideAppPayload,
    ImagePayload,
    FileContentsPayload,
    XmlPayload,
    JsonStringPayload,
    DecodedJsonPayload,
    ConfettiPayload,
    CreateLockPayload,
    MeasurePayload,
    CallerPayload,
    TracePayload,
    CarbonPayload,
    PythonInfoPayload,
    ExceptionPayload,
)


BeforeSendHook = Callable[[Sequence[Payload], Dict[str, Any]], None]


@dataclass
class Ray:
    """Python port of Spatie Ray.

    Not all methods are implemented yet; this is a starting point with core
    sending logic and a few convenience helpers.
    """

    settings: Settings
    uuid: str = field(default_factory=lambda: str(_uuid.uuid4()))

    # Class-level shared state
    _client: ClassVar[Optional[Client]] = None
    _counters: ClassVar[Optional[Counters]] = None
    _limiters: ClassVar[Optional[Limiters]] = None
    _rate_limiter: ClassVar[RateLimiter] = RateLimiter.disabled()
    _enabled: ClassVar[Optional[bool]] = None
    _project_name: ClassVar[str] = ""
    _before_send_request: ClassVar[Optional[BeforeSendHook]] = None

    # Per-instance flags
    can_send_payload: bool = True
    _limit_origin_fingerprint: Optional[str] = None

    # timers name -> (start_time, last_lap_time)
    _stopwatches: ClassVar[dict[str, tuple[float, float]]] = {}

    # collected exceptions (from callables passed into send())
    _caught_exceptions: ClassVar[list[BaseException]] = []

    def __post_init__(self) -> None:
        if Ray._client is None:
            Ray._client = Client(self.settings.port, self.settings.host)
        if Ray._counters is None:
            Ray._counters = Counters()
        if Ray._limiters is None:
            Ray._limiters = Limiters()
        if Ray._enabled is None:
            Ray._enabled = bool(self.settings.enable)

    # --- basic enable / disable -------------------------------------------------

    def project(self, project_name: str) -> "Ray":
        Ray._project_name = project_name
        return self

    def enable(self) -> "Ray":
        Ray._enabled = True
        return self

    def disable(self) -> "Ray":
        Ray._enabled = False
        return self

    def enabled(self) -> bool:
        return bool(Ray._enabled) or Ray._enabled is None

    def disabled(self) -> bool:
        return Ray._enabled is False

    # --- core send API ----------------------------------------------------------

    def new_screen(self, name: str = "") -> "Ray":
        return self._send_request([NewScreenPayload(name)])

    def clear_all(self) -> "Ray":
        return self._send_request([ClearAllPayload()])

    def clear_screen(self) -> "Ray":
        return self.new_screen("")

    def color(self, color: str) -> "Ray":
        return self._send_request([ColorPayload(color)])

    def screen_color(self, color: str) -> "Ray":
        return self._send_request([ScreenColorPayload(color)])

    def label(self, label: str) -> "Ray":
        return self._send_request([LabelPayload(label)])

    def size(self, size: str) -> "Ray":
        return self._send_request([SizePayload(size)])

    def remove(self) -> "Ray":
        return self._send_request([RemovePayload()])

    def hide(self) -> "Ray":
        return self._send_request([HidePayload()])

    def send(self, *values: Any) -> "Ray":
        if not values:
            return self

        processed: list[Any] = []
        for arg in values:
            if callable(arg) and not isinstance(arg, Ray):
                try:
                    result = arg(self)
                except TypeError:
                    # callable that doesn't accept Ray; treat as plain value
                    processed.append(arg)
                except Exception as exc:  # noqa: BLE001
                    Ray._caught_exceptions.append(exc)
                    processed.append(IgnoredValue.make())
                else:
                    if isinstance(result, Ray):
                        processed.append(IgnoredValue.make())
                    else:
                        processed.append(result)
            else:
                processed.append(arg)

        processed = [a for a in processed if not isinstance(a, IgnoredValue)]
        if not processed:
            return self

        if self.settings.always_send_raw_values:
            payloads: Iterable[Payload] = [PayloadFactory.log(value) for value in processed]
        else:
            payloads = PayloadFactory.create_for_values(processed)

        return self._send_request(list(payloads))

    def raw(self, *values: Any) -> "Ray":
        if not values:
            return self
        payloads = [PayloadFactory.log(value) for value in values]
        return self._send_request(payloads)

    def text(self, text: str) -> "Ray":
        return self._send_request([TextPayload(text)])

    def notify(self, text: str) -> "Ray":
        return self._send_request([NotifyPayload(text)])

    # color & size convenience methods (parity with RayColors/RaySizes traits)
    def green(self) -> "Ray":
        return self.color("green")

    def orange(self) -> "Ray":
        return self.color("orange")

    def red(self) -> "Ray":
        return self.color("red")

    def purple(self) -> "Ray":
        return self.color("purple")

    def blue(self) -> "Ray":
        return self.color("blue")

    def gray(self) -> "Ray":
        return self.color("gray")

    def screen_green(self) -> "Ray":
        return self.screen_color("green")

    def screen_orange(self) -> "Ray":
        return self.screen_color("orange")

    def screen_red(self) -> "Ray":
        return self.screen_color("red")

    def screen_purple(self) -> "Ray":
        return self.screen_color("purple")

    def screen_blue(self) -> "Ray":
        return self.screen_color("blue")

    def screen_gray(self) -> "Ray":
        return self.screen_color("gray")

    def small(self) -> "Ray":
        return self.size("sm")

    def large(self) -> "Ray":
        return self.size("lg")

    def table(self, values: Sequence[Any], label: str = "Table") -> "Ray":
        return self._send_request([TablePayload(values, label)])

    def to_json(self, *values: Any) -> "Ray":
        if not values:
            return self
        payloads = [JsonStringPayload(v) for v in values]
        return self._send_request(payloads)

    def json(self, *json_strings: str) -> "Ray":
        if not json_strings:
            return self
        payloads = [DecodedJsonPayload(js) for js in json_strings]
        return self._send_request(payloads)

    def file(self, filename: str) -> "Ray":
        return self._send_request([FileContentsPayload(filename)])

    def image(self, location: str) -> "Ray":
        return self._send_request([ImagePayload(location)])

    def pythoninfo(self, *properties: str) -> "Ray":
        return self._send_request([PythonInfoPayload(*properties)])

    def object(self, value: Any) -> "Ray":
        """Send an arbitrary object as JSON, if possible.

        This tries, in order:
        - dataclasses.asdict
        - Pydantic v2 model_dump()
        - .dict() on the object
        - public attributes from __dict__

        If conversion or JSON serialization fails, falls back to send(repr(value)).
        """
        import dataclasses
        import json as _json

        if isinstance(value, (str, int, float, bool)) or value is None:
            return self.send(value)

        data: Any | None = None

        if dataclasses.is_dataclass(value):
            data = dataclasses.asdict(value)
        elif hasattr(value, "model_dump") and callable(value.model_dump):  # type: ignore[attr-defined]
            data = value.model_dump()  # type: ignore[call-arg]
        elif hasattr(value, "dict") and callable(value.dict):  # type: ignore[attr-defined]
            data = value.dict()  # type: ignore[call-arg]
        elif hasattr(value, "__dict__"):
            raw = value.__dict__  # type: ignore[attr-defined]
            public = {k: v for k, v in raw.items() if not k.startswith("_")}
            # If there are public attributes, prefer those. If there are only
            # private attributes or the mapping is empty, treat this as
            # "no structured data" so we can fall back to repr(value).
            if public:
                data = public
            elif raw:
                data = raw
            else:
                data = None

        if data is None:
            return self.send(repr(value))

        # Send pretty-printed JSON as text so Buggregator shows it as a string
        # instead of a generic [object Object].
        json_str = _json.dumps(data, default=str, indent=2, sort_keys=True)
        return self.text(json_str)

    def xml(self, xml: str) -> "Ray":
        return self._send_request([XmlPayload(xml)])

    def url(self, url: str, label: str = "") -> "Ray":
        if not url.startswith("http"):
            url = f"https://{url}"
        if not label:
            label = url
        link = f"<a href='{url}'>{label}</a>"
        return self.html(link)

    def link(self, url: str, label: str = "") -> "Ray":
        return self.url(url, label)

    def html(self, html: str = "") -> "Ray":
        return self._send_request([CustomPayload(html, "")])

    def confetti(self) -> "Ray":
        return self._send_request([ConfettiPayload()])

    def send_custom(self, content: str, label: str = "") -> "Ray":
        return self._send_request([CustomPayload(content, label)])

    # --- rate limiter & meta / request building --------------------------------

    @classmethod
    def rate_limiter(cls) -> RateLimiter:
        return cls._rate_limiter

    @classmethod
    def before_send_request(cls, hook: Optional[BeforeSendHook]) -> None:
        cls._before_send_request = hook

    def _send_request(self, payloads: Sequence[Payload], meta: Optional[Dict[str, Any]] = None) -> "Ray":
        if not self.enabled():
            return self
        if not payloads:
            return self
        if not self.can_send_payload:
            return self

        # Per-origin limiter support (simplified): use origin fingerprint
        if self._limit_origin_fingerprint is not None and Ray._limiters is not None:
            if not Ray._limiters.can_send_payload(self._limit_origin_fingerprint):
                return self
            Ray._limiters.increment(self._limit_origin_fingerprint)

        # Rate limiting
        rl = Ray.rate_limiter()
        if rl.is_max_reached() or rl.is_max_per_second_reached():
            self._notify_rate_limit_reached()
            return self

        all_meta: Dict[str, Any] = {
            "python_version": sys.version.split()[0],
            "python_implementation": sys.implementation.name,
            "project_name": Ray._project_name,
        }
        if meta:
            all_meta.update(meta)

        if Ray._before_send_request is not None:
            Ray._before_send_request(payloads, all_meta)

        # Attach path mapping
        for payload in payloads:
            payload.remote_path = self.settings.remote_path
            payload.local_path = self.settings.local_path

        request = Request(self.uuid, list(payloads), all_meta)
        if Ray._client is not None:
            Ray._client.send(request)

        rl.hit()
        return self

    # --- tracing, caller, timers, counters, etc. --------------------------------

    def trace(self, offset: int | None = None, limit: int | None = None) -> "Ray":
        import inspect, os

        raw_frames = inspect.stack()[1:]
        frames = []
        for fi in raw_frames:
            filename = os.path.abspath(fi.filename)
            if os.sep + "python_ray" + os.sep in filename:
                continue

            # Try to infer a class name from the frame's locals (self/cls)
            cls_name = ""
            if "self" in fi.frame.f_locals:
                cls_name = fi.frame.f_locals["self"].__class__.__name__
            elif "cls" in fi.frame.f_locals:
                cls_obj = fi.frame.f_locals["cls"]
                if isinstance(cls_obj, type):
                    cls_name = cls_obj.__name__

            frames.append(
                {
                    "file_name": self._apply_path_mapping(filename),
                    "line_number": fi.lineno,
                    # Use an empty string when there is no class context so
                    # the Ray UI does not render "null:".
                    "class": cls_name,
                    "method": fi.function,
                    "vendor_frame": False,
                }
            )
        payload = TracePayload(frames, start_from_index=offset, limit=limit)
        return self._send_request([payload])

    def backtrace(self, offset: int | None = None, limit: int | None = None) -> "Ray":
        return self.trace(offset=offset, limit=limit)

    def caller(self) -> "Ray":
        import inspect, os

        raw_frames = inspect.stack()[1:]
        for fi in raw_frames:
            filename = os.path.abspath(fi.filename)
            if os.sep + "python_ray" + os.sep in filename:
                continue

            # Try to infer a class name from the frame's locals (self/cls)
            cls_name = ""
            if "self" in fi.frame.f_locals:
                cls_name = fi.frame.f_locals["self"].__class__.__name__
            elif "cls" in fi.frame.f_locals:
                cls_obj = fi.frame.f_locals["cls"]
                if isinstance(cls_obj, type):
                    cls_name = cls_obj.__name__

            frame = {
                "file_name": self._apply_path_mapping(filename),
                "line_number": fi.lineno,
                # Use an empty string when there is no class context so
                # the Ray UI does not render "null:".
                "class": cls_name,
                "method": fi.function,
                "vendor_frame": False,
            }
            payload = CallerPayload(frame)
            return self._send_request([payload])
        return self

    def measure(self, name: str = "default") -> "Ray":
        from time import perf_counter

        now = perf_counter()
        timers = Ray._stopwatches
        if name not in timers:
            timers[name] = (now, now)
            payload = MeasurePayload(name, True, 0.0, 0.0)
        else:
            start, last = timers[name]
            total = now - start
            since_last = now - last
            timers[name] = (start, now)
            payload = MeasurePayload(name, False, total, since_last)
        return self._send_request([payload])

    def stop_time(self, name: str = "") -> "Ray":
        if not name:
            Ray._stopwatches.clear()
            return self
        Ray._stopwatches.pop(name, None)
        return self

    def count(self, name: str | None = None) -> "Ray":
        origin = DefaultOriginFactory().get_origin()
        key = name or origin.fingerprint()
        assert Ray._counters is not None
        times = Ray._counters.increment(key)
        message = "Called "
        if name:
            message += f"`{name}` "
        message += f"{times} "
        message += "time" if times == 1 else "times"
        message += "."
        self.send_custom(message, "Count")
        return self

    def clear_counters(self) -> "Ray":
        assert Ray._counters is not None
        Ray._counters.clear()
        return self

    def counter_value(self, name: str) -> int:
        assert Ray._counters is not None
        return Ray._counters.get(name)

    def pause(self) -> "Ray":
        import hashlib
        from time import time, sleep

        lock_name = hashlib.md5(str(time()).encode("utf-8")).hexdigest()
        payload = CreateLockPayload(lock_name)
        self._send_request([payload])

        assert Ray._client is not None
        while Ray._client.lock_exists(lock_name, Ray._project_name):
            sleep(1)
        return self

    def separator(self) -> "Ray":
        return self._send_request([SeparatorPayload()])

    def show_app(self) -> "Ray":
        return self._send_request([ShowAppPayload()])

    def hide_app(self) -> "Ray":
        return self._send_request([HideAppPayload()])

    def limit(self, count: int) -> "Ray":
        origin = DefaultOriginFactory().get_origin()
        fp = origin.fingerprint()
        self._limit_origin_fingerprint = fp
        if Ray._limiters is not None:
            Ray._limiters.initialize(fp, count)
        return self

    def once(self, *values: Any) -> "Ray":
        origin = DefaultOriginFactory().get_origin()
        fp = origin.fingerprint()
        self._limit_origin_fingerprint = fp
        if Ray._limiters is not None:
            Ray._limiters.initialize(fp, 1)
        if values:
            return self.send(*values)
        return self

    def class_name(self, obj: object) -> "Ray":
        return self.send(obj.__class__.__name__)

    def exception(self, exc: BaseException, meta: Dict[str, Any] | None = None) -> "Ray":
        payload = ExceptionPayload(exc, meta or {})
        self._send_request([payload])
        return self.red()

    def python_exception(self, exc: BaseException) -> "Ray":
        return self.exception(exc)

    def _apply_path_mapping(self, path: str) -> str:
        if self.settings.remote_path and self.settings.local_path and path.startswith(self.settings.remote_path):
            return self.settings.local_path + path[len(self.settings.remote_path) :]
        return path

    # --- conditionals, exception handling, introspection ------------------------

    def if_(self, condition_or_callable: Any, callback: Callable[["Ray"], Any] | None = None) -> "Ray":
        condition = condition_or_callable
        if callable(condition_or_callable):
            condition = bool(condition_or_callable())
        else:
            condition = bool(condition_or_callable)

        if callback is not None and condition:
            callback(self)
        if callback is None:
            self.can_send_payload = condition
        return self

    def show_when(self, condition_or_callable: Any) -> "Ray":
        condition = condition_or_callable
        if callable(condition_or_callable):
            condition = bool(condition_or_callable())
        else:
            condition = bool(condition_or_callable)
        if not condition:
            self.remove()
        return self

    def show_if(self, condition_or_callable: Any) -> "Ray":
        return self.show_when(condition_or_callable)

    def remove_when(self, condition_or_callable: Any) -> "Ray":
        condition = condition_or_callable
        if callable(condition_or_callable):
            condition = bool(condition_or_callable())
        else:
            condition = bool(condition_or_callable)
        if condition:
            self.remove()
        return self

    def remove_if(self, condition_or_callable: Any) -> "Ray":
        return self.remove_when(condition_or_callable)

    def catch(self, callback: Any | None = None) -> "Ray":
        if not Ray._caught_exceptions:
            return self

        exc = Ray._caught_exceptions.pop(0)

        # No callback: just send exception payload
        if callback is None:
            return self.exception(exc)

        # List/tuple of callbacks or classes
        if isinstance(callback, (list, tuple)):
            for cb in callback:
                if not Ray._caught_exceptions:
                    Ray._caught_exceptions.insert(0, exc)
                self.catch(cb)
                if not Ray._caught_exceptions:
                    break
            return self

        # Callable: (exc, ray) -> Any
        if callable(callback):
            result = callback(exc, self)
            return result if isinstance(result, Ray) else self

        # Exception class or class name
        if isinstance(callback, type) and issubclass(callback, BaseException):
            if isinstance(exc, callback):
                return self.exception(exc)
            Ray._caught_exceptions.insert(0, exc)
            return self

        if isinstance(callback, str):
            full_name = f"{exc.__class__.__module__}.{exc.__class__.__name__}"
            if callback in {exc.__class__.__name__, full_name}:
                return self.exception(exc)

        # Fallback: push back and no-op
        Ray._caught_exceptions.insert(0, exc)
        return self

    def throw_exceptions(self) -> "Ray":
        if Ray._caught_exceptions:
            raise Ray._caught_exceptions.pop(0)
        return self

    def invade(self, obj: object) -> Invador:
        return Invador(obj, self)

    def _notify_rate_limit_reached(self) -> None:
        rl = Ray.rate_limiter()
        if rl.is_notified():
            return
        payload = CustomPayload("Rate limit has been reached...", "Rate limit")
        request = Request(self.uuid, [payload], {})
        if Ray._client is not None:
            Ray._client.send(request)
        rl.notify()
