from __future__ import annotations

from .base import Payload
from .text import TextPayload
from .notify import NotifyPayload
from .table import TablePayload
from .custom import CustomPayload
from .log import LogPayload
from .clear_all import ClearAllPayload
from .color import ColorPayload
from .screen_color import ScreenColorPayload
from .size import SizePayload
from .label import LabelPayload
from .hide import HidePayload
from .remove import RemovePayload
from .new_screen import NewScreenPayload
from .separator import SeparatorPayload
from .show_app import ShowAppPayload
from .hide_app import HideAppPayload
from .image import ImagePayload
from .file_contents import FileContentsPayload
from .xml import XmlPayload
from .json_string import JsonStringPayload
from .decoded_json import DecodedJsonPayload
from .confetti import ConfettiPayload
from .create_lock import CreateLockPayload
from .measure import MeasurePayload
from .caller import CallerPayload
from .trace import TracePayload
from .carbon import CarbonPayload
from .python_info import PythonInfoPayload
from .exception import ExceptionPayload
from .bool_payload import BoolPayload
from .null_payload import NullPayload

from datetime import datetime
from typing import Any, Callable, ClassVar, Iterable, List
import json
from .table import _to_primitive


class PayloadFactory:
    _payload_finder: ClassVar[Callable[[Any], Payload | None] | None] = None

    @classmethod
    def register_payload_finder(cls, func: Callable[[Any], Payload | None]) -> None:
        cls._payload_finder = func

    @classmethod
    def create_for_values(cls, values: Iterable[Any]) -> List[Payload]:
        return [cls._get_payload(v) for v in values]

    @classmethod
    def _get_payload(cls, value: Any) -> Payload:
        if cls._payload_finder is not None:
            custom = cls._payload_finder(value)
            if custom is not None:
                return custom

        if isinstance(value, bool):
            return BoolPayload(value)

        if value is None:
            return NullPayload()

        if isinstance(value, datetime):
            return CarbonPayload(value)

        primitive = _to_primitive(value)

        # When logging mappings and sequences, sending them as raw objects makes
        # the Ray / Buggregator UI display them as generic "[object Object]".
        # To keep them readable by default, encode them as a pretty-printed
        # JSON string inside the log payload.
        if isinstance(primitive, (dict, list, tuple, set)):
            try:
                text = json.dumps(primitive, default=str, indent=2, sort_keys=True)
            except TypeError:
                # Fallback to repr if something inside is not JSON-serializable
                text = repr(primitive)
            return LogPayload.from_arguments([text])

        return LogPayload.from_arguments([primitive])

    @staticmethod
    def log(value: Any) -> LogPayload:
        return LogPayload.from_arguments([value])


__all__ = [
    "Payload",
    "TextPayload",
    "NotifyPayload",
    "TablePayload",
    "CustomPayload",
    "LogPayload",
    "ClearAllPayload",
    "ColorPayload",
    "ScreenColorPayload",
    "SizePayload",
    "LabelPayload",
    "HidePayload",
    "RemovePayload",
    "NewScreenPayload",
    "SeparatorPayload",
    "ShowAppPayload",
    "HideAppPayload",
    "ImagePayload",
    "FileContentsPayload",
    "XmlPayload",
    "JsonStringPayload",
    "DecodedJsonPayload",
    "ConfettiPayload",
    "CreateLockPayload",
    "MeasurePayload",
    "CallerPayload",
    "TracePayload",
    "CarbonPayload",
    "PythonInfoPayload",
    "ExceptionPayload",
    "BoolPayload",
    "NullPayload",
    "PayloadFactory",
]
