from __future__ import annotations

from dataclasses import dataclass, field
from typing import Any, Dict, Optional

from ..origin import DefaultOriginFactory, Origin


@dataclass
class Payload:
    remote_path: Optional[str] = None
    local_path: Optional[str] = None

    def get_type(self) -> str:  # pragma: no cover - abstract
        raise NotImplementedError

    def get_content(self) -> Dict[str, Any]:  # pragma: no cover - abstract
        raise NotImplementedError

    def to_dict(self) -> Dict[str, Any]:
        origin = self._get_origin()
        return {
            "type": self.get_type(),
            "content": self.get_content(),
            "origin": origin.to_dict(),
        }

    def _get_origin(self) -> Origin:
        factory = DefaultOriginFactory()
        origin = factory.get_origin()
        if origin.file is not None:
            origin.file = self.replace_remote_path_with_local_path(origin.file)
        return origin

    def replace_remote_path_with_local_path(self, file_path: str) -> str:
        if self.remote_path is None or self.local_path is None:
            return file_path
        if not file_path.startswith(self.remote_path):
            return file_path
        return self.local_path + file_path[len(self.remote_path) :]
