from __future__ import annotations

from typing import Any, Dict
import json

from .base import Payload
from .table import _to_primitive


class DecodedJsonPayload(Payload):
    def __init__(self, value: str) -> None:
        super().__init__()
        self._value = value

    def get_type(self) -> str:
        return "custom"

    def get_content(self) -> Dict[str, Any]:
        try:
            decoded = json.loads(self._value)
        except json.JSONDecodeError:
            decoded = None

        # For objects/arrays, render as pretty-printed JSON text so the Ray
        # UI doesn't show generic "[object Object]" or comma-joined values.
        if isinstance(decoded, (dict, list, tuple, set)):
            json_str = json.dumps(decoded, default=str, indent=2, sort_keys=True)
            content: Any = json_str
        else:
            content = _to_primitive(decoded)

        return {
            "content": content,
            "label": "JSON",
        }
