from __future__ import annotations

from dataclasses import dataclass, field
from pathlib import Path
from typing import Any, Dict, Optional
import json
import os


_DEFAULT_SETTINGS: Dict[str, Any] = {
    "enable": True,
    "host": "localhost",
    "port": 23517,
    "remote_path": None,
    "local_path": None,
    "always_send_raw_values": False,
}


@dataclass
class Settings:
    _settings: Dict[str, Any] = field(default_factory=dict)

    def __post_init__(self) -> None:
        merged = {**_DEFAULT_SETTINGS, **self._settings}
        object.__setattr__(self, "_settings", merged)

    def __getattr__(self, name: str) -> Any:  # pragma: no cover - trivial
        try:
            return self._settings[name]
        except KeyError as exc:  # noqa: B904
            raise AttributeError(name) from exc

    def __setattr__(self, name: str, value: Any) -> None:  # pragma: no cover - trivial
        if name == "_settings":
            object.__setattr__(self, name, value)
        else:
            self._settings[name] = value


class SettingsFactory:
    @staticmethod
    def create_from_dict(settings: Optional[Dict[str, Any]] = None) -> Settings:
        return Settings(settings or {})

    @classmethod
    def create_from_file(cls, config_directory: Optional[str] = None) -> Settings:
        path = cls._search_config_files(config_directory)
        if path is None:
            return Settings({})

        try:
            with path.open("r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception:
            data = {}

        return Settings(data)

    @classmethod
    def _search_config_files(cls, config_directory: Optional[str]) -> Optional[Path]:
        config_names = ["ray.json"]
        cwd = Path(config_directory or os.getcwd())

        current = cwd
        while True:
            if not current.is_dir():
                break
            for name in config_names:
                candidate = current / name
                if candidate.exists():
                    return candidate
            parent = current.parent
            if parent == current:
                break
            current = parent

        return None
