from __future__ import annotations

from typing import Any
import json

from python_ray.payloads import (
    TextPayload,
    JsonStringPayload,
    DecodedJsonPayload,
    LogPayload,
)
from python_ray.payloads.table import _to_primitive


def test_to_primitive_handles_nested_structures():
    data: dict[str, Any] = {
        "a": 1,
        2: {"b": [1, 2, {"c": 3}]},
    }
    result = _to_primitive(data)
    # keys coerced to strings and nested structures preserved
    assert result == {"a": 1, "2": {"b": [1, 2, {"c": 3}]}}


def test_text_payload_escapes_and_formats():
    p = TextPayload("<b>x</b> y\n")
    content = p.get_content()
    assert content["label"] == "Text"
    # HTML-escaped and space/newline formatting
    assert content["content"] == "&lt;b&gt;x&lt;/b&gt;&nbsp;y<br>"


def test_json_string_payload_serializes_any_value():
    p = JsonStringPayload({"a": 1})
    content = p.get_content()
    # JsonStringPayload now produces a custom payload labeled "JSON" with a
    # pretty-printed JSON string.
    assert content["label"] == "JSON"
    data = json.loads(content["content"])
    assert data == {"a": 1}


def test_decoded_json_payload_parses_and_wraps():
    p = DecodedJsonPayload("{\"a\": 1}")
    content = p.get_content()
    assert content["label"] == "JSON"
    data = json.loads(content["content"])
    assert data == {"a": 1}


def test_decoded_json_payload_handles_invalid_json():
    p = DecodedJsonPayload("not-json")
    content = p.get_content()
    # invalid JSON becomes None; _to_primitive passes None through unchanged
    assert content["content"] is None
    # Label is still "JSON" because this is the JSON helper path
    assert content["label"] == "JSON"


def test_log_payload_holds_values_and_clipboard_meta():
    p = LogPayload.from_arguments([{"x": 1}, "abc"])  # type: ignore[list-item]
    content = p.get_content()
    assert content["values"] == [{"x": 1}, "abc"]
    # meta list with a clipboard_data entry derived from raw values
    assert isinstance(content["meta"], list)
    assert content["meta"][0]["clipboard_data"]


def test_payload_path_mapping_helper():
    # Directly exercise replace_remote_path_with_local_path logic
    p = TextPayload("x")
    p.remote_path = "/remote/path"
    p.local_path = "/local/root"

    mapped = p.replace_remote_path_with_local_path("/remote/path/file.py")
    assert mapped == "/local/root/file.py"

    # Non-matching prefix should be returned unchanged
    unmapped = p.replace_remote_path_with_local_path("/something/else.py")
    assert unmapped == "/something/else.py"
