# ReactFlow CSS

ReactFlow CSS is a Python package designed to simplify the integration of popular CSS frameworks like Tailwind CSS and Bootstrap into your ReactPy applications and other HTML projects. It provides a streamlined API for configuring, compiling, and serving CSS, making it easier to manage your styling directly from Python.

## Features

-   **Tailwind CSS Integration**: Easily configure and compile Tailwind CSS within your Python project.
-   **Bootstrap Integration**: Seamlessly include Bootstrap CSS and JavaScript.
-   **ReactPy Compatibility**: Designed to work effortlessly with ReactPy components.
-   **Simplified API**: A `StyleHelper` class to manage both frameworks with a unified interface.

## Installation

To install ReactFlow CSS, you can use pip:

```bash
pip install ReactFlow_CSS
```

## Usage

ReactFlow CSS provides a `StyleHelper` class that acts as a central point for managing both Tailwind CSS and Bootstrap.

### Initializing the StyleHelper

First, import and initialize the `StyleHelper` in your main application file. It requires the path to your main script to resolve relative paths correctly.

```python
from ReactFlow_CSS import Helper

# Initialize the helper in your main application file
# Pass __file__ to ensure correct path resolution
style_helper = Helper(__file__)
```

### Tailwind CSS Integration

#### 1. Configure Tailwind CSS

You can configure Tailwind CSS programmatically using the `tailwind.config()` method. This method accepts a dictionary for configuration or keyword arguments.

```python
# Example: Basic Tailwind CSS configuration
tailwind_config = style_helper.tailwind.config(
    content=["./src/**/*.{js,jsx,ts,tsx,html,py}"],
    theme={
        "extend": {
            "colors": {
                "custom-blue": "#007bff",
            }
        }
    },
    darkMode="class"
)

print(tailwind_config)
# This will print the generated tailwind.config.js content
```

#### 2. Compile Tailwind CSS

After configuring, you can compile your Tailwind CSS. This process generates an `output.css` file based on your configuration and input CSS.

```python
# Example: Compile Tailwind CSS
# Ensure you have an input.css file (e.g., with @tailwind directives)
# and a tailwind.config.js (generated by .config() or manually created)

# You can specify input and output paths
output_css_content = style_helper.tailwind.compile(
    path_config="./tailwind.config.js",  # Path to your tailwind.config.js
    path_index="./input.css",            # Path to your input.css (e.g., containing @tailwind directives)
    path_output="./static/css/tailwind_output.css" # Desired output path
)

# If you don't specify paths, it will use defaults:
# path_config: "./tailwind.config.js"
# path_index: "./input.css"
# path_output: "./output.css"
default_output_css = style_helper.tailwind.compile()

print("Tailwind CSS compiled successfully!")
```

**`input.css` example:**

Create a file named `input.css` (or whatever you specify in `path_index`) with the following content:

```css
@tailwind base;
@tailwind components;
@tailwind utilities;
```

#### 3. Using Tailwind CSS in ReactPy

Once compiled, you can include the generated CSS in your ReactPy application.

```python
from reactpy import html, component, run
from ReactFlow_CSS import Helper

style_helper = Helper(__file__)

# Compile Tailwind CSS (run this once or as part of your build process)
# This will generate static/css/tailwind_output.css
style_helper.tailwind.compile(path_output="./static/css/tailwind_output.css")

@component
def MyTailwindApp():
    return html.div(
        html.link(
            {"rel": "stylesheet", "href": "/static/css/tailwind_output.css"}
        ),
        html.h1(
            {"class_name": "text-3xl font-bold underline text-blue-500"},
            "Hello Tailwind CSS with ReactPy!"
        ),
        html.button(
            {"class_name": "bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded"},
            "Click me"
        )
    )

# To serve static files in ReactPy, you might need a web framework like Flask or FastAPI.
# Example with Flask:
# from flask import Flask, send_from_directory
# from reactpy.backend.flask import make_reactpy_flask_app
#
# app = Flask(__name__)
# make_reactpy_flask_app(app, MyTailwindApp)
#
# @app.route("/static/<path:filename>")
# def static_files(filename):
#     return send_from_directory("./static", filename)
#
# if __name__ == "__main__":
#     app.run(debug=True)
```

### Bootstrap Integration

#### 1. Configure Bootstrap

You can get the default Bootstrap CSS or configure it with additional imports.

```python
# Get the default Bootstrap CSS content
default_bootstrap_css = style_helper.bootstrap.default_css()
print(default_bootstrap_css)

# Configure Bootstrap with additional imports (e.g., custom CSS files)
# The paths in @import statements should be relative to your project root
custom_bootstrap_css = style_helper.bootstrap.config(
    style="@import './my_custom_styles.css';", # Your custom CSS
    output="./static/css/bootstrap_custom.css" # Output path
)
print(custom_bootstrap_css)
```

#### 2. Using Bootstrap in ReactPy

Bootstrap CSS can be directly included using `html.link`. For JavaScript components, you might need to include the Bootstrap JavaScript bundle.

```python
from reactpy import html, component, run
from ReactFlow_CSS import Helper

style_helper = Helper(__file__)

@component
def MyBootstrapApp():
    # Get default Bootstrap CSS
    bootstrap_css_link = style_helper.to_reactpy_links(
        style_helper.bootstrap.default_css()
    )

    return html.div(
        bootstrap_css_link, # Include Bootstrap CSS
        html.h1(
            {"class_name": "text-primary"}, # Bootstrap class
            "Hello Bootstrap with ReactPy!"
        ),
        html.button(
            {"class_name": "btn btn-primary"}, # Bootstrap button class
            "Bootstrap Button"
        ),
        # Include Bootstrap JS (optional, for interactive components like modals, carousels)
        html.script({"src": "https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js", "integrity": "sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz", "crossorigin": "anonymous"})
    )

# Example with Flask (similar to Tailwind CSS setup)
# if __name__ == "__main__":
#     from flask import Flask, send_from_directory
#     from reactpy.backend.flask import make_reactpy_flask_app
#
#     app = Flask(__name__)
#     make_reactpy_flask_app(app, MyBootstrapApp)
#
#     @app.route("/static/<path:filename>")
#     def static_files(filename):
#         return send_from_directory("./static", filename)
#
#     app.run(debug=True)
```

### Converting Imports to ReactPy Links

The `to_reactpy_links` method is useful for converting CSS `@import` rules into `html.link` components, which is particularly helpful for Bootstrap or other CSS files that rely on `@import` for modularity.

```python
from ReactFlow_CSS import Helper

style_helper = Helper(__file__)

# Example CSS string with @import rules
css_with_imports = """
@import 'https://fonts.googleapis.com/css2?family=Roboto&display=swap';
@import './local_styles.css';
"""

# Convert to ReactPy html.link elements
reactpy_links = style_helper.to_reactpy_links(css_with_imports)

# You can then include reactpy_links in your ReactPy component
# html.div(reactpy_links, html.h1("My App"))
```

## Contributing

Contributions are welcome! Please feel free to submit issues or pull requests.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.
