---
title: Configuration
slug: configuration
layout: doc
---

# Configuration Guide

Configuration for the render-engine PostgreSQL plugin is managed through `pyproject.toml` under the `[tool.render-engine.pg]` section.

## Configuration Structure

```toml
[tool.render-engine.pg]
insert_sql = { collection_name = "SQL QUERY1; SQL QUERY2" }
default_table = "default_table_name"
auto_commit = true
```

## Settings Reference

### `read_sql`

**Type:** `dict[str, str]`
**Required:** No (but recommended)
**Default:** `{}`

Defines SQL SELECT queries for reading collection content. These queries are automatically used by `PostgresContentManager` to fetch data from the database. Maps collection names to their SELECT queries.

Generated by the CLI tool from your schema:

```bash
uv run python -m render_engine_pg.cli.sql_cli schema.sql -o config.toml
```

This creates `read_sql` entries like:

```toml
[tool.render-engine.pg.read_sql]
posts = "SELECT posts.id, posts.slug, posts.title, posts.content, posts.author_id, authors.name as author_name FROM posts LEFT JOIN authors ON posts.author_id = authors.id ORDER BY posts.created_at DESC;"

docs = "SELECT docs.id, docs.slug, docs.content FROM docs WHERE docs.published = true ORDER BY docs.created_at DESC;"
```

**Usage in Collections:**

The `PostgresContentManager` automatically looks up `read_sql` using your collection class name (lowercased):

```python
from render_engine import Collection
from render_engine_pg.content_manager import PostgresContentManager
from render_engine_pg.parsers import PGPageParser
from render_engine_pg.connection import get_db_connection

connection = get_db_connection(
    host="localhost",
    database="myblog",
    user="postgres",
    password="secret"
)

@site.collection
class Posts(Collection):
    """Automatically uses read_sql['posts'] from pyproject.toml"""

    ContentManager = PostgresContentManager
    content_manager_extras = {"connection": connection}

    parser = PGPageParser
    routes = ["blog/{slug}/"]
```

The `ContentManager` will:
1. Look up `read_sql['posts']` (lowercased class name)
2. Execute the query against your database
3. Yield a Page object for each result row

### `insert_sql`

**Type:** `dict[str, list[str]]`
**Required:** No
**Default:** `{}`

Contains dependency-ordered INSERT statements for setting up collection-related data. These are generated by the CLI tool from your SQL schema.

Generated by the CLI:

```bash
uv run python -m render_engine_pg.cli.sql_cli schema.sql -o config.toml
```

This creates `insert_sql` entries like:

```toml
[tool.render-engine.pg.insert_sql]
blog = [
    "INSERT INTO tags (name) VALUES ({name}) ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name RETURNING id;",
    "INSERT INTO blog_tags (blog_id, tag_id) VALUES ({blog_id}, {tag_id});",
    "INSERT INTO blog (slug, title, content, date) VALUES ({slug}, {title}, {date});"
]
```

**Get-or-Create Pattern:** For attribute tables with UNIQUE constraints, the CLI generates `ON CONFLICT ... DO UPDATE ... RETURNING id` queries. This safely handles repeated runs:
- If the record doesn't exist → insert and return the new ID
- If the record exists → "update" it (no actual change) and return the existing ID

This allows junction table inserts to reference the correct ID even if the attribute already exists.

#### How It's Generated

The CLI tool:
1. Parses CREATE TABLE statements with `-- @collection` and `-- @attribute` annotations
2. Detects foreign key relationships
3. Orders INSERT statements so dependencies are created first (e.g., tags before blog_tags)
4. Filters out ignored columns (PRIMARY KEY, TIMESTAMP columns marked with `-- ignore`)
5. Groups all related inserts under the collection name

#### Manually Editing insert_sql

After generation, you can edit the queries if needed:

```toml
[tool.render-engine.pg.insert_sql]
blog = [
    "INSERT INTO categories (name) VALUES ('Technology'), ('Travel')",
    "INSERT INTO blog (slug, title, content) VALUES ('my-post', 'My Post', 'Content here')"
]

posts = [
    "INSERT INTO post_metadata (version) VALUES ('1.0')",
    "INSERT INTO posts (title, content) VALUES ('About', 'This is about me')"
]
```

#### Ignoring Columns from INSERT

Use the `-- ignore` comment in your schema:

```sql
-- @collection
CREATE TABLE blog (
    id SERIAL PRIMARY KEY, -- ignore
    slug VARCHAR(255) NOT NULL,
    title VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT NOW() -- ignore
);
```

Or use CLI flags:

```bash
uv run python -m render_engine_pg.cli.sql_cli schema.sql \
  --ignore-pk \
  --ignore-timestamps
```

The ignored columns:
- Are **excluded** from INSERT statements (so database defaults apply)
- Are still **included** in SELECT (read_sql) so they're available in templates

#### Multiple Collections

When your database has multiple collections, each gets its own separate entry in `insert_sql` and `read_sql`. This allows independent management of each collection's data setup and queries.

**Schema with Multiple Collections:**

```sql
-- @collection
CREATE TABLE blog (
    id SERIAL PRIMARY KEY,
    slug VARCHAR(255),
    title VARCHAR(255),
    content TEXT,
    date TIMESTAMP
);

-- @collection
CREATE TABLE microblog (
    id SERIAL PRIMARY KEY,
    slug VARCHAR(255),
    content TEXT,
    external_link VARCHAR(255),
    created_at TIMESTAMP
);

-- @attribute (shared between both collections)
CREATE TABLE tags (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) UNIQUE
);

-- @junction
CREATE TABLE blog_tags (
    blog_id INTEGER REFERENCES blog(id),
    tag_id INTEGER REFERENCES tags(id)
);

-- @junction
CREATE TABLE microblog_tags (
    microblog_id INTEGER REFERENCES microblog(id),
    tag_id INTEGER REFERENCES tags(id)
);
```

**Generated Configuration:**

```toml
[tool.render-engine.pg.insert_sql]
# Blog collection with its own queries
blog = [
    "INSERT INTO tags (name) VALUES ({name}) ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name RETURNING id;",
    "INSERT INTO blog (slug, title, content, date) VALUES ({slug}, {title}, {content}, {date});",
    "INSERT INTO blog_tags (blog_id, tag_id) VALUES ({blog_id}, {tag_id});"
]

# Microblog collection with its own queries
microblog = [
    "INSERT INTO tags (name) VALUES ({name}) ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name RETURNING id;",
    "INSERT INTO microblog (slug, content, external_link, created_at) VALUES ({slug}, {content}, {external_link}, {created_at});",
    "INSERT INTO microblog_tags (microblog_id, tag_id) VALUES ({microblog_id}, {tag_id});"
]

[tool.render-engine.pg.read_sql]
blog = "SELECT blog.id, blog.slug, blog.title, blog.content, blog.date, array_agg(DISTINCT tags.name) as tags_names FROM blog LEFT JOIN blog_tags ON blog.id = blog_tags.blog_id LEFT JOIN tags ON blog_tags.tag_id = tags.id GROUP BY blog.id ORDER BY blog.date DESC;"

microblog = "SELECT microblog.id, microblog.slug, microblog.content, microblog.external_link, microblog.created_at, array_agg(DISTINCT tags.name) as tags_names FROM microblog LEFT JOIN microblog_tags ON microblog.id = microblog_tags.microblog_id LEFT JOIN tags ON microblog_tags.tag_id = tags.id GROUP BY microblog.id ORDER BY microblog.created_at DESC;"
```

**Key Points:**

- **Separate Entries**: Each collection has its own list under `[tool.render-engine.pg.insert_sql]`
- **Shared Attributes**: The `tags` attribute is included in both collections' INSERT lists. The `ON CONFLICT` pattern ensures the same tag isn't duplicated across collections
- **Independent Queries**: Each collection's `read_sql` uses its own SELECT with appropriate JOINs
- **Collection Naming**: The dictionary key (e.g., `blog`, `microblog`) must match the lowercased class name in your Python code

**Python Usage:**

```python
@site.collection
class Blog(Collection):
    """Blog posts with tags"""
    ContentManager = PostgresContentManager
    content_manager_extras = {"connection": connection}
    parser = PGPageParser
    routes = ["blog/{slug}/"]

@site.collection
class Microblog(Collection):
    """Short-form posts with tags"""
    ContentManager = PostgresContentManager
    content_manager_extras = {"connection": connection}
    parser = PGPageParser
    routes = ["micro/{slug}/"]
```

Each collection automatically uses its corresponding configuration from `pyproject.toml`.

## Complete Configuration Example

Here's a typical `pyproject.toml` with render-engine.pg configuration:

```toml
[project]
name = "my-blog"
version = "0.1.0"
description = "A blog powered by render-engine and PostgreSQL"

[tool.render-engine]
# render-engine core configuration

[tool.render-engine.pg]
read_sql = {
    blog = "SELECT blog.id, blog.slug, blog.title, blog.content, blog.date, array_agg(DISTINCT tags.name) as tags_names FROM blog LEFT JOIN blog_tags ON blog.id = blog_tags.blog_id LEFT JOIN tags ON blog_tags.tag_id = tags.id GROUP BY blog.id ORDER BY blog.date DESC;",
    docs = "SELECT id, slug, title, content FROM docs ORDER BY created_at DESC;"
}

insert_sql = {
    blog = [
        "INSERT INTO tags (name) VALUES ('Technology'), ('Travel') ON CONFLICT (name) DO NOTHING;",
        "INSERT INTO blog_tags (blog_id, tag_id) VALUES (1, 1), (1, 2) ON CONFLICT DO NOTHING;",
        "INSERT INTO blog (slug, title, content, date) VALUES ('my-post', 'My Post', '...', NOW());"
    ],
    docs = [
        "INSERT INTO docs (slug, title, content) VALUES ('about', 'About', '...');"
    ]
}
```

## How Settings Are Loaded

The `PostgresContentManager` automatically:

1. Creates a `PGSettings()` instance
2. Looks for `pyproject.toml` in the project root or parent directories
3. Reads the `[tool.render-engine.pg]` section
4. Retrieves `read_sql` for your collection using the lowercased class name

```python
# In your Collection class definition:
@site.collection
class Blog(Collection):
    ContentManager = PostgresContentManager
    # ContentManager will automatically load read_sql['blog']
```

## Best Practices

1. **Use the CLI to Generate Configuration** - Run the CLI tool to generate initial config from your schema:
   ```bash
   uv run python -m render_engine_pg.cli.sql_cli schema.sql -o config.toml
   ```

2. **Annotate Your Schema** - Use consistent annotations:
   ```sql
   -- @collection
   CREATE TABLE blog (...);

   -- @attribute
   CREATE TABLE tags (...);

   -- @junction
   CREATE TABLE blog_tags (...);
   ```

3. **Mark Columns to Ignore** - Use `-- ignore` for auto-generated columns:
   ```sql
   id SERIAL PRIMARY KEY, -- ignore
   created_at TIMESTAMP DEFAULT NOW() -- ignore
   ```

4. **Group Related Collections** - Keep related tables together in your schema file

5. **Collection Naming** - The collection class name (lowercased) must match the key in `read_sql` and `insert_sql`:
   ```python
   class Blog(Collection):  # ← Will look for blog in read_sql/insert_sql
       ...
   ```

## Configuration Scope

The `[tool.render-engine.pg]` section is **plugin-specific** and won't affect other render-engine plugins or tools. Other parts of your `pyproject.toml` remain unchanged.

Next, see [usage examples](./03-usage.md).
