from .cal_tools import calculate_crc,verify_crc
from bleak import BleakClient
from bleak.exc import BleakError
from bleak import BleakScanner
import asyncio
from typing import Optional
from collections import deque

class modbus_comm:
    def __init__(self,write_service_uuid: str = "0000ffd0-0000-1000-8000-00805f9b34fb",
        write_char_uuid: str = "0000ffd1-0000-1000-8000-00805f9b34fb",
        notify_service_uuid: str = "0000fff0-0000-1000-8000-00805f9b34fb",
        notify_char_uuid: str = "0000fff1-0000-1000-8000-00805f9b34fb",
        slave_address: int = 1,
        timeout: float = 3.0) -> None:
        self.write_service_uuid = write_service_uuid
        self.write_char_uuid = write_char_uuid
        self.notify_service_uuid = notify_service_uuid
        self.notify_char_uuid = notify_char_uuid
        self.slave_address = slave_address
        self.timeout = timeout
        
        self.client: Optional[BleakClient] = None
        self.response_queue = deque()
        self.lock = asyncio.Lock()
        self.expected_length = 0
        # write characteristic cache
        self.write_char = None  
        # notify characteristic cache
        self.notify_char = None  

    async def scan_devices(self):
        """Scan and return a list of eligible BLE devices"""
        print("Scanning BLE devices (for 5 seconds)...")
        # device storage array
        self.valid_devices = []  
        try:
            devices = await BleakScanner.discover(timeout=5)
            
            for d in devices:
                if d.name and (d.name.startswith('RNG') or d.name.startswith('BT')):
                    print(f"found eligible BLE devices: {d.name} | address: {d.address} | rssi: {d.rssi}")
                    self.valid_devices.append({  
                        'name': d.name,
                        'address': d.address
                    })
                else:
                    print(f"Filter out non-target device: {d.name} | address: {d.address}")
            return self.valid_devices 
        except BleakError as e:
            print(str(e))
            return []

    async def connect(self, device_address:str) -> bool:
        """Connection method that returns connection status"""
        try:
            self.client = BleakClient(device_address)
            # Connection with timeout
            await asyncio.wait_for(self.client.connect(), timeout=10.0)
            
            # Get the write service and correct characteristic
            write_service = self.client.services.get_service(self.write_service_uuid)
            self.write_char = write_service.get_characteristic(self.write_char_uuid)
            
            # Get the notify service and find the correct characteristic
            service = self.client.services.get_service(self.notify_service_uuid)
            self.notify_char = service.get_characteristic(self.notify_char_uuid)

            # Enable notification subscription (using the characteristic handle)
            await self.client.start_notify(
                self.notify_char.handle,  # Use the characteristic handle instead of UUID
                self._notification_handler
            )
            print("Connected and notification enabled")
            
            # Enhanced characteristic property check
            if not self.write_char:
                raise RuntimeError(f"not found write characterister {self.write_char_uuid}")
            if not self.write_char.properties:
                raise RuntimeError("write characterister not defined")
                
            return True
            
        except BleakError as e:
            print(f"connection failture: {str(e)}")
            return False
        except asyncio.TimeoutError:
            print("connection timeout")
            return False
        except Exception as e:
            print(f"unknown connection error: {str(e)}")
            return False

    async def disconnect(self):
        if self.client and self.client.is_connected:
            # Use the characteristic handle obtained during the previous connection
            if hasattr(self, 'notify_char'):
                await self.client.stop_notify(self.notify_char.handle)
            await self.client.disconnect()
            print("Disconnected")

    def _notification_handler(self,sender,data:bytearray):
        """Callback function to handle BLE notification data"""
        print(f"收到原始响应: {data.hex()}")  # 新增原始数据日志
        try:
            # Check address 
            if data[0] != self.slave_address:
                return
            # CRC verify
            if not verify_crc(data):
                print("crc verify failture,dropped the data")
                return
            self.response_queue.append(data)
        except BleakError as e:
            print(e)

    async def _send_request(self,pdu_data:bytes,flag:bool=False) -> bytes:
        """Send request and wait for response"""
        # Construct complete frame command: address PDU command CRC
        async with self.lock:
            frame  = (self.slave_address.to_bytes(1,'big')+pdu_data)
            frame += calculate_crc(frame)
            # Clear old responses
            self.response_queue.clear()

            # Check characteristic properties
            if self.write_char is None:
                raise RuntimeError("Write characteristic not initialized")
                
            # Select write method based on characteristic properties
            if 'write-without-response' in self.write_char.properties:
                write_method = self.client.write_gatt_char
                response_flag = False
            elif 'write' in self.write_char.properties:
                write_method = self.client.write_gatt_char
                response_flag = True
            else:
                raise RuntimeError("Characteristic does not support write operation")
            
            try:
                await write_method(
                    self.write_char.handle,
                    frame,
                    response=response_flag
                )
            except BleakError as e:
                print(f"write failed: {str(e)}")
                raise
                
            print(f"send data: {frame.hex()}")

            # wait response, write mode return directly
            if flag:
                return True
            try:
                return await asyncio.wait_for(
                    self._wait_for_response(),
                    timeout=self.timeout
                )
            except asyncio.TimeoutError:
                print(f"超时发生时响应队列内容: {self.response_queue}")
                raise

    async def _wait_for_response(self) -> bytes:
        """Get response data from the notification queue"""
        while True:
            if self.response_queue:
                response = self.response_queue.popleft()
                return response
            await asyncio.sleep(0.1)
    
    async def read_holding_registers(
        self,
        start_address: int,
        register_count: int
    ) -> bytes:
        """Read holding registers optimized implementation (returns raw byte data)"""
        pdu = bytes([
            0x03,  # Function code
            (start_address >> 8) & 0xFF,
            start_address & 0xFF,
            (register_count >> 8) & 0xFF,
            register_count & 0xFF
        ])
        response = await self._send_request(pdu)
        
        # Parse response
        # if response[1] != 0x03:
        #     raise ValueError(f"Invalid function code response: {response[1]:02x}")
        
        # Directly return the data part of the bytes (excluding address, function code, byte count, and CRC)
        # Response structure: [address][function code][byte count][data...][CRC]
        return bytes(response[3:-2])  

    async def write_single_register(
        self,
        register_address: int,
        value: int
    ) -> bool:
        """write single register"""
        pdu = bytes([
            0x06,  # function code
            (register_address >> 8) & 0xFF,
            register_address & 0xFF,
            (value >> 8) & 0xFF,
            value & 0xFF
        ])
        try:
            return await self._send_request(pdu,True)
            # Verify response frame structure
        except Exception as e:
            print(f"Write failed: {str(e)}")
            return False

class EnhancedModbusClient(modbus_comm):
    """Enhanced client with automatic reconnection mechanism"""
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._reconnect_flag = False
        
    async def safe_send(self, method, *args, **kwargs):
        try:
            return await method(*args, **kwargs)
        except BleakError as e:
            print(f"communication error: {str(e)},try reconnect...")
            await self._reconnect()
            return await method(*args, **kwargs)
            
    async def _reconnect(self):
        if self.client and self.client.is_connected:
            await self.disconnect()
        await self.connect(self.client.address)
