"""
Example usage of the X.com API Client

This demonstrates various ways to use the client for posting tweets.
"""

from api_client import XClient
import time

# Method 1: Direct credential usage
def example_basic_post():
    """Basic example of posting a tweet"""
    print("Example 1: Basic Tweet Post")
    print("-" * 60)

    # Replace these with your actual credentials
    AUTH_TOKEN = "your_auth_token_here"
    CT0 = "your_ct0_token_here"

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    result = client.post_tweet("Hello from Python! 👋")

    print(f"✓ Tweet posted successfully!")
    print(f"  Tweet ID: {result['tweet_id']}")
    print(f"  URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
    print()


# Method 2: Using config file
def example_with_config():
    """Example using a config file"""
    print("Example 2: Using Config File")
    print("-" * 60)

    try:
        from config import AUTH_TOKEN, CT0
    except ImportError:
        print("⚠ Config file not found. Copy config.example.py to config.py")
        return

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    result = client.post_tweet("Posted using config file! 📝")

    print(f"✓ Tweet posted!")
    print(f"  URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
    print()


# Method 3: Interactive posting
def example_interactive():
    """Interactive example - prompts for tweet content"""
    print("Example 3: Interactive Tweet Posting")
    print("-" * 60)

    try:
        from config import AUTH_TOKEN, CT0
    except ImportError:
        AUTH_TOKEN = input("Enter your auth_token: ")
        CT0 = input("Enter your ct0: ")

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    tweet_text = input("What would you like to tweet? ")

    if not tweet_text:
        print("⚠ No tweet text provided")
        return

    try:
        result = client.post_tweet(tweet_text)
        print(f"\n✓ Tweet posted successfully!")
        print(f"  URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
    except Exception as e:
        print(f"\n✗ Error posting tweet: {e}")

    print()


# Method 4: Posting multiple tweets with delay
def example_thread():
    """Example of posting multiple tweets (like a thread)"""
    print("Example 4: Posting Multiple Tweets")
    print("-" * 60)

    try:
        from config import AUTH_TOKEN, CT0
    except ImportError:
        print("⚠ Config file not found. Copy config.example.py to config.py")
        return

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    tweets = [
        "🧵 Thread about reverse engineering APIs (1/3)",
        "Analyzing HAR files can reveal API endpoints, authentication methods, and request patterns (2/3)",
        "With this knowledge, you can automate interactions with web services! (3/3)"
    ]

    for i, tweet_text in enumerate(tweets, 1):
        try:
            result = client.post_tweet(tweet_text)
            print(f"✓ Tweet {i}/{len(tweets)} posted")
            print(f"  {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")

            # Wait between tweets to avoid rate limiting
            if i < len(tweets):
                print(f"  Waiting 5 seconds before next tweet...")
                time.sleep(5)
        except Exception as e:
            print(f"✗ Error posting tweet {i}: {e}")
            break

    print()


# Method 5: Error handling example
def example_error_handling():
    """Example with comprehensive error handling"""
    print("Example 5: Error Handling")
    print("-" * 60)

    try:
        from config import AUTH_TOKEN, CT0
    except ImportError:
        print("⚠ Config file not found. Copy config.example.py to config.py")
        return

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    # Try to post a tweet that's too long
    long_text = "a" * 300  # Exceeds 280 character limit

    try:
        result = client.post_tweet(long_text)
        print(f"✓ Tweet posted (shouldn't happen)")
    except ValueError as e:
        print(f"✓ Caught expected error: {e}")
    except Exception as e:
        print(f"✗ Unexpected error: {e}")

    print()


# Method 6: Using environment variables
def example_env_vars():
    """Example using environment variables for credentials"""
    print("Example 6: Using Environment Variables")
    print("-" * 60)

    import os

    AUTH_TOKEN = os.getenv('X_AUTH_TOKEN')
    CT0 = os.getenv('X_CT0')

    if not AUTH_TOKEN or not CT0:
        print("⚠ Environment variables not set")
        print("  Set X_AUTH_TOKEN and X_CT0 environment variables")
        print("  Example: export X_AUTH_TOKEN='your_token'")
        return

    client = XClient(auth_token=AUTH_TOKEN, ct0=CT0)

    result = client.post_tweet("Posted using environment variables! 🌍")

    print(f"✓ Tweet posted!")
    print(f"  URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
    print()


def main():
    """Run all examples (commented out to prevent accidental posting)"""
    print()
    print("=" * 60)
    print("X.com API Client - Example Usage")
    print("=" * 60)
    print()

    # Uncomment the examples you want to run:

    # example_basic_post()
    # example_with_config()
    example_interactive()
    # example_thread()
    # example_error_handling()
    # example_env_vars()

    print("=" * 60)
    print("Examples completed!")
    print("=" * 60)


if __name__ == "__main__":
    main()
