# X.com (Twitter) API Client - Reverse Engineered

This is a production-ready Python client for posting tweets to X.com, reverse-engineered from HAR file analysis.

## Overview

This client uses X.com's internal GraphQL API to post tweets programmatically. It was created by analyzing network traffic from the X.com web application.

## APIs Discovered

### 1. CreateTweet API

**Endpoint:** `https://x.com/i/api/graphql/Uf3io9zVp1DsYxrmL5FJ7g/CreateTweet`

**Method:** POST

**Description:** Creates a new tweet on X.com

**Authentication Required:**
- Bearer Token (static, embedded in web app)
- CSRF Token (ct0 cookie)
- Auth Token (session cookie)

**Request Structure:**

```json
{
  "variables": {
    "tweet_text": "Your tweet text here",
    "dark_request": false,
    "media": {
      "media_entities": [],
      "possibly_sensitive": false
    },
    "semantic_annotation_ids": [],
    "disallowed_reply_options": null
  },
  "features": { ... },
  "queryId": "Uf3io9zVp1DsYxrmL5FJ7g"
}
```

**Response Structure:**

```json
{
  "data": {
    "create_tweet": {
      "tweet_results": {
        "result": {
          "rest_id": "tweet_id_here",
          "core": {
            "user_results": {
              "result": {
                "core": {
                  "screen_name": "username",
                  "name": "Display Name"
                }
              }
            }
          }
        }
      }
    }
  }
}
```

## Authentication

The client requires two pieces of authentication data from your X.com session:

1. **auth_token** - Your session authentication token
2. **ct0** - CSRF token

### How to Get Your Credentials

1. Log in to [x.com](https://x.com) in your browser
2. Open Developer Tools (F12)
3. Go to Application/Storage → Cookies → https://x.com
4. Copy the values of:
   - `auth_token` cookie
   - `ct0` cookie

**Security Warning:** These tokens give full access to your X.com account. Keep them secure and never share them publicly.

## Installation

```bash
# Install required dependencies
pip install requests
```

## Usage

### Basic Usage

```python
from api_client import XClient

# Initialize the client
client = XClient(
    auth_token="your_auth_token_here",
    ct0="your_ct0_token_here"
)

# Post a tweet
result = client.post_tweet("Hello, World!")

print(f"Tweet posted! ID: {result['tweet_id']}")
print(f"URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
```

### Command Line Usage

```bash
python api_client.py "your_auth_token" "your_ct0" "Your tweet text here"
```

### Advanced Usage

```python
from api_client import XClient

client = XClient(auth_token="...", ct0="...")

# Post a simple tweet
result = client.post_tweet("My first programmatic tweet!")

# Reply to a tweet
reply_result = client.post_tweet(
    "This is a reply",
    reply_to_tweet_id="1234567890"
)

# Get tweet URL
url = client.get_tweet_url(
    tweet_id=result['tweet_id'],
    username=result['user']['screen_name']
)
print(f"Tweet URL: {url}")
```

## API Client Features

- ✅ **Production-Ready**: Error handling, validation, and proper authentication
- ✅ **Type Hints**: Full type annotations for better IDE support
- ✅ **Well-Documented**: Comprehensive docstrings and comments
- ✅ **Browser Mimicking**: Proper headers and request structure to avoid detection
- ✅ **Session Management**: Persistent session with proper cookie handling
- ✅ **Transaction IDs**: Generates proper client transaction IDs
- ✅ **Error Handling**: Custom exceptions for different error types

## Important Headers

The following headers are critical for the API to work:

| Header | Description |
|--------|-------------|
| `authorization` | Bearer token (static, extracted from web app) |
| `x-csrf-token` | CSRF token (same as ct0 cookie) |
| `x-twitter-active-user` | Must be "yes" |
| `x-client-transaction-id` | Unique transaction ID for each request |
| `content-type` | application/json |
| `origin` | https://x.com |
| `referer` | https://x.com/home |

## Rate Limits

X.com enforces rate limits on API requests. Be mindful of:
- Tweet posting frequency
- Account-specific limits
- IP-based rate limiting

## Bot Detection

X.com has bot detection mechanisms. This client includes several mitigations:
- Proper User-Agent headers
- Browser-like request patterns
- Session persistence
- Transaction ID generation
- CSRF token handling

## Limitations

- **No media upload support yet**: Currently supports text-only tweets
- **Session expiration**: Cookies expire after some time, requiring re-authentication
- **Rate limits**: Subject to X.com's rate limiting policies
- **Account requirements**: May require phone verification or other account features

## Future Enhancements

Potential improvements for this client:

- [ ] Media upload support (images, videos, GIFs)
- [ ] Thread posting (tweet chains)
- [ ] Quote tweets
- [ ] Retweet functionality
- [ ] Delete tweet functionality
- [ ] Automatic session refresh
- [ ] Proxy support for IP rotation
- [ ] Retry logic with exponential backoff

## Troubleshooting

### "401 Unauthorized" Error
- Your cookies have expired, get fresh ones from the browser
- Make sure both auth_token and ct0 are correct

### "403 Forbidden" Error
- CSRF token mismatch, ensure ct0 cookie matches x-csrf-token header
- Account may be locked or restricted

### "429 Too Many Requests" Error
- You've hit rate limits, wait before trying again
- Consider implementing rate limiting in your code

### Bot Detection / Captcha
- The account may be flagged for suspicious activity
- Try using the account normally in a browser first
- Consider using Playwright with CDP for more advanced bot detection bypass

## Security Considerations

- **Never commit credentials**: Don't hardcode tokens in your code
- **Use environment variables**: Store credentials in .env files
- **Rotate credentials**: Get fresh tokens regularly
- **Monitor usage**: Watch for unusual activity on your account
- **Respect Terms of Service**: Use responsibly and within X.com's terms

## Disclaimer

This client is for educational purposes and personal automation. Use at your own risk. The author is not responsible for any misuse or violations of X.com's Terms of Service.

## License

MIT License - Feel free to use and modify as needed.
