"""
X.com (Twitter) API Client - Reverse Engineered

This module provides a production-ready Python client for posting tweets to X.com
using the internal GraphQL API. It was reverse-engineered from HAR file analysis.

Authentication:
    - Requires auth_token cookie (session token)
    - Requires ct0 token (CSRF token)
    - Uses Bearer token for authorization

Usage:
    client = XClient(auth_token="your_auth_token", ct0="your_ct0_token")
    result = client.post_tweet("Hello, World!")
    print(f"Tweet posted! ID: {result['tweet_id']}")
"""

import json
import secrets
import base64
from typing import Dict, Any, Optional, List
import requests
from dataclasses import dataclass


@dataclass
class XCredentials:
    """Credentials required for X.com API authentication"""
    auth_token: str
    ct0: str  # CSRF token


class XClient:
    """
    Production-ready client for X.com API interactions.

    This client handles tweet creation using X.com's internal GraphQL API.
    It includes proper authentication, error handling, and bot detection mitigation.
    """

    # Static bearer token extracted from X.com web app
    BEARER_TOKEN = "AAAAAAAAAAAAAAAAAAAAANRILgAAAAAAnNwIzUejRCOuH5E6I8xnZz4puTs%3D1Zv7ttfk8LF81IUq16cHjhLTvJu4FA33AGWWjCpTnA"

    # GraphQL endpoint for creating tweets
    CREATE_TWEET_ENDPOINT = "https://x.com/i/api/graphql/Uf3io9zVp1DsYxrmL5FJ7g/CreateTweet"

    def __init__(self, auth_token: str, ct0: str, user_id: Optional[str] = None):
        """
        Initialize the X.com API client.

        Args:
            auth_token: The auth_token cookie value from an authenticated X.com session
            ct0: The ct0 cookie value (CSRF token) from the same session
            user_id: Optional user ID (will be extracted from cookies if not provided)
        """
        self.credentials = XCredentials(auth_token=auth_token, ct0=ct0)
        self.user_id = user_id
        self.session = self._create_session()

    def _create_session(self) -> requests.Session:
        """
        Create a requests session with proper headers and cookies.

        Returns:
            Configured requests.Session object
        """
        session = requests.Session()

        # Set cookies
        session.cookies.set('auth_token', self.credentials.auth_token, domain='.x.com')
        session.cookies.set('ct0', self.credentials.ct0, domain='.x.com')

        # Set default headers to mimic browser
        session.headers.update({
            'authorization': f'Bearer {self.BEARER_TOKEN}',
            'content-type': 'application/json',
            'origin': 'https://x.com',
            'referer': 'https://x.com/home',
            'user-agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/143.0.0.0 Safari/537.36',
            'x-csrf-token': self.credentials.ct0,
            'x-twitter-active-user': 'yes',
            'accept': '*/*',
            'accept-encoding': 'gzip, deflate, br, zstd',
            'accept-language': 'en-US,en;q=0.9',
            'sec-ch-ua': '"Google Chrome";v="143", "Chromium";v="143", "Not A(Brand";v="24"',
            'sec-ch-ua-mobile': '?0',
            'sec-ch-ua-platform': '"macOS"',
            'sec-fetch-dest': 'empty',
            'sec-fetch-mode': 'cors',
            'sec-fetch-site': 'same-origin',
        })

        return session

    def _generate_client_transaction_id(self) -> str:
        """
        Generate a client transaction ID for the request.

        X.com uses this to track client-side transactions. We generate a random
        base64-encoded string to mimic the browser behavior.

        Returns:
            Base64-encoded transaction ID
        """
        random_bytes = secrets.token_bytes(64)
        return base64.b64encode(random_bytes).decode('utf-8')

    def _build_tweet_payload(
        self,
        text: str,
        media_ids: Optional[List[str]] = None,
        reply_to_tweet_id: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Build the GraphQL payload for creating a tweet.

        Args:
            text: The text content of the tweet
            media_ids: Optional list of media IDs to attach
            reply_to_tweet_id: Optional tweet ID to reply to

        Returns:
            Dictionary containing the GraphQL payload
        """
        payload = {
            "variables": {
                "tweet_text": text,
                "dark_request": False,
                "media": {
                    "media_entities": [],
                    "possibly_sensitive": False
                },
                "semantic_annotation_ids": [],
                "disallowed_reply_options": None
            },
            "features": {
                "premium_content_api_read_enabled": False,
                "communities_web_enable_tweet_community_results_fetch": True,
                "c9s_tweet_anatomy_moderator_badge_enabled": True,
                "responsive_web_grok_analyze_button_fetch_trends_enabled": False,
                "responsive_web_grok_analyze_post_followups_enabled": True,
                "responsive_web_jetfuel_frame": True,
                "responsive_web_grok_share_attachment_enabled": True,
                "responsive_web_edit_tweet_api_enabled": True,
                "graphql_is_translatable_rweb_tweet_is_translatable_enabled": True,
                "view_counts_everywhere_api_enabled": True,
                "longform_notetweets_consumption_enabled": True,
                "responsive_web_twitter_article_tweet_consumption_enabled": True,
                "tweet_awards_web_tipping_enabled": False,
                "responsive_web_grok_show_grok_translated_post": False,
                "responsive_web_grok_analysis_button_from_backend": True,
                "creator_subscriptions_quote_tweet_preview_enabled": False,
                "longform_notetweets_rich_text_read_enabled": True,
                "longform_notetweets_inline_media_enabled": True,
                "profile_label_improvements_pcf_label_in_post_enabled": True,
                "responsive_web_profile_redirect_enabled": False,
                "rweb_tipjar_consumption_enabled": True,
                "verified_phone_label_enabled": False,
                "articles_preview_enabled": True,
                "responsive_web_grok_community_note_auto_translation_is_enabled": False,
                "responsive_web_graphql_skip_user_profile_image_extensions_enabled": False,
                "freedom_of_speech_not_reach_fetch_enabled": True,
                "standardized_nudges_misinfo": True,
                "tweet_with_visibility_results_prefer_gql_limited_actions_policy_enabled": True,
                "responsive_web_grok_image_annotation_enabled": True,
                "responsive_web_grok_imagine_annotation_enabled": True,
                "responsive_web_graphql_timeline_navigation_enabled": True,
                "responsive_web_enhance_cards_enabled": False
            },
            "queryId": "Uf3io9zVp1DsYxrmL5FJ7g"
        }

        # Add media if provided
        if media_ids:
            payload["variables"]["media"]["media_entities"] = [
                {"media_id": media_id, "tagged_users": []}
                for media_id in media_ids
            ]

        # Add reply information if provided
        if reply_to_tweet_id:
            payload["variables"]["reply"] = {
                "in_reply_to_tweet_id": reply_to_tweet_id,
                "exclude_reply_user_ids": []
            }

        return payload

    def post_tweet(
        self,
        text: str,
        media_ids: Optional[List[str]] = None,
        reply_to_tweet_id: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Post a tweet to X.com.

        Args:
            text: The text content of the tweet (max 280 characters for regular users)
            media_ids: Optional list of media IDs to attach (use upload_media first)
            reply_to_tweet_id: Optional tweet ID to reply to

        Returns:
            Dictionary containing tweet information including:
                - tweet_id: The ID of the created tweet
                - rest_id: Same as tweet_id
                - user: Information about the posting user

        Raises:
            requests.exceptions.HTTPError: If the API request fails
            ValueError: If the response format is unexpected
        """
        # Validate tweet length
        if len(text) > 280:
            raise ValueError(f"Tweet text too long: {len(text)} characters (max 280)")

        # Build payload
        payload = self._build_tweet_payload(text, media_ids, reply_to_tweet_id)

        # Generate transaction ID
        transaction_id = self._generate_client_transaction_id()

        # Add transaction ID to headers for this request
        headers = {'x-client-transaction-id': transaction_id}

        # Make the request
        response = self.session.post(
            self.CREATE_TWEET_ENDPOINT,
            json=payload,
            headers=headers,
            timeout=30
        )

        # Check for errors
        response.raise_for_status()

        # Parse response
        data = response.json()

        # Debug: Print response to see what we're getting
        print(f"\n[DEBUG] Response status: {response.status_code}")
        print(f"[DEBUG] Response data: {json.dumps(data, indent=2)}\n")

        # Check for API-level errors (X.com returns 200 OK even for some errors)
        if 'errors' in data:
            error_messages = [error.get('message', str(error)) for error in data['errors']]
            raise ValueError(f"X.com API returned errors: {'; '.join(error_messages)}")

        # Extract tweet information
        if 'data' not in data or 'create_tweet' not in data['data']:
            raise ValueError(f"Unexpected response format: {data}")

        tweet_result = data['data']['create_tweet']['tweet_results']['result']

        return {
            'tweet_id': tweet_result['rest_id'],
            'rest_id': tweet_result['rest_id'],
            'user': tweet_result['core']['user_results']['result']['core'],
            'edit_control': tweet_result.get('edit_control', {}),
            'raw_response': data
        }

    def get_tweet_url(self, tweet_id: str, username: Optional[str] = None) -> str:
        """
        Generate the URL for a tweet.

        Args:
            tweet_id: The ID of the tweet
            username: Optional username (will use a generic URL if not provided)

        Returns:
            Full URL to the tweet
        """
        if username:
            return f"https://x.com/{username}/status/{tweet_id}"
        return f"https://x.com/i/web/status/{tweet_id}"


class XAPIError(Exception):
    """Base exception for X.com API errors"""
    pass


class AuthenticationError(XAPIError):
    """Raised when authentication fails"""
    pass


class RateLimitError(XAPIError):
    """Raised when rate limit is exceeded"""
    pass


def extract_credentials_from_cookies(cookie_string: str) -> XCredentials:
    """
    Extract X.com credentials from a cookie string.

    Args:
        cookie_string: Cookie string in the format "name=value; name2=value2; ..."

    Returns:
        XCredentials object

    Raises:
        ValueError: If required cookies are not found
    """
    cookies = {}
    for cookie in cookie_string.split('; '):
        if '=' in cookie:
            name, value = cookie.split('=', 1)
            cookies[name] = value

    if 'auth_token' not in cookies:
        raise ValueError("auth_token cookie not found")
    if 'ct0' not in cookies:
        raise ValueError("ct0 cookie not found")

    return XCredentials(
        auth_token=cookies['auth_token'],
        ct0=cookies['ct0']
    )


# Example usage and testing
if __name__ == "__main__":
    import sys

    print("X.com API Client - Tweet Poster")
    print("=" * 60)
    print()

    # Check if credentials are provided
    if len(sys.argv) < 3:
        print("Usage: python api_client.py <auth_token> <ct0> [tweet_text]")
        print()
        print("To get your credentials:")
        print("1. Log in to x.com in your browser")
        print("2. Open Developer Tools (F12)")
        print("3. Go to Application/Storage > Cookies > https://x.com")
        print("4. Copy the values of 'auth_token' and 'ct0' cookies")
        print()
        print("Example:")
        print('  python api_client.py "your_auth_token" "your_ct0" "Hello from Python!"')
        sys.exit(1)

    auth_token = sys.argv[1]
    ct0 = sys.argv[2]
    tweet_text = sys.argv[3] if len(sys.argv) > 3 else "Test tweet from Python API client!"

    try:
        # Create client
        print(f"Initializing client...")
        client = XClient(auth_token=auth_token, ct0=ct0)

        # Post tweet
        print(f"Posting tweet: '{tweet_text}'")
        result = client.post_tweet(tweet_text)

        # Display result
        print()
        print("SUCCESS! Tweet posted!")
        print("=" * 60)
        print(f"Tweet ID: {result['tweet_id']}")
        print(f"Posted by: @{result['user']['screen_name']} ({result['user']['name']})")
        print(f"URL: {client.get_tweet_url(result['tweet_id'], result['user']['screen_name'])}")
        print()

    except requests.exceptions.HTTPError as e:
        print(f"\nERROR: HTTP request failed: {e}")
        if e.response is not None:
            print(f"Status code: {e.response.status_code}")
            print(f"Response: {e.response.text[:500]}")
        sys.exit(1)
    except Exception as e:
        print(f"\nERROR: {type(e).__name__}: {e}")
        sys.exit(1)
