#!/usr/bin/env python3
"""
Example usage scripts for Stapply.ai Jobs API Client
"""

from api_client import StapplyAPIClient, Job
from typing import List
import json


def example_1_get_all_jobs():
    """Example 1: Fetch and display all jobs"""
    print("=" * 80)
    print("Example 1: Get All Jobs")
    print("=" * 80)

    with StapplyAPIClient(headless=True) as client:
        jobs = client.get_jobs()

        print(f"\nFound {len(jobs)} jobs:\n")

        for i, job in enumerate(jobs, 1):
            print(f"{i}. {job.title}")
            print(f"   Company: {job.company}")
            print(f"   URL: {job.url}")
            print()


def example_2_click_first_job():
    """Example 2: Replicate the original HAR file action"""
    print("=" * 80)
    print("Example 2: Click First Job (Original HAR Action)")
    print("=" * 80)

    with StapplyAPIClient(headless=True) as client:
        job_details = client.click_first_job()

        print("\nJob Details:")
        print(f"  Title: {job_details['title']}")
        print(f"  Company: {job_details['company']}")
        print(f"  URL: {job_details['url']}")

        if job_details['external_apply_url']:
            print(f"  Apply URL: {job_details['external_apply_url']}")

        if job_details['description']:
            desc = job_details['description'][:200] + "..." if len(job_details['description']) > 200 else job_details['description']
            print(f"  Description: {desc}")


def example_3_get_specific_job():
    """Example 3: Get details for a specific job"""
    print("=" * 80)
    print("Example 3: Get Specific Job Details")
    print("=" * 80)

    with StapplyAPIClient(headless=True) as client:
        # First get all jobs
        jobs = client.get_jobs()

        if jobs:
            # Get details for the first job
            job = jobs[0]
            print(f"\nFetching details for: {job.title}")

            details = client.get_job_details(job.relative_url)

            print(f"\nDetailed Information:")
            print(f"  Title: {details['title']}")
            print(f"  Company: {details['company']}")
            print(f"  Location: {details['location']}")
            print(f"  Apply URL: {details['external_apply_url']}")
            print(f"  Scraped At: {details['scraped_at']}")


def example_4_filter_by_company():
    """Example 4: Filter jobs by company"""
    print("=" * 80)
    print("Example 4: Filter Jobs by Company")
    print("=" * 80)

    target_company = "vercel"

    with StapplyAPIClient(headless=True) as client:
        all_jobs = client.get_jobs()

        # Filter jobs
        company_jobs = [job for job in all_jobs if job.company.lower() == target_company.lower()]

        print(f"\nFound {len(company_jobs)} jobs at {target_company}:\n")

        for job in company_jobs:
            print(f"  • {job.title}")
            print(f"    {job.url}")
            print()


def example_5_export_to_json():
    """Example 5: Export all jobs to JSON"""
    print("=" * 80)
    print("Example 5: Export Jobs to JSON")
    print("=" * 80)

    output_file = "my_jobs_export.json"

    with StapplyAPIClient(headless=True) as client:
        jobs = client.get_jobs()

        # Save using built-in method
        client.save_jobs_to_json(jobs, output_file)

        print(f"\nExported {len(jobs)} jobs to: {output_file}")

        # Read and display
        with open(output_file, 'r') as f:
            data = json.load(f)
            print(f"\nFirst job in export:")
            print(json.dumps(data[0], indent=2))


def example_6_visible_browser():
    """Example 6: Run with visible browser (for debugging)"""
    print("=" * 80)
    print("Example 6: Run with Visible Browser")
    print("=" * 80)

    # Set headless=False to see the browser
    with StapplyAPIClient(headless=False) as client:
        jobs = client.get_jobs(wait_time=3000)  # Wait longer to observe

        print(f"\nFetched {len(jobs)} jobs")
        print("You should have seen the browser window open!")


def example_7_manual_session():
    """Example 7: Manual session management"""
    print("=" * 80)
    print("Example 7: Manual Session Management")
    print("=" * 80)

    client = StapplyAPIClient(headless=True, timeout=60000)

    try:
        # Start session
        client.start()
        print("Browser session started")

        # Get jobs
        jobs = client.get_jobs()
        print(f"Found {len(jobs)} jobs")

        # Get details for multiple jobs
        for job in jobs[:3]:  # First 3 jobs
            details = client.get_job_details(job.relative_url)
            print(f"  - {details['title']} at {details['company']}")

    finally:
        # Always clean up
        client.stop()
        print("Browser session stopped")


if __name__ == '__main__':
    import sys

    examples = {
        '1': ('Get all jobs', example_1_get_all_jobs),
        '2': ('Click first job (HAR action)', example_2_click_first_job),
        '3': ('Get specific job details', example_3_get_specific_job),
        '4': ('Filter by company', example_4_filter_by_company),
        '5': ('Export to JSON', example_5_export_to_json),
        '6': ('Visible browser', example_6_visible_browser),
        '7': ('Manual session', example_7_manual_session),
    }

    if len(sys.argv) > 1:
        example_num = sys.argv[1]
        if example_num in examples:
            _, func = examples[example_num]
            func()
        else:
            print("Invalid example number")
            print("\nAvailable examples:")
            for num, (desc, _) in examples.items():
                print(f"  {num}: {desc}")
    else:
        print("Stapply.ai API Client - Example Usage")
        print("\nUsage: python example_usage.py <example_number>")
        print("\nAvailable examples:")
        for num, (desc, _) in examples.items():
            print(f"  {num}: {desc}")
        print("\nExample: python example_usage.py 1")
