"""Generates Playwright automation scripts from recorded actions."""

import json
from typing import List

from .action_recorder import RecordedAction


class PlaywrightCodeGenerator:
    """Generates Playwright automation scripts from recorded actions."""

    def __init__(self, actions: List[RecordedAction], start_url: str | None = None):
        self.actions = self._clean_actions(actions)
        self.start_url = start_url

    def _clean_actions(self, actions: List[RecordedAction]) -> List[RecordedAction]:
        """Clean up actions (deduplicate fills, remove redundant events)."""
        cleaned = []
        
        for i, action in enumerate(actions):
            # For 'fill' actions on the same selector, only keep the last one
            if action.type == "fill":
                # Look ahead to see if there's another fill for this selector
                is_last_fill = True
                for next_action in actions[i + 1:]:
                    if next_action.type == "fill" and next_action.selector == action.selector:
                        is_last_fill = False
                        break
                    if next_action.type != "fill":
                        # If interaction changes (e.g. click), we must keep the current fill
                        break
                
                if is_last_fill:
                    cleaned.append(action)
            else:
                cleaned.append(action)
                
        return cleaned
    
    def _get_base_url(self, url: str | None) -> str | None:
        """Extract base URL (without query params) for comparison."""
        if not url:
            return None
        from urllib.parse import urlparse
        parsed = urlparse(url)
        return f"{parsed.scheme}://{parsed.netloc}{parsed.path}"

    def generate(self) -> str:
        """Generate complete Playwright script."""
        lines = [
            '"""',
            "Playwright automation script",
            "Generated by reverse-api-engineer",
            '"""',
            "",
            "from playwright.sync_api import sync_playwright",
            "",
            "",
            "# Stealth Chrome arguments to avoid bot detection",
            "STEALTH_ARGS = [",
            '    "--disable-blink-features=AutomationControlled",',
            '    "--disable-infobars",',
            '    "--disable-dev-shm-usage",',
            '    "--disable-extensions",',
            '    "--no-first-run",',
            '    "--no-default-browser-check",',
            ']',
            "",
            "",
            "def main():",
            "    with sync_playwright() as p:",
            "        # Launch browser with stealth settings",
            "        browser = p.chromium.launch(",
            "            headless=False,",
            "            args=STEALTH_ARGS,",
            '            ignore_default_args=["--enable-automation"],',
            "        )",
            "        context = browser.new_context(",
            '            user_agent="Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Safari/537.36",',
            "        )",
            "        page = context.new_page()",
            "",
        ]

        if self.start_url:
            lines.append(f'        # Start URL')
            lines.append(f'        page.goto({json.dumps(self.start_url)})')
            lines.append("")

        current_url = self.start_url
        last_nav_base = self._get_base_url(self.start_url) if self.start_url else None

        for action in self.actions:
            if action.type == "click" and action.selector:
                lines.append(f"        page.click({json.dumps(action.selector)})")
            
            elif action.type == "fill" and action.selector and action.value is not None:
                # Use json.dumps to safely escape newlines, quotes, and backslashes
                lines.append(f"        page.fill({json.dumps(action.selector)}, {json.dumps(action.value)})")
            
            elif action.type == "press" and action.selector and action.value is not None:
                lines.append(f"        page.press({json.dumps(action.selector)}, {json.dumps(action.value)})")
            
            elif action.type == "navigate":
                # Skip if same as start URL or very similar (just different query params)
                nav_base = self._get_base_url(action.url)
                if nav_base != last_nav_base:
                    lines.append(f"        page.goto({json.dumps(action.url)})")
                    last_nav_base = nav_base
                else:
                    continue  # Skip duplicate navigation, don't add delay

            # Add a small delay between actions as requested
            lines.append("        page.wait_for_timeout(1000)")

        lines.extend([
            "",
            "        # Keep browser open for a moment",
            "        page.wait_for_timeout(2000)",
            "",
            "        context.close()",
            "        browser.close()",
            "",
            "",
            "if __name__ == '__main__':",
            "    main()",
            ""
        ])

        return "\n".join(lines)
