# Authentication Patterns Reference

This document covers common authentication patterns found in web APIs and how to handle them when generating API clients. These patterns align with the reverse-api-engineer codebase conventions.

## Overview

When analyzing HAR files, identify authentication by checking:
- Request headers: `Authorization`, `X-API-Key`, `X-Token`, `Cookie`
- Response headers: `Set-Cookie`, `X-CSRF-Token`
- Request body: `username`, `password`, `token`, `api_key`
- Response body: `access_token`, `refresh_token`, `session_id`
- Query parameters: `key`, `api_key`, `token`, `access_token`
- Login endpoints: `/login`, `/auth`, `/oauth/token`, `/api/auth`

---

## 1. Bearer Token (JWT)

### Detection in HAR
```json
{
  "name": "Authorization",
  "value": "Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
}
```

### Generated Client Pattern
```python
"""
API client with Bearer token authentication.
Generated by reverse-api-engineer.
"""

import requests
from typing import Optional, Dict, Any
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class APIClient:
    """API client with Bearer token authentication."""

    def __init__(
        self,
        base_url: str,
        token: str,
        session: Optional[requests.Session] = None,
    ):
        self.base_url = base_url.rstrip("/")
        self.token = token
        self.session = session or requests.Session()
        self._setup_session()

    def _setup_session(self):
        """Configure session with authentication headers."""
        self.session.headers.update({
            "Authorization": f"Bearer {self.token}",
            "Accept": "application/json",
            "Content-Type": "application/json",
        })

    def _request(
        self,
        method: str,
        endpoint: str,
        **kwargs,
    ) -> requests.Response:
        """Make an HTTP request with error handling."""
        url = f"{self.base_url}{endpoint}"
        try:
            response = self.session.request(method, url, **kwargs)
            response.raise_for_status()
            return response
        except requests.exceptions.RequestException as e:
            logger.error(f"Request failed: {e}")
            raise
```

### Token Refresh Pattern
If you detect a `/refresh` or `/token/refresh` endpoint in the HAR:
```python
def refresh_token(self, refresh_token: str) -> str:
    """
    Refresh the access token.
    
    Args:
        refresh_token: The refresh token from initial auth
        
    Returns:
        New access token
    """
    response = self._request("POST", "/auth/refresh", json={
        "refresh_token": refresh_token
    })
    data = response.json()
    self.token = data["access_token"]
    self.session.headers["Authorization"] = f"Bearer {self.token}"
    return self.token
```

---

## 2. API Key

### Detection in HAR
Header-based:
```json
{"name": "X-API-Key", "value": "sk_live_xxxxx"}
```

Query parameter:
```json
{"name": "api_key", "value": "xxxxx"}
```

### Generated Client Pattern (Header)
```python
class APIClient:
    """API client with API key authentication."""

    def __init__(
        self,
        base_url: str,
        api_key: str,
        session: Optional[requests.Session] = None,
    ):
        self.base_url = base_url.rstrip("/")
        self.api_key = api_key
        self.session = session or requests.Session()
        self._setup_session()

    def _setup_session(self):
        """Configure session with API key header."""
        self.session.headers.update({
            "X-API-Key": self.api_key,
            "Accept": "application/json",
        })
```

### Generated Client Pattern (Query Parameter)
```python
class APIClient:
    """API client with API key in query parameters."""

    def __init__(self, base_url: str, api_key: str):
        self.base_url = base_url.rstrip("/")
        self.api_key = api_key
        self.session = requests.Session()

    def _request(
        self,
        method: str,
        endpoint: str,
        **kwargs,
    ) -> requests.Response:
        """Make request with API key in query params."""
        url = f"{self.base_url}{endpoint}"
        params = kwargs.pop("params", {})
        params["api_key"] = self.api_key
        try:
            response = self.session.request(method, url, params=params, **kwargs)
            response.raise_for_status()
            return response
        except requests.exceptions.RequestException as e:
            logger.error(f"Request failed: {e}")
            raise
```

---

## 3. Session Cookies

### Detection in HAR
Look for `Set-Cookie` in response headers after login:
```json
{"name": "Set-Cookie", "value": "sessionid=abc123; Path=/; HttpOnly"}
```

Common cookie names: `sessionid`, `PHPSESSID`, `connect.sid`, `_session`

### Generated Client Pattern
```python
class APIClient:
    """API client with session-based authentication."""

    def __init__(self, base_url: str):
        self.base_url = base_url.rstrip("/")
        self.session = requests.Session()  # Automatically handles cookies

    def login(self, username: str, password: str) -> bool:
        """
        Authenticate with username and password.
        
        Args:
            username: User's username
            password: User's password
            
        Returns:
            True if login successful
        """
        try:
            response = self.session.post(
                f"{self.base_url}/login",
                data={"username": username, "password": password}
            )
            response.raise_for_status()
            return True
        except requests.exceptions.RequestException as e:
            logger.error(f"Login failed: {e}")
            return False

    def _request(
        self,
        method: str,
        endpoint: str,
        **kwargs,
    ) -> requests.Response:
        """Make request with session cookies."""
        url = f"{self.base_url}{endpoint}"
        try:
            response = self.session.request(method, url, **kwargs)
            response.raise_for_status()
            return response
        except requests.exceptions.RequestException as e:
            logger.error(f"Request failed: {e}")
            raise
```

---

## 4. CSRF Tokens

### Detection in HAR
Hidden form field in HTML:
```html
<input type="hidden" name="csrf_token" value="...">
```

Header in requests:
```json
{"name": "X-CSRF-Token", "value": "..."}
```

Cookie:
```json
{"name": "csrftoken", "value": "..."}
```

### Generated Client Pattern
```python
import re


class APIClient:
    """API client with CSRF token handling."""

    def __init__(self, base_url: str):
        self.base_url = base_url.rstrip("/")
        self.session = requests.Session()
        self.csrf_token: Optional[str] = None

    def _get_csrf_token(self) -> str:
        """Fetch CSRF token from cookie or page."""
        # Option 1: From cookie
        if "csrftoken" in self.session.cookies:
            return self.session.cookies["csrftoken"]

        # Option 2: From page content
        response = self.session.get(f"{self.base_url}/")
        match = re.search(r'csrf_token["\']:\s*["\']([^"\']+)', response.text)
        if match:
            return match.group(1)
        
        # Option 3: From meta tag
        match = re.search(r'<meta name="csrf-token" content="([^"]+)"', response.text)
        if match:
            return match.group(1)
        
        return ""

    def _request(
        self,
        method: str,
        endpoint: str,
        **kwargs,
    ) -> requests.Response:
        """Make request with CSRF token for mutating operations."""
        url = f"{self.base_url}{endpoint}"
        
        # Add CSRF token for POST, PUT, DELETE, PATCH
        if method.upper() in ("POST", "PUT", "DELETE", "PATCH"):
            if not self.csrf_token:
                self.csrf_token = self._get_csrf_token()
            headers = kwargs.pop("headers", {})
            headers["X-CSRF-Token"] = self.csrf_token
            kwargs["headers"] = headers
        
        try:
            response = self.session.request(method, url, **kwargs)
            response.raise_for_status()
            return response
        except requests.exceptions.RequestException as e:
            logger.error(f"Request failed: {e}")
            raise
```

---

## 5. OAuth 2.0

### Detection in HAR
Look for:
- Token endpoint: `/oauth/token`, `/auth/token`
- `grant_type` in request body
- `access_token` and `refresh_token` in response

### Generated Client Pattern (Client Credentials)
```python
class APIClient:
    """API client with OAuth 2.0 Client Credentials flow."""

    def __init__(
        self,
        base_url: str,
        client_id: str,
        client_secret: str,
    ):
        self.base_url = base_url.rstrip("/")
        self.client_id = client_id
        self.client_secret = client_secret
        self.access_token: Optional[str] = None
        self.refresh_token: Optional[str] = None
        self.session = requests.Session()

    def authenticate(self) -> bool:
        """
        Authenticate using Client Credentials flow.
        
        Returns:
            True if authentication successful
        """
        try:
            response = requests.post(
                f"{self.base_url}/oauth/token",
                data={
                    "grant_type": "client_credentials",
                    "client_id": self.client_id,
                    "client_secret": self.client_secret,
                }
            )
            response.raise_for_status()
            data = response.json()
            self.access_token = data["access_token"]
            self.refresh_token = data.get("refresh_token")
            self.session.headers["Authorization"] = f"Bearer {self.access_token}"
            return True
        except requests.exceptions.RequestException as e:
            logger.error(f"OAuth authentication failed: {e}")
            return False
```

---

## 6. Basic Authentication

### Detection in HAR
```json
{"name": "Authorization", "value": "Basic dXNlcm5hbWU6cGFzc3dvcmQ="}
```

The value is base64 encoded `username:password`.

### Generated Client Pattern
```python
from requests.auth import HTTPBasicAuth


class APIClient:
    """API client with HTTP Basic authentication."""

    def __init__(
        self,
        base_url: str,
        username: str,
        password: str,
    ):
        self.base_url = base_url.rstrip("/")
        self.session = requests.Session()
        self.session.auth = HTTPBasicAuth(username, password)
        self.session.headers.update({
            "Accept": "application/json",
        })
```

---

## 7. Custom Headers

### Detection in HAR
Non-standard headers like:
```json
{"name": "X-App-Token", "value": "..."}
{"name": "X-Client-Id", "value": "..."}
{"name": "X-Request-Id", "value": "..."}
```

### Generated Client Pattern
```python
class APIClient:
    """API client with custom header authentication."""

    def __init__(
        self,
        base_url: str,
        app_token: str,
        client_id: str,
    ):
        self.base_url = base_url.rstrip("/")
        self.session = requests.Session()
        self._setup_session(app_token, client_id)

    def _setup_session(self, app_token: str, client_id: str):
        """Configure session with custom headers."""
        self.session.headers.update({
            "X-App-Token": app_token,
            "X-Client-Id": client_id,
            "Accept": "application/json",
        })
```

---

## 8. Signed Requests (HMAC)

### Detection in HAR
Look for:
- Signature headers: `X-Signature`, `X-Hmac-Signature`
- Timestamp headers: `X-Timestamp`, `X-Request-Timestamp`
- Nonce headers: `X-Nonce`

### Generated Client Pattern
```python
import hmac
import hashlib
import time
import uuid


class APIClient:
    """API client with HMAC request signing."""

    def __init__(
        self,
        base_url: str,
        api_key: str,
        secret: str,
    ):
        self.base_url = base_url.rstrip("/")
        self.api_key = api_key
        self.secret = secret
        self.session = requests.Session()

    def _sign_request(
        self,
        method: str,
        path: str,
        body: str = "",
    ) -> Dict[str, str]:
        """
        Generate signature headers for request.
        
        Args:
            method: HTTP method
            path: Request path
            body: Request body as string
            
        Returns:
            Dictionary of signature headers
        """
        timestamp = str(int(time.time()))
        nonce = str(uuid.uuid4())
        message = f"{method}{path}{timestamp}{nonce}{body}"
        
        signature = hmac.new(
            self.secret.encode(),
            message.encode(),
            hashlib.sha256
        ).hexdigest()

        return {
            "X-API-Key": self.api_key,
            "X-Timestamp": timestamp,
            "X-Nonce": nonce,
            "X-Signature": signature,
        }

    def _request(
        self,
        method: str,
        endpoint: str,
        **kwargs,
    ) -> requests.Response:
        """Make signed request."""
        url = f"{self.base_url}{endpoint}"
        body = kwargs.get("json", kwargs.get("data", ""))
        if isinstance(body, dict):
            import json
            body = json.dumps(body)
        
        headers = kwargs.pop("headers", {})
        headers.update(self._sign_request(method, endpoint, body))
        kwargs["headers"] = headers
        
        try:
            response = self.session.request(method, url, **kwargs)
            response.raise_for_status()
            return response
        except requests.exceptions.RequestException as e:
            logger.error(f"Request failed: {e}")
            raise
```

---

## Priority Order

When multiple auth methods are detected in HAR, use this priority:

1. **OAuth 2.0** - Most robust for production
2. **Bearer Token** - Simple and stateless  
3. **Session Cookies** - Required for some sites
4. **API Key** - Simple but less secure
5. **Basic Auth** - Only if no other option

---

## Bot Detection Fallback

If `requests` fails with 403/captcha, switch to Playwright:

```python
from playwright.sync_api import sync_playwright


class PlaywrightClient:
    """Playwright-based client for sites with bot detection."""

    def __init__(self, base_url: str):
        self.base_url = base_url.rstrip("/")
        self.playwright = None
        self.browser = None
        self.context = None
        self.page = None

    def __enter__(self):
        self.playwright = sync_playwright().start()
        self.browser = self.playwright.chromium.launch(headless=True)
        self.context = self.browser.new_context()
        self.page = self.context.new_page()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.browser:
            self.browser.close()
        if self.playwright:
            self.playwright.stop()

    def get(self, endpoint: str) -> dict:
        """Make GET request via browser."""
        url = f"{self.base_url}{endpoint}"
        response = self.page.goto(url)
        return self.page.evaluate("() => JSON.parse(document.body.innerText)")
```
