#include "riptide_python_test.h"

#include "CommonInc.h"
#include "Reduce.h"

#include "ut_extensions.h"

#include "MathWorker.h"
#include "numpy_traits.h"
#include "missing_values.h"
#include "simd/avx2.h"

#include "buffer.h"
#include "mem_buffer.h"
#include "np_util.h"
#include "ut_extensions.h"

#include "ut_core.h"

#include <algorithm>
#include <array>
#include <iomanip>
#include <numeric>
#include <random>
#include <optional>
#include <utility>
#include <vector>

using namespace riptide_python_test::internal;
using namespace riptide_utility::internal;
using namespace boost::ut;
using riptide_utility::ut::file_suite;

namespace
{
    template <REDUCE_FUNCTIONS Fn>
    using reducefn_to_type = std::integral_constant<REDUCE_FUNCTIONS, Fn>;

    std::random_device dev{};

    using SupportedTypeCodeTypes =
        std::tuple<typecode_to_type<NPY_TYPES::NPY_BOOL>, typecode_to_type<NPY_TYPES::NPY_INT8>,
                   typecode_to_type<NPY_TYPES::NPY_INT16>, typecode_to_type<NPY_TYPES::NPY_INT32>,
                   typecode_to_type<NPY_TYPES::NPY_INT64>, typecode_to_type<NPY_TYPES::NPY_UINT8>,
                   typecode_to_type<NPY_TYPES::NPY_UINT16>, typecode_to_type<NPY_TYPES::NPY_UINT32>,
                   typecode_to_type<NPY_TYPES::NPY_UINT64>, typecode_to_type<NPY_TYPES::NPY_FLOAT>,
                   typecode_to_type<NPY_TYPES::NPY_DOUBLE>, typecode_to_type<NPY_TYPES::NPY_LONGDOUBLE>>;

    template <NPY_TYPES TypeCode>
    auto get_prim_value(PyObject * const obj)
    {
        if constexpr (TypeCode == NPY_FLOAT || TypeCode == NPY_DOUBLE || TypeCode == NPY_LONGDOUBLE)
        {
            double result{};
            if (PyFloat_Check(obj))
            {
                result = PyFloat_AsDouble(obj);
            }
            return result;
        }
        else if constexpr (TypeCode == NPY_UINT64)
        {
            uint64_t result{};
            if (PyLong_Check(obj))
            {
                result = PyLong_AsUnsignedLongLong(obj);
            }
            return result;
        }
        else
        {
            riptide::numpy_cpp_type_t<TypeCode> result{};
            if (PyLong_Check(obj))
            {
                result = static_cast<decltype(result)>(PyLong_AsLongLong(obj));
            }
            return result;
        }
    }

    constexpr NPY_TYPES get_output_typecode(NPY_TYPES const typecode_in, REDUCE_FUNCTIONS const fn)
    {
        if (fn >= REDUCE_ARGMAX && fn <= REDUCE_NANARGMAX)
        {
            return NPY_UINT64;
        }

        if (fn >= 100 && fn < REDUCE_MIN)
        {
            return NPY_DOUBLE;
        }

        if (fn >= REDUCE_MIN && fn <= REDUCE_NANMAX)
        {
            switch (typecode_in)
            {
            case NPY_FLOAT:
            case NPY_DOUBLE:
            case NPY_LONGDOUBLE:
                return NPY_DOUBLE;
            default:
                return typecode_in;
            }
        }

        switch (typecode_in)
        {
        case NPY_UINT64:
            return NPY_UINT64;

        case NPY_FLOAT:
        case NPY_DOUBLE:
        case NPY_LONGDOUBLE:
            return NPY_DOUBLE;

        default:
            return NPY_INT64;
        }
    }

    template <NPY_TYPES TypeCodeIn, REDUCE_FUNCTIONS ReduceFn>
    struct reduce_tester
    {
        using cpp_type_in = riptide::numpy_cpp_type_t<TypeCodeIn>;

        static constexpr auto typecode_out = get_output_typecode(TypeCodeIn, ReduceFn);
        using cpp_type_out = riptide::numpy_cpp_type_t<typecode_out>;

        static void exec(const_buffer<cpp_type_in> const test_values, std::optional<cpp_type_out> const maybe_expected_value,
                         std::optional<int64_t> const arg,
                         reflection::source_location const & loc = reflection::source_location::current())
        {
            using desc_type = std::tuple<typecode_to_type<TypeCodeIn>, reducefn_to_type<ReduceFn>>;

            auto const caller_loc{ [&loc]
                                   {
                                       std::ostringstream stream;
                                       stream << "; caller: " << loc.file_name() << ':' << loc.line();
                                       return stream.str();
                                   }() };

            auto const input_array{ pyarray_from_array<TypeCodeIn>(test_values) };
            typed_expect<desc_type>((input_array != nullptr) >> fatal) << caller_loc;

            pyobject_ptr const retval{ arg.has_value() ?
                                           PyObject_CallMethod(riptide_module_p, "Reduce", "Oii", input_array.get(), ReduceFn,
                                                               arg.value()) :
                                           PyObject_CallMethod(riptide_module_p, "Reduce", "Oi", input_array.get(), ReduceFn) };
            if (! maybe_expected_value.has_value())
            {
                typed_expect<desc_type>(yes_pyerr(PyExc_ValueError)) << caller_loc;
                return;
            }
            typed_expect<desc_type>(no_pyerr() >> fatal) << caller_loc;

            auto const expected_value{ maybe_expected_value.value() };

            auto const actual_value{ get_prim_value<typecode_out>(retval.get()) };
            typed_expect<desc_type>(no_pyerr() >> fatal) << caller_loc;

            bool is_equal;
            if constexpr (std::is_floating_point_v<cpp_type_out>)
            {
                is_equal = equal_to_nan_aware(actual_value, expected_value, equal_within<cpp_type_out>{ 1e6 });
            }
            else
            {
                is_equal = equal_to_nan_aware(actual_value, expected_value);
            }

            typed_expect<desc_type>(is_equal) << "for" << test_values << "expected : " << to_out(expected_value)
                                              << ", actual : " << to_out(actual_value) << caller_loc;
        }
    };

    enum class test_case_id
    {
        VALID,
        MIXED,
        INVALID,
    };

    template <REDUCE_FUNCTIONS>
    struct reduce_tests;

    template <REDUCE_FUNCTIONS ReduceFn>
    struct reduce_tests_base
    {
        using Derived = reduce_tests<ReduceFn>;

        static constexpr REDUCE_FUNCTIONS reduce_fn = ReduceFn;

        template <NPY_TYPES TypeCode>
        using tester_type = reduce_tester<TypeCode, ReduceFn>;

        template <NPY_TYPES TypeCode>
        struct test_case
        {
            using cpp_type_in = typename tester_type<TypeCode>::cpp_type_in;
            using cpp_type_out = typename tester_type<TypeCode>::cpp_type_out;

            any_const_buffer<cpp_type_in> test_values_;
            std::optional<cpp_type_out> expected_value_;
            std::optional<int64_t> arg_;
            reflection::source_location loc_;

            template <template <typename> typename BufferInT>
            test_case(BufferInT<cpp_type_in> && test_values, std::optional<cpp_type_out> const expected_value, int64_t const arg,
                      reflection::source_location const & loc = reflection::source_location::current())
                : test_values_{ std::move(test_values) }
                , expected_value_{ expected_value }
                , arg_{ arg }
                , loc_{ loc }
            {
            }

            template <template <typename> typename BufferInT>
            test_case(BufferInT<cpp_type_in> && test_values, std::optional<cpp_type_out> const expected_value,
                      reflection::source_location const & loc = reflection::source_location::current())
                : test_values_{ std::move(test_values) }
                , expected_value_{ expected_value }
                , loc_{ loc }
            {
            }
        };

        template <test_case_id Id>
        struct test
        {
            template <typename T, NPY_TYPES TypeCode = T::value>
            void operator()() const
            {
                auto const testcase = Derived::template get_test_case<Id, TypeCode>();
                tester_type<TypeCode>::exec(testcase.test_values_, testcase.expected_value_, testcase.arg_, testcase.loc_);
            }
        };
    };

    //static constexpr size_t big_size = 2 * CMathWorker::WORK_ITEM_BIG;
    static constexpr size_t big_size = 3;

    template <>
    struct reduce_tests<REDUCE_SUM> : reduce_tests_base<REDUCE_SUM>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    get_same_values<cpp_type_in>(big_size, 1),
                    big_size,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(big_size),
                    TypeCode == NPY_BOOL ? true : invalid,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(big_size),
                    TypeCode == NPY_BOOL ? false : invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANSUM> : reduce_tests_base<REDUCE_NANSUM>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    get_same_values<cpp_type_in>(3, 1),
                    3,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    0,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_MEAN> : reduce_tests_base<REDUCE_MEAN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    get_same_values<cpp_type_in>(3, 1),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 1. / 3. : invalid,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? false : invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANMEAN> : reduce_tests_base<REDUCE_NANMEAN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    get_same_values<cpp_type_in>(3, 1),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 1. / 3. : 1. / 2.,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    0.,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_VAR> : reduce_tests_base<REDUCE_VAR>
    {
        static constexpr int64_t ddof = 0; // numpy

        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    2. / 9.,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 2. / 9. : invalid,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? false : invalid,
                    ddof,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANVAR> : reduce_tests_base<REDUCE_NANVAR>
    {
        static constexpr int64_t ddof = 0; // numpy

        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    2. / 9.,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 2. / 9. : 1. / 4.,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0. : invalid,
                    ddof,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_STD> : reduce_tests_base<REDUCE_STD>
    {
        static constexpr int64_t ddof = 0; // numpy

        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0.4714045207910317,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0.4714045207910317 : invalid,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? false : invalid,
                    ddof,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANSTD> : reduce_tests_base<REDUCE_NANSTD>
    {
        static constexpr int64_t ddof = 0; // numpy

        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0.4714045207910317,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0.4714045207910317 : 0.5,
                    ddof,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0. : invalid,
                    ddof,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_MIN> : reduce_tests_base<REDUCE_MIN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0 : invalid,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0 : invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANMIN> : reduce_tests_base<REDUCE_NANMIN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_MAX> : reduce_tests_base<REDUCE_MAX>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 1 : invalid,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANMAX> : reduce_tests_base<REDUCE_NANMAX>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    invalid,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_ARGMIN> : reduce_tests_base<REDUCE_ARGMIN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 0 : 1,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    0,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANARGMIN> : reduce_tests_base<REDUCE_NANARGMIN>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    1,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? std::optional<cpp_type_out>{ 0 } : std::nullopt,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_ARGMAX> : reduce_tests_base<REDUCE_ARGMAX>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? 2 : 1,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    0,
                };
            }
        }
    };

    template <>
    struct reduce_tests<REDUCE_NANARGMAX> : reduce_tests_base<REDUCE_NANARGMAX>
    {
        template <test_case_id Id, NPY_TYPES TypeCode>
        static auto get_test_case()
        {
            using test_case_type = test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;
            using cpp_type_out = typename test_case_type::cpp_type_out;

            if constexpr (Id == test_case_id::VALID)
            {
                return test_case_type{
                    make_mem_buffer<cpp_type_in>({ 1, 0, 1 }),
                    0,
                };
            }

            else if constexpr (Id == test_case_id::MIXED)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_mixed_values<cpp_type_in>(3),
                    2,
                };
            }

            else if constexpr (Id == test_case_id::INVALID)
            {
                constexpr auto invalid{ riptide::invalid_for_type<cpp_type_out>::value };
                return test_case_type{
                    get_invalid_values<cpp_type_in>(3),
                    TypeCode == NPY_BOOL ? std::optional<cpp_type_out>{ 0 } : std::nullopt,
                };
            }
        }
    };

    template <REDUCE_FUNCTIONS ReduceFn>
    struct reduce_bignum_tests : reduce_tests_base<ReduceFn>
    {
        using base_type = reduce_tests_base<ReduceFn>;

        static constexpr NPY_TYPES TypeCode = NPY_UINT64;

        void operator()() const
        {
            using tester_type = base_type::template tester_type<TypeCode>;
            using test_case_type = base_type::template test_case<TypeCode>;
            using cpp_type_in = typename test_case_type::cpp_type_in;

            constexpr cpp_type_in minv{ 7065021915437483413U };
            constexpr cpp_type_in maxv{ 9518076337470921052U };

            test_case_type const testcase{
                get_split_values<cpp_type_in>(CMathWorker::WORK_ITEM_BIG, minv, maxv),
                (ReduceFn == REDUCE_MIN || ReduceFn == REDUCE_NANMIN) ? minv : maxv,
            };

            tester_type::exec(testcase.test_values_, testcase.expected_value_, testcase.arg_, testcase.loc_);
        };
    };

    file_suite reduce_ops = []
    {
        "reduce_sum_valid"_test = reduce_tests<REDUCE_SUM>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_sum_mixed"_test = reduce_tests<REDUCE_SUM>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_sum_invalid"_test = reduce_tests<REDUCE_SUM>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nansum_valid"_test = reduce_tests<REDUCE_NANSUM>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nansum_mixed"_test = reduce_tests<REDUCE_NANSUM>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nansum_invalid"_test = reduce_tests<REDUCE_NANSUM>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_mean_valid"_test = reduce_tests<REDUCE_MEAN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_mean_mixed"_test = reduce_tests<REDUCE_MEAN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_mean_invalid"_test = reduce_tests<REDUCE_MEAN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanmean_valid"_test = reduce_tests<REDUCE_NANMEAN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanmean_mixed"_test = reduce_tests<REDUCE_NANMEAN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanmean_invalid"_test = reduce_tests<REDUCE_NANMEAN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_var_valid"_test = reduce_tests<REDUCE_VAR>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_var_mixed"_test = reduce_tests<REDUCE_VAR>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_var_invalid"_test = reduce_tests<REDUCE_VAR>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanvar_valid"_test = reduce_tests<REDUCE_NANVAR>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanvar_mixed"_test = reduce_tests<REDUCE_NANVAR>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanvar_invalid"_test = reduce_tests<REDUCE_NANVAR>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_std_valid"_test = reduce_tests<REDUCE_STD>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_std_mixed"_test = reduce_tests<REDUCE_STD>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_std_invalid"_test = reduce_tests<REDUCE_STD>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanstd_valid"_test = reduce_tests<REDUCE_NANSTD>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanstd_mixed"_test = reduce_tests<REDUCE_NANSTD>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanstd_invalid"_test = reduce_tests<REDUCE_NANSTD>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_min_valid"_test = reduce_tests<REDUCE_MIN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_min_mixed"_test = reduce_tests<REDUCE_MIN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_min_invalid"_test = reduce_tests<REDUCE_MIN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};
        "reduce_min_bignum"_test = reduce_bignum_tests<REDUCE_MIN>{};

        "reduce_nanmin_valid"_test = reduce_tests<REDUCE_NANMIN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanmin_mixed"_test = reduce_tests<REDUCE_NANMIN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanmin_invalid"_test = reduce_tests<REDUCE_NANMIN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanmin_bignum"_test = reduce_bignum_tests<REDUCE_NANMIN>{};

        "reduce_max_valid"_test = reduce_tests<REDUCE_MAX>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_max_mixed"_test = reduce_tests<REDUCE_MAX>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_max_invalid"_test = reduce_tests<REDUCE_MAX>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};
        "reduce_max_bignum"_test = reduce_bignum_tests<REDUCE_MAX>{};

        "reduce_nanmax_valid"_test = reduce_tests<REDUCE_NANMAX>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanmax_mixed"_test = reduce_tests<REDUCE_NANMAX>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanmax_invalid"_test = reduce_tests<REDUCE_NANMAX>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanmax_bignum"_test = reduce_bignum_tests<REDUCE_NANMAX>{};

        "reduce_argmin_valid"_test = reduce_tests<REDUCE_ARGMIN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_argmin_mixed"_test = reduce_tests<REDUCE_ARGMIN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_argmin_invalid"_test = reduce_tests<REDUCE_ARGMIN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanargmin_valid"_test = reduce_tests<REDUCE_NANARGMIN>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanargmin_mixed"_test = reduce_tests<REDUCE_NANARGMIN>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanargmin_invalid"_test = reduce_tests<REDUCE_NANARGMIN>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_argmax_valid"_test = reduce_tests<REDUCE_ARGMAX>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_argmax_mixed"_test = reduce_tests<REDUCE_ARGMAX>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_argmax_invalid"_test = reduce_tests<REDUCE_ARGMAX>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanargmax_valid"_test = reduce_tests<REDUCE_NANARGMAX>::test<test_case_id::VALID>{} | SupportedTypeCodeTypes{};
        "reduce_nanargmax_mixed"_test = reduce_tests<REDUCE_NANARGMAX>::test<test_case_id::MIXED>{} | SupportedTypeCodeTypes{};
        "reduce_nanargmax_invalid"_test = reduce_tests<REDUCE_NANARGMAX>::test<test_case_id::INVALID>{} | SupportedTypeCodeTypes{};

        "reduce_nanmean_char"_test = [&]
        {
            std::array<char, 5> test_values{ 'a', 'a', 'a', 'b', 'c' };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_BYTE, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANMEAN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
        };

        "reduce_mean_float"_test = [&]
        {
            std::array<float, 5> test_values{ 1.0f, 2.0f, 0.0f, 3.0f, 4.0f };

            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_MEAN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            expect(no_pyerr() >> fatal);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 2.0_f);
            expect(is_float_type == 1_i);
        };

        "reduce_nanmean_float"_test = [&]
        {
            std::array<float, 5> test_values{ 1.0f, 2.0f, NAN, 3.0f, 4.0f };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANMEAN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            expect(no_pyerr() >> fatal);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 2.5_f);
            expect(is_float_type == 1_i);
        };

        "reduce_nanmax"_test = [&]
        {
            std::array<double, 5> test_values{ 3.0, NAN, 2.0, 3.142, -5.0 };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_DOUBLE, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANMAX) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            expect(no_pyerr() >> fatal);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            double ret_val = PyFloat_AsDouble(retval);
            expect(ret_val == 3.142_d);
            expect(is_float_type == 1_i);
        };

        "reduce_min"_test = [&]
        {
            std::array<double, 5> test_values{ 3.0, -42.5, 2.0, 3.142, -5.0 };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_DOUBLE, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_MIN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            double ret_val = PyFloat_AsDouble(retval);
            expect(ret_val == -42.5_d);
            expect(is_float_type == 1_i);
        };

        "reduce_nanmin"_test = [&]
        {
            std::array<double, 5> test_values{ 3.0, NAN, 2.0, 3.142, -5.0 };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_DOUBLE, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANMIN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            double ret_val = PyFloat_AsDouble(retval);
            expect(ret_val == -5.0_d);
            expect(is_float_type == 1_i);
        };

        "reduce_nanmean_float"_test = [&]
        {
            std::array<float, 5> test_values{ 1.0f, 2.0f, NAN, 3.0f, 4.0f };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANMEAN) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 2.5_f);
            expect(is_float_type == 1_i);
        };

        "reduce_sum"_test = [&]
        {
            std::array<int32_t, 5> test_values{ 1, 2, 42, 3, 4 };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_INT32, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_SUM) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            int32_t ret_val = PyLong_AsLong(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 52_i);
            expect(is_float_type == 0_i);
        };

        "reduce_sum_float"_test = [&]
        {
            std::array<float, 5> test_values{ 1.0f, 2.0f, 42.5, 3.0f, 4.0f };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_SUM) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 52.5_f);
            expect(is_float_type == 1_i);
        };

        "reduce_nansum_float"_test = [&]
        {
            std::array<float, 5> test_values{ 1.0f, NAN, 42.5, 3.0f, 4.0f };
            npy_intp const dim_len{ 5 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_values.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_NANSUM) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            expect(ret_val == 50.5_f);
            expect(is_float_type == 1_i);
        };

        "reduce_float_long"_test = [&]
        {
            auto const d{ dev() };
            std::vector<float> test_data(65536);
            std::iota(std::begin(test_data), std::end(test_data), -10000.0f);
            std::shuffle(std::begin(test_data), std::end(test_data), std::mt19937{ d });
            npy_intp const dim_len{ 65536 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_FLOAT, test_data.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_sum{ Py_BuildValue("i", REDUCE_SUM) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_sum, NULL);
            double ret_val = PyFloat_AsDouble(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_sum != nullptr);
            expect(retval != nullptr);
            float x = 1492090880.0f;
            expect(ret_val == x) << "d=" << d << "We got " << ret_val << " but wanted " << x << "\n";
            expect(is_float_type == 1_i);

            PyObject * reduce_fn_mean{ Py_BuildValue("i", REDUCE_MEAN) };
            retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_mean, NULL);
            double calc_avg = PyFloat_AsDouble(retval);
            double actual_avg = x / test_data.size();
            is_float_type = PyFloat_Check(retval);

            expect(reduce_fn_mean != nullptr);
            expect(retval != nullptr);
            expect(calc_avg == actual_avg);
            expect(is_float_type == 1_i);

            PyObject * reduce_fn_var{ Py_BuildValue("i", REDUCE_VAR) };
            retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_var, NULL);
            ret_val = PyFloat_AsDouble(retval);
            is_float_type = PyFloat_Check(retval);

            expect(reduce_fn_var != nullptr);
            expect(retval != nullptr);
            double y = 357919402.0 + 2.0 / 3.0;
            expect(ret_val == y) << "Calculated variance [" << std::setprecision(21) << ret_val << "] expected was [" << y
                                 << "]\n";
            ;
            expect(is_float_type == 1_i);

            PyObject * reduce_fn_std{ Py_BuildValue("i", REDUCE_STD) };
            retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_std, NULL);
            ret_val = PyFloat_AsDouble(retval);
            is_float_type = PyFloat_Check(retval);

            expect(reduce_fn_var != nullptr);
            expect(retval != nullptr);
            y = std::sqrt(y);
            expect(ret_val == y) << "Calculated population standard deviation [" << std::setprecision(21) << ret_val
                                 << "] expected was [" << y << "]\n";
            ;
            expect(is_float_type == 1_i);
        };

        "reduce_sum_int32_long"_test = [&]
        {
            std::vector<int32_t> test_data(65535);
            std::iota(std::begin(test_data), std::end(test_data), -10000);
            std::shuffle(std::begin(test_data), std::end(test_data), std::mt19937{ dev() });
            npy_intp const dim_len{ 65535 };
            PyObject * array{ PyArray_SimpleNewFromData(1, &dim_len, NPY_INT32, test_data.data()) };
            PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
            PyObject * reduce_fn_num{ Py_BuildValue("i", REDUCE_SUM) };
            PyObject * retval = PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            int32_t ret_val = PyLong_AsLong(retval);
            int is_float_type = PyFloat_Check(retval);

            expect(array != nullptr);
            expect(function_object != nullptr);
            expect(reduce_fn_num != nullptr);
            expect(retval != nullptr);
            auto const expected{ std::accumulate(std::begin(test_data), std::end(test_data), 0) };
            expect(ret_val == expected) << "expected=" << expected << "actual=" << ret_val;
            expect(is_float_type == 0_i);
        };

        "reduce_invalid_propagation"_test = [&]<typename TypeCode, NPY_TYPES typecode = TypeCode::value,
                                                typename T = riptide::numpy_cpp_type<typecode>::type>
        {
            std::vector<T> input(100000, 0);
            std::fill(input.begin(), input.begin() + 13, riptide::invalid_for_type<T>::value);
            std::shuffle(input.begin(), input.end(), std::mt19937{ dev() });

            auto reduce = [&](REDUCE_FUNCTIONS function)
            {
                auto dims = static_cast<npy_intp>(input.size());
                PyObject * array{ PyArray_SimpleNewFromData(1, &dims, typecode, input.data()) };
                PyObject * function_object = get_named_function(riptide_module_p, "Reduce");
                PyObject * reduce_fn_num{ Py_BuildValue("i", function) };
                return PyObject_CallFunctionObjArgs(function_object, array, reduce_fn_num, NULL);
            };

            auto check =
                [&](REDUCE_FUNCTIONS function, const reflection::source_location & sl = reflection::source_location::current())
            {
                auto pyobject = reduce(function);
                bool is_result_valid;

                if (function >= REDUCE_ARGMIN && function <= REDUCE_NANARGMAX)
                {
                    uint64_t index = PyLong_AsUnsignedLongLong(pyobject);
                    is_result_valid = riptide::invalid_for_type<T>::is_valid(input[index]);
                }
                else if (function >= REDUCE_MEAN && function <= REDUCE_NANSTD)
                {
                    double result = PyFloat_AsDouble(pyobject);
                    is_result_valid = riptide::invalid_for_type<double>::is_valid(result);
                }
                else if (function >= REDUCE_MIN && function <= REDUCE_NANMAX)
                {
                    if (typecode == NPY_FLOAT || typecode == NPY_DOUBLE || typecode == NPY_LONGDOUBLE)
                    {
                        double result = PyFloat_AsDouble(pyobject);
                        is_result_valid = riptide::invalid_for_type<double>::is_valid(result);
                    }
                    else if (typecode == NPY_UINT64)
                    {
                        uint64_t result = PyLong_AsUnsignedLongLong(pyobject);
                        is_result_valid = riptide::invalid_for_type<uint64_t>::is_valid(result);
                    }
                    else
                    {
                        T result = static_cast<T>(PyLong_AsLongLong(pyobject));
                        is_result_valid = riptide::invalid_for_type<T>::is_valid(result);
                    }
                }
                else // SUM, NANSUM
                {
                    if (typecode == NPY_FLOAT || typecode == NPY_DOUBLE || typecode == NPY_LONGDOUBLE)
                    {
                        double result = PyFloat_AsDouble(pyobject);
                        is_result_valid = riptide::invalid_for_type<double>::is_valid(result);
                    }
                    else if (typecode == NPY_UINT64)
                    {
                        uint64_t result = PyLong_AsUnsignedLongLong(pyobject);
                        is_result_valid = riptide::invalid_for_type<uint64_t>::is_valid(result);
                    }
                    else
                    {
                        int64_t result = PyLong_AsLongLong(pyobject);
                        is_result_valid = riptide::invalid_for_type<int64_t>::is_valid(result);
                    }
                }

                bool is_nan_function = (function & 1) == 1;
                return expect(is_result_valid == is_nan_function, sl);
            };

            check(REDUCE_SUM) << "REDUCE_SUM" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_MEAN) << "REDUCE_MEAN" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_VAR) << "REDUCE_VAR" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_STD) << "REDUCE_STD" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_MIN) << "REDUCE_MIN" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_MAX) << "REDUCE_MAX" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_ARGMIN) << "REDUCE_ARGMIN" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_ARGMAX) << "REDUCE_ARGMAX" << reflection::type_name<T>() << "expected invalid result";
            check(REDUCE_NANSUM) << "REDUCE_NANSUM" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANMEAN) << "REDUCE_NANMEAN" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANVAR) << "REDUCE_NANVAR" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANSTD) << "REDUCE_NANSTD" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANMIN) << "REDUCE_NANMIN" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANMAX) << "REDUCE_NANMAX" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANARGMIN) << "REDUCE_NANARGMIN" << reflection::type_name<T>() << "expected valid result";
            check(REDUCE_NANARGMAX) << "REDUCE_NANARGMAX" << reflection::type_name<T>() << "expected valid result";
        }
        // bools aren't tested because they have no invalid handling
        | std::tuple<typecode_to_type<NPY_TYPES::NPY_INT8>, typecode_to_type<NPY_TYPES::NPY_INT16>,
                     typecode_to_type<NPY_TYPES::NPY_INT32>, typecode_to_type<NPY_TYPES::NPY_INT64>,
                     typecode_to_type<NPY_TYPES::NPY_UINT8>, typecode_to_type<NPY_TYPES::NPY_UINT16>,
                     typecode_to_type<NPY_TYPES::NPY_UINT32>, typecode_to_type<NPY_TYPES::NPY_UINT64>,
                     typecode_to_type<NPY_TYPES::NPY_FLOAT>, typecode_to_type<NPY_TYPES::NPY_DOUBLE>,
                     typecode_to_type<NPY_TYPES::NPY_LONGDOUBLE>>{};
    };
}