#!/usr/bin/env python3
"""
Simple DAG visualizer as fallback when Graphviz is not available
"""

import sys
import re
from pathlib import Path

def parse_dot_file(dot_content):
    """Parse DOT file and extract nodes and edges"""
    nodes = {}
    edges = []
    
    # Extract nodes
    node_pattern = r'(\d+)\[label = "([^"]+)"'
    for match in re.finditer(node_pattern, dot_content):
        node_id = match.group(1)
        label = match.group(2).replace('\\n', ' ')
        nodes[node_id] = label
    
    # Extract edges  
    edge_pattern = r'(\d+) -> (\d+)'
    for match in re.finditer(edge_pattern, dot_content):
        from_node = match.group(1)
        to_node = match.group(2)
        edges.append((from_node, to_node))
    
    return nodes, edges

def create_text_dag(nodes, edges):
    """Create a simple text representation of the DAG"""
    result = []
    result.append("📊 Workflow DAG (Text Representation)")
    result.append("=" * 50)
    result.append("")
    
    # Find root nodes (no incoming edges)
    incoming = {node_id for _, to_node in edges for node_id in [to_node]}
    root_nodes = [node_id for node_id in nodes.keys() if node_id not in incoming]
    
    def print_node(node_id, level=0):
        indent = "  " * level
        label = nodes.get(node_id, f"Node {node_id}")
        
        if "all" in label.lower():
            result.append(f"{indent}🎯 {label}")
        elif "rlooper_simulation" in label.lower():
            result.append(f"{indent}⚙️  {label}")
        else:
            result.append(f"{indent}📋 {label}")
        
        # Find child nodes
        children = [to_node for from_node, to_node in edges if from_node == node_id]
        for child in children:
            print_node(child, level + 1)
    
    # Print from root nodes
    for root in root_nodes:
        print_node(root)
    
    result.append("")
    result.append("Legend:")
    result.append("🎯 Target rule (final output)")
    result.append("⚙️  Processing rule")
    result.append("📋 Other rule")
    
    return "\n".join(result)

def create_html_dag(nodes, edges):
    """Create an HTML representation with simple styling"""
    html_content = f"""
<!DOCTYPE html>
<html>
<head>
    <title>Workflow DAG</title>
    <style>
        body {{ font-family: Arial, sans-serif; margin: 20px; }}
        .dag-container {{ max-width: 800px; }}
        .node {{ 
            margin: 10px 0; 
            padding: 10px; 
            border-radius: 5px; 
            border: 2px solid #ddd;
        }}
        .target {{ background-color: #e8f5e8; border-color: #4caf50; }}
        .process {{ background-color: #fff3e0; border-color: #ff9800; }}
        .other {{ background-color: #f5f5f5; border-color: #9e9e9e; }}
        .edge {{ margin: 5px 0; color: #666; }}
    </style>
</head>
<body>
    <div class="dag-container">
        <h1>🔗 Workflow DAG</h1>
        <p>Generated by rlooper_sim_python</p>
        
        <h2>📋 Workflow Rules:</h2>
"""
    
    for node_id, label in nodes.items():
        node_class = "target" if "all" in label.lower() else "process" if "rlooper" in label.lower() else "other"
        html_content += f'        <div class="node {node_class}"><strong>{label}</strong></div>\n'
    
    html_content += """
        <h2>🔗 Dependencies:</h2>
"""
    
    for from_node, to_node in edges:
        from_label = nodes.get(from_node, f"Node {from_node}")
        to_label = nodes.get(to_node, f"Node {to_node}")
        html_content += f'        <div class="edge">"{from_label}" → "{to_label}"</div>\n'
    
    html_content += """
    </div>
</body>
</html>
"""
    
    return html_content

def main(dot_file):
    
    if not dot_file.exists():
        print("❌ dag.dot not found. Run 'python run_workflow.py dag' first.")
        return 1
    
    print("📊 Creating fallback DAG visualizations...")
    
    # Read and parse DOT file
    with open(dot_file, 'r') as f:
        dot_content = f.read()
    
    nodes, edges = parse_dot_file(dot_content)
    
    if not nodes:
        print("⚠️  Could not parse DAG structure from dag.dot")
        return 1
    
    # Create text representation
    text_dag = create_text_dag(nodes, edges)
    with open("dag.txt", "w", encoding="utf-8") as f:
        f.write(text_dag)
    print("✅ Created dag.txt (text representation)")
    
    # Create HTML representation  
    html_dag = create_html_dag(nodes, edges)
    with open("dag.html", "w", encoding="utf-8") as f:
        f.write(html_dag)
    print("✅ Created dag.html (open in web browser)")
    
    # Print text version to console
    print("\n" + text_dag)
    
    print(f"\n💡 View options:")
    print(f"  - Text: cat dag.txt")
    print(f"  - HTML: Open dag.html in web browser")
    print(f"  - Online: Upload dag.dot to http://magjac.com/graphviz-visual-editor/")
    
    return 0

if __name__ == "__main__":
    sys.exit(main())