from __future__ import annotations

import asyncio
import concurrent.futures
import contextlib
import contextvars
import functools
import inspect
import threading
import time
import weakref
from collections import deque
from concurrent.futures.thread import ThreadPoolExecutor
from contextlib import asynccontextmanager
from types import TracebackType
from typing import (
    Any,
    AsyncGenerator,
    AsyncIterator,
    Awaitable,
    Callable,
    Coroutine,
    Deque,
    Generic,
    Iterator,
    List,
    MutableSet,
    Optional,
    Protocol,
    Type,
    TypeVar,
    Union,
    cast,
    runtime_checkable,
)

from ..utils.inspect import ensure_coroutine

__all__ = [
    "AsyncEventIterator",
    "AsyncEvent",
    "async_event",
    "AsyncTaskingEventIterator",
    "AsyncTaskingEvent",
    "async_tasking_event_iterator",
    "async_tasking_event",
    "check_canceled",
    "run_in_thread",
    "run_coroutine_in_thread",
    "Lock",
    "create_sub_task",
    "FutureInfo",
]

_T = TypeVar("_T")

_TResult = TypeVar("_TResult")
_TCallable = TypeVar("_TCallable", bound=Callable[..., Any])


class AsyncEventResultIteratorBase(Generic[_TCallable, _TResult]):
    def __init__(self) -> None:
        self._lock = threading.RLock()

        self._listeners: MutableSet[weakref.ref[Any]] = set()
        self._loop = asyncio.get_event_loop()

    def add(self, callback: _TCallable) -> None:
        def remove_listener(ref: Any) -> None:
            with self._lock:
                self._listeners.remove(ref)

        with self._lock:
            if inspect.ismethod(callback):
                self._listeners.add(weakref.WeakMethod(callback, remove_listener))
            else:
                self._listeners.add(weakref.ref(callback, remove_listener))

    def remove(self, callback: _TCallable) -> None:
        with self._lock:
            try:
                if inspect.ismethod(callback):
                    self._listeners.remove(weakref.WeakMethod(callback))
                else:
                    self._listeners.remove(weakref.ref(callback))
            except KeyError:
                pass

    def __contains__(self, obj: Any) -> bool:
        if inspect.ismethod(obj):
            return weakref.WeakMethod(obj) in self._listeners
        else:
            return weakref.ref(obj) in self._listeners

    def __len__(self) -> int:
        return len(self._listeners)

    def __iter__(self) -> Iterator[_TCallable]:
        for r in self._listeners:
            c = r()
            if c is not None:
                yield c

    async def __aiter__(self) -> AsyncIterator[_TCallable]:
        for r in self.__iter__():
            yield r

    async def _notify(
        self, *args: Any, callback_filter: Optional[Callable[[_TCallable], bool]] = None, **kwargs: Any
    ) -> AsyncIterator[_TResult]:

        for method in filter(
            lambda x: callback_filter(x) if callback_filter is not None else True,
            set(self),
        ):
            result = method(*args, **kwargs)
            if inspect.isawaitable(result):
                result = await result

            yield result


class AsyncEventIterator(AsyncEventResultIteratorBase[_TCallable, _TResult]):
    def __call__(self, *args: Any, **kwargs: Any) -> AsyncIterator[_TResult]:
        return self._notify(*args, **kwargs)


class AsyncEvent(AsyncEventResultIteratorBase[_TCallable, _TResult]):
    async def __call__(self, *args: Any, **kwargs: Any) -> List[_TResult]:
        return [a async for a in self._notify(*args, **kwargs)]


_TEvent = TypeVar("_TEvent")


class AsyncEventDescriptorBase(Generic[_TCallable, _TResult, _TEvent]):
    def __init__(
        self, _func: _TCallable, factory: Callable[..., _TEvent], *factory_args: Any, **factory_kwargs: Any
    ) -> None:
        self._func = _func
        self.__factory = factory
        self.__factory_args = factory_args
        self.__factory_kwargs = factory_kwargs
        self._owner: Optional[Any] = None
        self._owner_name: Optional[str] = None

    def __set_name__(self, owner: Any, name: str) -> None:
        self._owner = owner
        self._owner_name = name

    def __get__(self, obj: Any, objtype: Type[Any]) -> _TEvent:
        if obj is None:
            return self  # type: ignore

        name = f"__async_event_{self._func.__name__}__"
        if not hasattr(obj, name):
            setattr(obj, name, self.__factory(*self.__factory_args, **self.__factory_kwargs))

        return cast("_TEvent", getattr(obj, name))


class async_event_iterator(  # noqa: N801
    AsyncEventDescriptorBase[_TCallable, Any, AsyncEventIterator[_TCallable, Any]]
):
    def __init__(self, _func: _TCallable) -> None:
        super().__init__(_func, AsyncEventIterator[_TCallable, _TResult])


class async_event(AsyncEventDescriptorBase[_TCallable, Any, AsyncEvent[_TCallable, Any]]):  # noqa: N801
    def __init__(self, _func: _TCallable) -> None:
        super().__init__(_func, AsyncEvent[_TCallable, _TResult])


_F = TypeVar("_F", bound=Callable[..., Any])


def threaded(enabled: bool = True) -> Callable[[_F], _F]:
    def decorator(func: _F) -> _F:
        setattr(func, "__threaded__", enabled)
        return func

    return decorator


@runtime_checkable
class HasThreaded(Protocol):
    __threaded__: bool


class AsyncTaskingEventResultIteratorBase(AsyncEventResultIteratorBase[_TCallable, _TResult]):
    def __init__(self, *, task_name_prefix: Optional[str] = None) -> None:
        super().__init__()
        self._task_name_prefix = task_name_prefix or type(self).__qualname__

    async def _notify(  # type: ignore
        self,
        *args: Any,
        result_callback: Optional[Callable[[Optional[_TResult], Optional[BaseException]], Any]] = None,
        return_exceptions: Optional[bool] = True,
        callback_filter: Optional[Callable[[_TCallable], bool]] = None,
        threaded: Optional[bool] = True,
        **kwargs: Any,
    ) -> AsyncGenerator[Union[_TResult, BaseException], None]:
        def _done(f: asyncio.Future[_TResult]) -> None:
            if result_callback is not None:
                try:
                    result_callback(f.result(), f.exception())
                except (SystemExit, KeyboardInterrupt):
                    raise
                except BaseException as e:
                    result_callback(None, e)

        awaitables: List[asyncio.Future[_TResult]] = []
        for method in filter(
            lambda x: callback_filter(x) if callback_filter is not None else True,
            set(self),
        ):
            if method is not None:
                if threaded and isinstance(method, HasThreaded) and cast(HasThreaded, method).__threaded__:
                    future = run_coroutine_in_thread(ensure_coroutine(method), *args, **kwargs)
                else:
                    future = create_sub_task(ensure_coroutine(method)(*args, **kwargs))

                awaitables.append(future)

                if result_callback is not None:
                    future.add_done_callback(_done)

        for a in asyncio.as_completed(awaitables):
            try:
                yield await a

            except (SystemExit, KeyboardInterrupt):
                raise
            except BaseException as e:
                if return_exceptions:
                    yield e
                else:
                    raise


class AsyncTaskingEventIterator(AsyncTaskingEventResultIteratorBase[_TCallable, _TResult]):
    def __call__(self, *args: Any, **kwargs: Any) -> AsyncGenerator[Union[_TResult, BaseException], None]:
        return self._notify(*args, **kwargs)


def _get_name_prefix(descriptor: AsyncEventDescriptorBase[Any, Any, Any]) -> str:
    if descriptor._owner is None:
        return type(descriptor).__qualname__

    return f"{descriptor._owner.__qualname__}.{descriptor._owner_name}"


class AsyncTaskingEvent(AsyncTaskingEventResultIteratorBase[_TCallable, _TResult]):
    async def __call__(self, *args: Any, **kwargs: Any) -> List[Union[_TResult, BaseException]]:
        return [a async for a in self._notify(*args, **kwargs)]


class async_tasking_event_iterator(  # noqa: N801
    AsyncEventDescriptorBase[_TCallable, Any, AsyncTaskingEventIterator[_TCallable, Any]]
):
    def __init__(self, _func: _TCallable) -> None:
        super().__init__(
            _func, AsyncTaskingEventIterator[_TCallable, Any], task_name_prefix=lambda: _get_name_prefix(self)
        )


class async_tasking_event(AsyncEventDescriptorBase[_TCallable, Any, AsyncTaskingEvent[_TCallable, Any]]):  # noqa: N801
    def __init__(self, _func: _TCallable) -> None:
        super().__init__(_func, AsyncTaskingEvent[_TCallable, Any], task_name_prefix=lambda: _get_name_prefix(self))


async def check_canceled() -> bool:
    await asyncio.sleep(0)

    return True


def check_canceled_sync() -> bool:
    info = get_current_future_info()
    if info is not None and info.canceled():
        raise asyncio.CancelledError()
    return True


# __threadpool_executor = ThreadPoolExecutor(thread_name_prefix="sub_asyncio")


def run_in_thread(func: Callable[..., _T], /, *args: Any, **kwargs: Any) -> asyncio.Future[_T]:
    global __tread_pool_executor

    loop = asyncio.get_running_loop()

    ctx = contextvars.copy_context()
    func_call = functools.partial(ctx.run, func, *args, **kwargs)

    # return cast(
    #     "asyncio.Future[_T]",
    #     # loop.run_in_executor(__threadpool_executor, cast(Callable[..., _T], func_call)),
    #     loop.run_in_executor(None, cast(Callable[..., _T], func_call)),
    # )

    executor = ThreadPoolExecutor(max_workers=1, thread_name_prefix="sub_asyncio")
    try:
        return cast(
            "asyncio.Future[_T]",
            loop.run_in_executor(executor, cast(Callable[..., _T], func_call)),
        )
    finally:
        executor.shutdown(wait=False)


def run_coroutine_in_thread(
    coro: Callable[..., Coroutine[Any, Any, _T]], *args: Any, **kwargs: Any
) -> asyncio.Future[_T]:
    callback_added_event = threading.Event()
    inner_task: Optional[asyncio.Task[_T]] = None
    canceled = False
    result: Optional[asyncio.Future[_T]] = None

    async def create_inner_task(coro: Callable[..., Coroutine[Any, Any, _T]], *args: Any, **kwargs: Any) -> _T:
        nonlocal inner_task

        ct = asyncio.current_task()

        loop = asyncio.get_event_loop()
        loop.slow_callback_duration = 10

        callback_added_event.wait(600)

        if ct is not None and result is not None:
            _running_tasks[result].children.add(ct)

        inner_task = create_sub_task(coro(*args, **kwargs))

        if canceled:
            inner_task.cancel()

        return await inner_task

    def run(coro: Callable[..., Coroutine[Any, Any, _T]], *args: Any, **kwargs: Any) -> _T:

        old_name = threading.current_thread().name
        threading.current_thread().name = coro.__qualname__
        try:
            return asyncio.run(create_inner_task(coro, *args, **kwargs))
        finally:
            threading.current_thread().name = old_name

        # loop = asyncio.new_event_loop()

        # try:
        #     asyncio.set_event_loop(loop)

        #     t = loop.create_task(create_inner_task(coro, *args, **kwargs), name=coro.__qualname__)

        #     return loop.run_until_complete(t)
        # finally:
        #     try:
        #         running_tasks = asyncio.all_tasks(loop)
        #         if running_tasks:
        #             loop.run_until_complete(asyncio.gather(*running_tasks, return_exceptions=True))

        #         loop.run_until_complete(loop.shutdown_asyncgens())
        #     finally:
        #         asyncio.set_event_loop(None)
        #         loop.close()
        #         threading.current_thread().setName(old_name)

    cti = get_current_future_info()
    result = run_in_thread(run, coro, *args, **kwargs)

    _running_tasks[result] = FutureInfo(result)
    if cti is not None:
        cti.children.add(result)

    def done(task: asyncio.Future[_T]) -> None:
        nonlocal canceled

        canceled = task.cancelled()

        if canceled and inner_task is not None and not inner_task.done():
            inner_task.get_loop().call_soon_threadsafe(inner_task.cancel)

    result.add_done_callback(done)

    callback_added_event.set()

    return result


@contextlib.asynccontextmanager
async def async_lock(lock: threading.RLock) -> AsyncGenerator[None, None]:
    import time

    start_time = time.monotonic()
    locked = lock.acquire(blocking=False)
    while not locked:
        if time.monotonic() - start_time >= 1800:
            raise TimeoutError("Timeout waiting for lock")

        await asyncio.sleep(0.001)

        locked = lock.acquire(blocking=False)
    try:
        yield
    finally:
        if locked:
            lock.release()
    # with lock:
    #     yield


class Event:
    """Thread safe version of an async Event"""

    def __init__(self, value: bool = False) -> None:
        self._waiters: Deque[asyncio.Future[Any]] = deque()
        self._value = value
        self._lock = threading.RLock()

    def __repr__(self) -> str:
        res = super().__repr__()
        extra = "set" if self._value else "unset"
        if self._waiters:
            extra = f"{extra}, waiters:{len(self._waiters)}"
        return f"<{res[1:-1]} [{extra}]>"

    def is_set(self) -> bool:
        # with self._lock:
        return self._value

    def set(self) -> None:
        with self._lock:
            if not self._value:
                self._value = True

                while self._waiters:
                    fut = self._waiters.popleft()

                    if not fut.done():
                        if fut.get_loop() == asyncio.get_running_loop():
                            if not fut.done():
                                fut.set_result(True)
                        else:

                            def set_result(w: asyncio.Future[Any], ev: threading.Event) -> None:
                                try:
                                    if not w.done():
                                        w.set_result(True)
                                finally:
                                    ev.set()

                            if not fut.done():
                                done = threading.Event()

                                fut.get_loop().call_soon_threadsafe(set_result, fut, done)

                                if not done.wait(120):
                                    raise TimeoutError("Callback timeout.")

    def clear(self) -> None:
        with self._lock:
            self._value = False

    async def wait(self, timeout: Optional[float] = None) -> bool:
        with self._lock:
            if self._value:
                return True

            fut = create_sub_future()
            self._waiters.append(fut)

        try:
            await asyncio.wait_for(fut, timeout)
            return True
        except asyncio.TimeoutError:
            return False


class Semaphore:
    """Thread safe version of a Semaphore"""

    def __init__(self, value: int = 1) -> None:
        if value < 0:
            raise ValueError("Semaphore initial value must be >= 0")
        self._value = value
        self._waiters: Deque[asyncio.Future[Any]] = deque()

        self._lock = threading.RLock()

    def __repr__(self) -> str:
        res = super().__repr__()
        extra = "locked" if self.locked() else f"unlocked, value:{self._value}"
        if self._waiters:
            extra = f"{extra}, waiters:{len(self._waiters)}"
        return f"<{res[1:-1]} [{extra}]>"

    async def _wake_up_next(self) -> None:
        async with async_lock(self._lock):
            while self._waiters:
                waiter = self._waiters.popleft()

                if not waiter.done():
                    if waiter.get_loop() == asyncio.get_running_loop():
                        if not waiter.done():
                            waiter.set_result(True)
                    else:
                        if waiter.get_loop().is_running():

                            def set_result(w: asyncio.Future[Any], ev: threading.Event) -> None:
                                try:
                                    if w.get_loop().is_running() and not w.done():
                                        w.set_result(True)
                                finally:
                                    ev.set()

                            if not waiter.done():
                                done = threading.Event()

                                waiter.get_loop().call_soon_threadsafe(set_result, waiter, done)

                                start = time.monotonic()
                                while not done.is_set():

                                    if time.monotonic() - start > 120:
                                        raise TimeoutError("Can't set future result.")

                                    await asyncio.sleep(0.001)

    def locked(self) -> bool:
        with self._lock:
            return self._value == 0

    async def acquire(self, timeout: Optional[float] = None) -> bool:
        while True:
            async with async_lock(self._lock):
                if self._value > 0:
                    break
                fut = create_sub_future()
                self._waiters.append(fut)

            try:
                await asyncio.wait_for(fut, timeout)
            except asyncio.TimeoutError:
                return False
            except (SystemExit, KeyboardInterrupt):
                raise

            except BaseException:
                if not fut.done():
                    fut.cancel()
                if self._value > 0 and not fut.cancelled():
                    await self._wake_up_next()

                raise

        async with async_lock(self._lock):
            self._value -= 1

        return True

    async def release(self) -> None:
        self._value += 1
        await self._wake_up_next()

    async def __aenter__(self) -> None:
        await self.acquire()

    async def __aexit__(
        self, exc_type: Optional[Type[BaseException]], exc_val: Optional[BaseException], exc_tb: Optional[TracebackType]
    ) -> None:
        await self.release()


class BoundedSemaphore(Semaphore):
    """Thread safe version of a BoundedSemaphore"""

    def __init__(self, value: int = 1) -> None:
        self._bound_value = value
        super().__init__(value)

    async def release(self) -> None:
        if self._value >= self._bound_value:
            raise ValueError("BoundedSemaphore released too many times")
        await super().release()


class Lock:
    def __init__(self) -> None:
        self._block = BoundedSemaphore(value=1)

    def __repr__(self) -> str:
        return "<%s _block=%s>" % (self.__class__.__name__, self._block)

    async def acquire(self, timeout: Optional[float] = None) -> bool:
        return await self._block.acquire(timeout)

    async def release(self) -> None:
        await self._block.release()

    @property
    def locked(self) -> bool:
        return self._block.locked()

    async def __aenter__(self) -> None:
        await self.acquire()

    async def __aexit__(
        self, exc_type: Optional[Type[BaseException]], exc_val: Optional[BaseException], exc_tb: Optional[TracebackType]
    ) -> None:
        await self.release()


class OldLock:
    """Threadsafe version of an async Lock."""

    def __init__(self) -> None:
        self._waiters: Optional[Deque[asyncio.Future[Any]]] = None
        self._locked = False
        self._lock = threading.RLock()

    async def __aenter__(self) -> None:
        await self.acquire()

    async def __aexit__(
        self, exc_type: Optional[Type[BaseException]], exc_val: Optional[BaseException], exc_tb: Optional[TracebackType]
    ) -> None:
        await self.release()

    def __repr__(self) -> str:
        res = super().__repr__()
        extra = "locked" if self._locked else "unlocked"
        if self._waiters:
            extra = f"{extra}, waiters:{len(self._waiters)}"
        return f"<{res[1:-1]} [{extra}]>"

    @asynccontextmanager
    async def __inner_lock(self) -> AsyncGenerator[Any, None]:
        b = self._lock.acquire(blocking=False)
        while not b:
            await asyncio.sleep(0)
            b = self._lock.acquire(blocking=False)
        try:
            yield None
        finally:
            self._lock.release()
        # with self._lock:
        #     yield None

    @property
    def locked(self) -> bool:
        return self._locked

    async def acquire(self) -> bool:
        async with self.__inner_lock():
            if not self._locked and (self._waiters is None or all(w.cancelled() for w in self._waiters)):
                self._locked = True
                return True

            if self._waiters is None:
                self._waiters = deque()

            fut = create_sub_future()
            self._waiters.append(fut)

        try:
            try:
                await fut
            finally:
                async with self.__inner_lock():
                    if fut in self._waiters:
                        self._waiters.remove(fut)
                    self._locked = True
        except asyncio.CancelledError:
            await self._wake_up_next()
            raise

        return True

    async def release(self) -> None:
        async with self.__inner_lock():
            if self._waiters is None or len(self._waiters) == 0:
                if self._locked:
                    self._locked = False

        await self._wake_up_next()

    async def _wake_up_next(self) -> None:
        if not self._waiters:
            return

        async with self.__inner_lock():
            try:
                fut = next(iter(self._waiters))
            except StopIteration:
                return
            if fut in self._waiters:
                self._waiters.remove(fut)

        if fut.get_loop() == asyncio.get_running_loop():
            if not fut.done():
                fut.set_result(True)
        else:
            if fut.get_loop().is_running():

                def set_result(w: asyncio.Future[Any], ev: threading.Event) -> None:
                    try:
                        if w.get_loop().is_running() and not w.done():
                            w.set_result(True)
                    finally:
                        ev.set()

                if not fut.done():
                    done = threading.Event()

                    fut.get_loop().call_soon_threadsafe(set_result, fut, done)

                    start = time.monotonic()
                    while not done.is_set():

                        if time.monotonic() - start > 120:
                            raise TimeoutError("Can't set future result.")

                        await asyncio.sleep(0.001)

                    # if not done.wait(120):
                    #     raise TimeoutError("Callback timeout.")


class FutureInfo:
    def __init__(self, future: asyncio.Future[Any]) -> None:
        self.task: weakref.ref[asyncio.Future[Any]] = weakref.ref(future)
        self.children: weakref.WeakSet[asyncio.Future[Any]] = weakref.WeakSet()

        future.add_done_callback(self._done)

    def _done(self, future: asyncio.Future[Any]) -> None:
        if future.cancelled():
            for t in self.children.copy():
                if not t.done() and not t.cancelled() and t.get_loop().is_running():
                    if t.get_loop() == asyncio.get_running_loop():
                        t.cancel()
                    else:
                        t.get_loop().call_soon_threadsafe(t.cancel)

    def canceled(self) -> bool:
        task = self.task()
        if task is not None and task.cancelled():
            return True
        return False


_running_tasks: weakref.WeakKeyDictionary[asyncio.Future[Any], FutureInfo] = weakref.WeakKeyDictionary()


def get_current_future_info() -> Optional[FutureInfo]:
    ct = asyncio.current_task()

    if ct is None:
        return None

    if ct not in _running_tasks:
        _running_tasks[ct] = FutureInfo(ct)

    return _running_tasks[ct]


def create_sub_task(
    coro: Coroutine[Any, Any, _T], *, name: Optional[str] = None, loop: Optional[asyncio.AbstractEventLoop] = None
) -> asyncio.Task[_T]:

    ct = get_current_future_info()

    if loop is not None:
        if loop == asyncio.get_running_loop():
            result = loop.create_task(coro, name=name)
        else:

            async def create_task(
                lo: asyncio.AbstractEventLoop, c: Coroutine[Any, Any, _T], n: Optional[str]
            ) -> asyncio.Task[_T]:
                return create_sub_task(c, name=n, loop=lo)

            return asyncio.run_coroutine_threadsafe(create_task(loop, coro, name), loop=loop).result()
    else:
        result = asyncio.create_task(coro, name=name)

    if ct is not None:
        ct.children.add(result)

    _running_tasks[result] = FutureInfo(result)
    return result


def create_delayed_sub_task(
    coro: Awaitable[_T], *, delay: float, name: Optional[str] = None, loop: Optional[asyncio.AbstractEventLoop] = None
) -> asyncio.Task[Optional[_T]]:
    async def run() -> Optional[_T]:
        try:
            await asyncio.sleep(delay)
            return await coro
        except asyncio.CancelledError:
            return None

    return create_sub_task(run(), name=name, loop=loop)


def create_sub_future(loop: Optional[asyncio.AbstractEventLoop] = None) -> asyncio.Future[Any]:

    ct = get_current_future_info()

    if loop is None:
        loop = asyncio.get_running_loop()

    result = loop.create_future()

    _running_tasks[result] = FutureInfo(result)

    if ct is not None:
        ct.children.add(result)

    return result


class _FutureHolder(Generic[_T]):
    def __init__(self, cfuture: concurrent.futures.Future[_T]):
        self.cfuture = cfuture
        self.afuture = wrap_sub_future(cfuture)


def spawn_coroutine_from_thread(
    func: Callable[..., Coroutine[Any, Any, _T]],
    *args: Any,
    loop: Optional[asyncio.AbstractEventLoop] = None,
    **kwargs: Any,
) -> concurrent.futures.Future[_T]:
    if loop is None:
        loop = asyncio.get_running_loop()

    result = _FutureHolder(asyncio.run_coroutine_threadsafe(func(*args), loop))
    return result.cfuture


def run_coroutine_from_thread(
    func: Callable[..., Coroutine[Any, Any, _T]],
    *args: Any,
    loop: Optional[asyncio.AbstractEventLoop] = None,
    **kwargs: Any,
) -> _T:
    if loop is None:
        loop = asyncio.get_running_loop()

    result = _FutureHolder(asyncio.run_coroutine_threadsafe(func(*args), loop))

    return result.cfuture.result()


def run_coroutine_from_thread_as_future(
    func: Callable[..., Coroutine[Any, Any, _T]],
    *args: Any,
    loop: Optional[asyncio.AbstractEventLoop] = None,
    **kwargs: Any,
) -> asyncio.Future[_T]:
    if loop is None:
        loop = asyncio.get_running_loop()

    return wrap_sub_future(asyncio.run_coroutine_threadsafe(func(*args, **kwargs), loop))


async def run_coroutine_from_thread_async(
    func: Callable[..., Coroutine[Any, Any, _T]],
    *args: Any,
    loop: Optional[asyncio.AbstractEventLoop] = None,
    **kwargs: Any,
) -> _T:
    if loop is None:
        loop = asyncio.get_running_loop()

    return await run_coroutine_from_thread_as_future(func, *args, loop=loop, **kwargs)


def wrap_sub_future(
    future: Union[asyncio.Future[_T], concurrent.futures.Future[_T]],
    *,
    loop: Optional[asyncio.AbstractEventLoop] = None,
) -> asyncio.Future[_T]:
    result = asyncio.wrap_future(future, loop=loop)
    ci = get_current_future_info()
    if ci is not None:
        ci.children.add(result)
    return result
